function out = perf_prop_exam(in)
% PERF_PROP_EXAM  Calculadora rápida para test de performance (propeller)
% Unidades: ft, s, lbf, slug, hp (imperial)

% ---------- Defaults si no se entregan argumentos ----------
if nargin < 1 || isempty(in)
    in = struct('S_ft2',160, ...
                'b_ft',33 + 4/12, ...
                'W0_lbf',1670, ...
                'Wfuel_lbf',177, ...
                'CD0',0.03, ...
                'e',0.7, ...
                'P_SL_hp',110, ...
                'eta_p',0.875, ...
                'TSFC_lb_per_hp_hr',0.40, ...
                'h1_ft',0, ...
                'h2_ft',5000);
end

% ---------- Lectura (con AR opcional) ----------
S   = in.S_ft2;
b   = in.b_ft;
W0  = in.W0_lbf;
Wfuel = in.Wfuel_lbf;
CD0 = in.CD0;
e   = in.e;
if isfield(in,'AR') && ~isempty(in.AR)
    AR = in.AR;
else
    AR = b^2/S;
end
P_SL_hp = in.P_SL_hp;     % potencia motor a SL (shaft)
eta_p   = in.eta_p;       % eficiencia propulsiva
Ctsfc   = in.TSFC_lb_per_hp_hr;  % lb/(hp·h)
h1 = in.h1_ft; h2 = in.h2_ft;

% ---------- Constantes y helpers ----------
hp2ftlbfps = 550;         % 1 hp = 550 ft·lbf/s
nmi_per_hp_hr_per_lbf = 325.6;  % (hp·h)/lbf -> nmi (factor del ejemplo)
rhoSL = isa_rho(0);       % slug/ft^3
rho1  = isa_rho(h1);
rho2  = isa_rho(h2);

k = 1/(pi*AR*e);

% ---------- Óptimos aerodinámicos ----------
CL_star = sqrt(CD0/k);                  % CL @ L/Dmax
LD_max  = 0.5*sqrt(pi*e*AR/CD0);
V_star_SL = sqrt( 2*W0/(rhoSL*S*CL_star) );
V_star_h2 = sqrt( 2*W0/(rho2 *S*CL_star) );

% Condición de mínima potencia requerida (≈ mejor RC)
CL_minP = sqrt(3)*CL_star;
% En esa condición, CD = 4*CD0 (resultado de la polar parásito+inducido)
V_y_SL = sqrt( 2*W0/(rhoSL*S*CL_minP) );
V_y_h2 = sqrt( 2*W0/(rho2 *S*CL_minP) );

% ---------- Potencias y RC ----------
P_av_SL_hp = eta_p * P_SL_hp;
P_av_h2_hp = eta_p * P_SL_hp * (rho2/rhoSL);

P_req_SL_hp = (0.5*rhoSL*V_y_SL^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);
P_req_h2_hp = (0.5*rho2 *V_y_h2^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);

RC_SL_ftps = (P_av_SL_hp - P_req_SL_hp)*hp2ftlbfps / W0;
RC_h2_ftps = (P_av_h2_hp - P_req_h2_hp)*hp2ftlbfps / W0;

% ---------- Tiempo de ascenso (RC lineal) ----------
a = (RC_h2_ftps - RC_SL_ftps) / (h2 - h1);      % [1/s]
dt_sec = (1/a) * log(RC_h2_ftps/RC_SL_ftps);    % s
dt_min = dt_sec/60;

% ---------- Range (Breguet prop) con factor del ejemplo ----------
Wi = W0;
Wf = W0 - 0.95*Wfuel;      % 95% del fuel
lnW = log(Wi/Wf);
LD_at_star = CL_star/(2*CD0);   % L/D en L/Dmax
R_nmi = (eta_p/Ctsfc) * LD_at_star * lnW * nmi_per_hp_hr_per_lbf;

% ---------- Conversión de velocidades y RC ----------
ftps2kt = 0.592484;
V_star_SL_kt = V_star_SL*ftps2kt;  V_star_h2_kt = V_star_h2*ftps2kt;
V_y_SL_kt    = V_y_SL*ftps2kt;     V_y_h2_kt    = V_y_h2*ftps2kt;
RC_SL_fpm    = RC_SL_ftps*60;      RC_h2_fpm    = RC_h2_ftps*60;

% ---------- Salida ----------
out = struct();
out.AR = AR; out.k = k; out.CD0 = CD0; out.e = e;
out.CL_star = CL_star; out.LD_max = LD_max;
out.V_star_SL_ftps = V_star_SL; out.V_star_SL_kt = V_star_SL_kt;
out.V_star_h2_ftps = V_star_h2; out.V_star_h2_kt = V_star_h2_kt;
out.CL_minP = CL_minP;
out.Vy_SL_ftps = V_y_SL; out.Vy_SL_kt = V_y_SL_kt;
out.Vy_h2_ftps = V_y_h2; out.Vy_h2_kt = V_y_h2_kt;
out.RC_SL_ftps = RC_SL_ftps; out.RC_SL_fpm = RC_SL_fpm;
out.RC_h2_ftps = RC_h2_ftps; out.RC_h2_fpm = RC_h2_fpm;
out.climb_time_min = dt_min;
out.Range_nmi = R_nmi;
out.Wi_lbf = Wi; out.Wf_lbf = Wf; out.lnWiWf = lnW;

% ---------- Resumen consola ----------
fprintf('\n=========== RESUMEN (imperial) ===========\n');
fprintf('AR=%.2f, e=%.3f, k=%.5f, CD0=%.4f\n',AR,e,k,CD0);
fprintf('CL* (L/Dmax)=%.4f,  L/Dmax=%.2f\n',CL_star,LD_max);
fprintf('V* (SL)=%.1f ft/s (%.1f kt),  V* (%0.0fft)=%.1f ft/s (%.1f kt)\n', ...
    V_star_SL,V_star_SL_kt,h2,V_star_h2,V_star_h2_kt);
fprintf('Vy (SL)=%.1f ft/s (%.1f kt),  RC(SL)=%.1f ft/s (%.0f fpm)\n', ...
    V_y_SL,V_y_SL_kt,RC_SL_ftps,RC_SL_fpm);
fprintf('Vy (%0.0fft)=%.1f ft/s (%.1f kt),  RC(%0.0fft)=%.1f ft/s (%.0f fpm)\n', ...
    h2,V_y_h2,V_y_h2_kt,h2,RC_h2_ftps,RC_h2_fpm);
fprintf('Climb time %0.0fft:  %.2f min\n',h2-h1,dt_min);
fprintf('Range (95%% fuel, L/Dmax):  %.0f nmi\n',R_nmi);
fprintf('==========================================\n');

end

% ------- ISA simplificada (imperial) para densidad (slug/ft^3)
function rho = isa_rho(h_ft)
T0R = 518.67;                 % Rankine
L   = 0.00356616;             % dT/dh [R/ft] (troposfera)
T   = T0R - L*h_ft;
p0  = 2116.22;                % lbf/ft^2
p   = p0*(T/T0R)^(5.2559);
R   = 1716.59;                % ft·lbf/(slug·R)
rho = p/(R*T);                % slug/ft^3
end
