%% perf_prop_exam_run.m  — Script interactivo (MATLAB Online)  ES/EN
% Performance quick-test for propeller aircraft (Imperial units)
% Prueba rápida de performance para avión con hélice (unidades imperiales)
% Run / Ejecutar:  run('perf_prop_exam_run.m')

clc;

%% ===================== ENUNCIADOS / PROMPTS =====================
% ES: Ingrese los datos del avión. Presione Enter para aceptar el valor por defecto.
% EN: Enter the aircraft inputs. Press Enter to accept the default value.

%% -------- Default example (from professor's handout) / Ejemplo por defecto --------
S_def     = 240;           % ft^2   (wing area / área alar)
b_def     = 37;            % ft     (wingspan / envergadura)
W0_def    = 7800;          % lbf    (gross weight / peso)
Wfuel_def = 1600;          % lbf    (fuel weight / peso combustible)
CD0_def   = 0.03;          % -      (zero-lift drag)
e_def     = 0.82;          % -      (Oswald efficiency)
P_SL_def  = 310;           % hp     (engine power @SL / potencia a nivel del mar)
eta_p_def = 0.84;          % -      (propulsive efficiency / eficiencia propulsiva)
TSFC_def  = 0.45;          % lb/(hp·h)  Consumo específico de freno / Brake Specific Fuel Consumption
h1_def    = 5000;          % ft     (initial altitude / altitud inicial)
h2_def    = 5000;          % ft     (final altitude / altitud final)
                           %  falta AR alargamiento / Aspect Ratio y 
                           %  a (ft/s) velocidad del sonido a 5000 ft /  speed sound
                       

%% -------- Inputs with default (ES/EN prompts) --------
S     = str2double(input(sprintf('S [ft^2] (wing area / área alar)  [default %.3g]: ',S_def),'s'));     if isnan(S),     S=S_def;     end
b     = str2double(input(sprintf('b [ft] (wingspan / envergadura)    [default %.3g]: ',b_def),'s'));    if isnan(b),     b=b_def;     end
W0    = str2double(input(sprintf('W0 [lbf] (gross weight / peso)     [default %.3g]: ',W0_def),'s'));   if isnan(W0),    W0=W0_def;   end
Wfuel = str2double(input(sprintf('W_fuel [lbf] (fuel weight / comb.) [default %.3g]: ',Wfuel_def),'s'));if isnan(Wfuel), Wfuel=Wfuel_def; end
CD0   = str2double(input(sprintf('CD0 [-] (zero-lift drag / parásita)[default %.3g]: ',CD0_def),'s'));  if isnan(CD0),   CD0=CD0_def;   end
e     = str2double(input(sprintf('e [-] (Oswald efficiency)          [default %.3g]: ',e_def),'s'));    if isnan(e),     e=e_def;       end
AR_in = str2double(input(       'AR [-] (aspect ratio / rel. aspecto) (Enter=use b^2/S / usar b^2/S): ','s'));          % may remain NaN
P_SL  = str2double(input(sprintf('P_SL [hp] (engine power @SL / potencia a SL) [default %.3g]: ',P_SL_def),'s')); if isnan(P_SL), P_SL=P_SL_def; end
eta_p = str2double(input(sprintf('eta_p [-] (propulsive eff. / ef. propulsiva) [default %.3g]: ',eta_p_def),'s')); if isnan(eta_p), eta_p=eta_p_def; end
TSFC  = str2double(input(sprintf('TSFC [lb/(hp·h)] (specific fuel consumption) [default %.3g]: ',TSFC_def),'s')); if isnan(TSFC), TSFC=TSFC_def; end
h1    = str2double(input(sprintf('h1 [ft] (initial altitude / altitud inicial) [default %.3g]: ',h1_def),'s')); if isnan(h1), h1=h1_def; end
h2    = str2double(input(sprintf('h2 [ft] (final altitude / altitud final)     [default %.3g]: ',h2_def),'s'));   if isnan(h2), h2=h2_def; end

if isnan(AR_in)
    AR = b^2/S;                   % ES: se calcula con b^2/S  | EN: computed as b^2/S
else
    AR = AR_in;
end

%% ===================== CONSTANTES / CONSTANTS =====================
hp2ftlbfps = 550;                       % 1 hp = 550 ft·lbf/s
nmi_per_hp_hr_per_lbf = 325.6;          % factor usado en el ejemplo para Range (nmi)
% ISA densidad (slug/ft^3) como función anónima (troposfera)
isa_rho = @(h_ft) ( (2116.22 * ( (518.67 - 0.00356616*h_ft)/518.67 )^5.2559) / ...
                    (1716.59 * (518.67 - 0.00356616*h_ft)) );

rhoSL = isa_rho(0);
rho1  = isa_rho(h1);
rho2  = isa_rho(h2);

k = 1/(pi*AR*e);

%% ===================== TEORÍA / THEORY (ES/EN) =====================
% Polar: C_D = C_D0 + k*C_L^2,  k = 1/(pi*AR*e)
% ES: Máxima eficiencia L/D_max ocurre en C_L* = sqrt(C_D0/k).
% EN: Maximum efficiency occurs at C_L* = sqrt(C_D0/k).
%
% ES: Mínima potencia requerida (≈ mejor razón de ascenso para motor casi-constante):
%     C_L|minP = sqrt(3)*C_L*, y en ese punto C_D = 4*C_D0.
% EN: Minimum power required (≈ best rate-of-climb for near-constant power):
%     C_L|minP = sqrt(3)*C_L*, and there C_D = 4*C_D0.

%% ===================== ÓPTIMOS / OPTIMA =====================
CL_star   = sqrt(CD0/k);                       % C_L @ L/D_max
LD_max    = 0.5*sqrt(pi*e*AR/CD0);
V_star_SL = sqrt( 2*W0/(rhoSL*S*CL_star) );
V_star_h2 = sqrt( 2*W0/(rho2 *S*CL_star) );

% Mínima potencia (≈ mejor RC / best ROC)
CL_minP = sqrt(3)*CL_star;
V_y_SL  = sqrt( 2*W0/(rhoSL*S*CL_minP) );
V_y_h2  = sqrt( 2*W0/(rho2 *S*CL_minP) );

%% ===================== POTENCIAS y RC / POWER and ROC =====================
P_av_SL_hp = eta_p * P_SL;                        % available power @SL
P_av_h2_hp = eta_p * P_SL * (rho2/rhoSL);         % available power @h2

% Requerida en minP (C_D = 4*C_D0) / Required in minP
P_req_SL_hp = (0.5*rhoSL*V_y_SL^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);
P_req_h2_hp = (0.5*rho2 *V_y_h2^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);

RC_SL_ftps = (P_av_SL_hp - P_req_SL_hp)*hp2ftlbfps / W0;
RC_h2_ftps = (P_av_h2_hp - P_req_h2_hp)*hp2ftlbfps / W0;

%% ===================== TIEMPO ASCENSO / CLIMB TIME =====================
% Modelo: RC(h) lineal entre h1 y h2  / linear model between h1 and h2
if h2 == h1
    a = 0; dt_min = 0;
else
    a = (RC_h2_ftps - RC_SL_ftps) / (h2 - h1);      % [1/s]
    dt_sec = (1/a) * log(RC_h2_ftps/RC_SL_ftps);    % s
    dt_min = dt_sec/60;
end

%% ===================== RANGE (BREGUET, PROP) =====================
Wi = W0;
Wf = W0 - 0.95*Wfuel;              % ES/EN: 95% del fuel como en el ejemplo
lnW = log(Wi/Wf);
LD_at_star = CL_star/(2*CD0);      % L/D @ L/D_max
R_nmi = (eta_p/TSFC) * LD_at_star * lnW * nmi_per_hp_hr_per_lbf;

%% ===================== CONVERSIONES PRÁCTICAS / PRACTICAL CONVERSIONS =====================
ftps2kt = 0.592484;
V_star_SL_kt = V_star_SL*ftps2kt;  V_star_h2_kt = V_star_h2*ftps2kt;
V_y_SL_kt    = V_y_SL*ftps2kt;     V_y_h2_kt    = V_y_h2*ftps2kt;
RC_SL_fpm    = RC_SL_ftps*60;      RC_h2_fpm    = RC_h2_ftps*60;

%% ===================== REPORTE / REPORT (ES/EN) =====================
fprintf('\n================= ENTRADAS / INPUTS =================\n');
fprintf('S (wing area)        = %8.3f ft^2   -> Área alar de referencia\n', S);
fprintf('b (wingspan)         = %8.3f ft     -> Envergadura\n', b);
fprintf('W0 (gross weight)    = %8.3f lbf    -> Peso de despegue/actual\n', W0);
fprintf('W_fuel (fuel weight) = %8.3f lbf    -> Peso de combustible\n', Wfuel);
fprintf('CD0 (zero-lift drag) = %8.4f  -     -> Arrastre parásito\n', CD0);
fprintf('e (Oswald eff.)      = %8.3f  -     -> Eficiencia de Oswald\n', e);
fprintf('AR (aspect ratio)    = %8.3f  -     -> Relación de aspecto (b^2/S si no se ingresó)\n', AR);
fprintf('P_SL (engine power)  = %8.3f hp     -> Potencia a nivel del mar\n', P_SL);
fprintf('eta_p (prop. eff.)   = %8.3f  -     -> Eficiencia propulsiva\n', eta_p);
fprintf('TSFC                 = %8.3f lb/(hp·h) -> Consumo específico (hélice)\n', TSFC);
fprintf('Altitudes h1→h2      = [%g, %g] ft  -> Tramo para tiempo de ascenso\n', h1, h2);

fprintf('\n================= SALIDAS / OUTPUTS =================\n');
fprintf('k (induced factor)   = %.5f   -> 1/(pi·AR·e): escala el arrastre inducido / scales induced drag\n', k);
fprintf('CL* (at L/D_max)     = %.4f   -> Coef. óptimo de sustentación / optimum lift coeff.\n', CL_star);
fprintf('L/D_max              = %.2f    -> Máxima eficiencia aerodinámica / max aero efficiency\n', LD_max);
fprintf('V* @SL (range speed) = %.1f ft/s  (%.1f kt) -> Velocidad óptima para alcance / best range speed\n', V_star_SL, V_star_SL_kt);
fprintf('V* @%g ft            = %.1f ft/s  (%.1f kt) -> Óptima para alcance en altitud / best range at altitude\n', h2, V_star_h2, V_star_h2_kt);
fprintf('CL|min power (≈Vy)   = %.4f   -> CL para mínima potencia / min power CL (≈best ROC)\n', CL_minP);
fprintf('Vy @SL               = %.1f ft/s  (%.1f kt) -> Velocidad de mejor ROC a SL / best ROC speed at SL\n', V_y_SL, V_y_SL_kt);
fprintf('RC @SL               = %.2f ft/s  (%.0f fpm)-> Mejor razón de ascenso / best rate of climb\n', RC_SL_ftps, RC_SL_fpm);
fprintf('Vy @%g ft            = %.1f ft/s  (%.1f kt) -> Mejor ROC a altitud / best ROC speed at altitude\n', h2, V_y_h2, V_y_h2_kt);
fprintf('RC @%g ft            = %.2f ft/s  (%.0f fpm)-> Rate of climb en altitud / ROC at altitude\n', h2, RC_h2_ftps, RC_h2_fpm);
fprintf('Δt climb h1→h2       = %.2f min      -> Tiempo de ascenso / climb time (linear RC)\n', dt_min);
fprintf('Range (95%% fuel)     = %.0f nmi       -> Alcance Breguet en L/D_max / Breguet range at L/D_max\n', R_nmi);

fprintf('\nNotas / Notes:\n');
fprintf('- En CL|minP se cumple C_D = 4·C_D0. P_avail(h)≈η_p·P_SL·(ρ/ρ_SL) (sin turbo).\n');
fprintf('- Endurance ~ mínima potencia requerida; Range ~ L/D_max.\n');
fprintf('=====================================================\n');
