%% perf_prop_exam_run.m  — Script interactivo (MATLAB Online)  ES/EN
% Performance quick-test for propeller aircraft (Imperial units)
% Prueba rápida de performance para avión con hélice (unidades imperiales)
% Run / Ejecutar:  run('perf_prop_exam_run.m')

clc;

%% ===================== ENUNCIADOS / PROMPTS =====================
% ES: Ingrese los datos del avión. Presione Enter para aceptar el valor por defecto.
% EN: Enter the aircraft inputs. Press Enter to accept the default value.

%% -------- Default example (from professor's handout) / Ejemplo por defecto --------
S_def     = 240;           % ft^2   (wing area / área alar)
b_def     = 37;            % ft     (wingspan / envergadura)
W0_def    = 7800;          % lbf    (gross weight / peso)
Wfuel_def = 1600;          % lbf    (fuel weight / peso combustible)
CD0_def   = 0.03;          % -      (zero-lift drag)
e_def     = 0.82;          % -      (Oswald efficiency)
P_SL_def  = 310;           % hp     (engine power @SL / potencia a nivel del mar)
eta_p_def = 0.84;          % -      (propulsive efficiency / eficiencia propulsiva)
TSFC_def  = 0.45;          % lb/(hp·h)  Consumo específico de freno / Brake Specific Fuel Consumption
h1_def    = 5000;          % ft     (initial altitude / altitud inicial)
h2_def    = 5000;          % ft     (final altitude / altitud final)
                           %  falta AR alargamiento / Aspect Ratio y 
                           %  a (ft/s) velocidad del sonido a 5000 ft /  speed sound
                       

%% -------- Inputs with default (ES/EN prompts) --------
S     = str2double(input(sprintf('S [ft^2] (wing area / área alar)  [default %.3g]: ',S_def),'s'));     if isnan(S),     S=S_def;     end
b     = str2double(input(sprintf('b [ft] (wingspan / envergadura)    [default %.3g]: ',b_def),'s'));    if isnan(b),     b=b_def;     end
W0    = str2double(input(sprintf('W0 [lbf] (gross weight / peso)     [default %.3g]: ',W0_def),'s'));   if isnan(W0),    W0=W0_def;   end
Wfuel = str2double(input(sprintf('W_fuel [lbf] (fuel weight / comb.) [default %.3g]: ',Wfuel_def),'s'));if isnan(Wfuel), Wfuel=Wfuel_def; end
CD0   = str2double(input(sprintf('CD0 [-] (zero-lift drag / parásita)[default %.3g]: ',CD0_def),'s'));  if isnan(CD0),   CD0=CD0_def;   end
e     = str2double(input(sprintf('e [-] (Oswald efficiency)          [default %.3g]: ',e_def),'s'));    if isnan(e),     e=e_def;       end
AR_in = str2double(input(       'AR [-] (aspect ratio / rel. aspecto) (Enter=use b^2/S / usar b^2/S): ','s'));          % may remain NaN
P_SL  = str2double(input(sprintf('P_SL [hp] (engine power @SL / potencia a SL) [default %.3g]: ',P_SL_def),'s')); if isnan(P_SL), P_SL=P_SL_def; end
eta_p = str2double(input(sprintf('eta_p [-] (propulsive eff. / ef. propulsiva) [default %.3g]: ',eta_p_def),'s')); if isnan(eta_p), eta_p=eta_p_def; end
TSFC  = str2double(input(sprintf('TSFC [lb/(hp·h)] (specific fuel consumption) [default %.3g]: ',TSFC_def),'s')); if isnan(TSFC), TSFC=TSFC_def; end
h1    = str2double(input(sprintf('h1 [ft] (initial altitude / altitud inicial) [default %.3g]: ',h1_def),'s')); if isnan(h1), h1=h1_def; end
h2    = str2double(input(sprintf('h2 [ft] (final altitude / altitud final)     [default %.3g]: ',h2_def),'s'));   if isnan(h2), h2=h2_def; end

if isnan(AR_in)
    AR = b^2/S;                   % ES: se calcula con b^2/S  | EN: computed as b^2/S
else
    AR = AR_in;
end

%% ===== ISA variables (editable) justo después de leer h1 y h2 =====
% Troposfera estándar (imperial). Defaults calculados y ofrecidos a edición.
% Unidades: T [R], p [lb/ft^2], rho [slug/ft^3], a [ft/s]

% --- Constantes ISA (imperial) ---
T0_R   = 518.67;        % Rankine (59 F)
p0_psf = 2116.22;       % lb/ft^2 (14.696 psi * 144)
R_air  = 1716.59;       % (ft·lbf)/(slug·R)
gamma  = 1.4;
L_Rft  = 0.00356616;    % lapse rate en R/ft (troposfera)
exp_trop = 5.2559;      % exponente para p ~ (T/T0)^exp

% --- Defaults a partir de h1, h2 (troposfera) ---
T1_def = T0_R - L_Rft*h1;
p1_def = p0_psf*(T1_def/T0_R)^exp_trop;
rho1_def = p1_def/(R_air*T1_def);
a1_def = sqrt(gamma*R_air*T1_def);

T2_def = T0_R - L_Rft*h2;
p2_def = p0_psf*(T2_def/T0_R)^exp_trop;
rho2_def = p2_def/(R_air*T2_def);
a2_def = sqrt(gamma*R_air*T2_def);

% Coeficientes adimensionales por defecto
rhoSL_def = p0_psf/(R_air*T0_R);   % densidad SL teórica (slug/ft^3)
theta1_def = T1_def/T0_R;  delta1_def = p1_def/p0_psf;  sigma1_def = rho1_def/rhoSL_def;
theta2_def = T2_def/T0_R;  delta2_def = p2_def/p0_psf;  sigma2_def = rho2_def/rhoSL_def;

% --- Prompts editables (Enter acepta default) ---
fprintf('\n=== ISA @ h1 = %g ft (editable, Enter = default calculado) ===\n', h1);
T1_in   = str2double(input(sprintf('T1  [R]   (default %.3f): ', T1_def),'s'));   if isnan(T1_in),   T1_in   = T1_def;   end
p1_in   = str2double(input(sprintf('p1  [lb/ft^2] (default %.4f): ', p1_def),'s')); if isnan(p1_in),   p1_in   = p1_def;   end
rho1_in = str2double(input(sprintf('rho1[slug/ft^3] (default %.6f): ', rho1_def),'s')); if isnan(rho1_in), rho1_in = rho1_def; end
a1_in   = str2double(input(sprintf('a1  [ft/s] (default %.2f): ', a1_def),'s'));   if isnan(a1_in),   a1_in   = a1_def;   end
theta1_in = str2double(input(sprintf('theta1 [-]  T/T0 (default %.6f): ', theta1_def),'s')); if isnan(theta1_in), theta1_in = theta1_def; end
delta1_in = str2double(input(sprintf('delta1 [-]  p/p0 (default %.6f): ', delta1_def),'s'));  if isnan(delta1_in), delta1_in = delta1_def; end
sigma1_in = str2double(input(sprintf('sigma1 [-]  rho/rho0 (default %.6f): ', sigma1_def),'s')); if isnan(sigma1_in), sigma1_in = sigma1_def; end

fprintf('\n=== ISA @ h2 = %g ft (editable, Enter = default calculado) ===\n', h2);
T2_in   = str2double(input(sprintf('T2  [R]   (default %.3f): ', T2_def),'s'));   if isnan(T2_in),   T2_in   = T2_def;   end
p2_in   = str2double(input(sprintf('p2  [lb/ft^2] (default %.4f): ', p2_def),'s')); if isnan(p2_in),   p2_in   = p2_def;   end
rho2_in = str2double(input(sprintf('rho2[slug/ft^3] (default %.6f): ', rho2_def),'s')); if isnan(rho2_in), rho2_in = rho2_def; end
a2_in   = str2double(input(sprintf('a2  [ft/s] (default %.2f): ', a2_def),'s'));   if isnan(a2_in),   a2_in   = a2_def;   end
theta2_in = str2double(input(sprintf('theta2 [-]  T/T0 (default %.6f): ', theta2_def),'s')); if isnan(theta2_in), theta2_in = theta2_def; end
delta2_in = str2double(input(sprintf('delta2 [-]  p/p0 (default %.6f): ', delta2_def),'s'));  if isnan(delta2_in), delta2_in = delta2_def; end
sigma2_in = str2double(input(sprintf('sigma2 [-]  rho/rho0 (default %.6f): ', sigma2_def),'s')); if isnan(sigma2_in), sigma2_in = sigma2_def; end

%% ===================== CONSTANTES / CONSTANTS =====================
hp2ftlbfps = 550;                       % 1 hp = 550 ft·lbf/s
nmi_per_hp_hr_per_lbf = 325.6;          % factor usado en el ejemplo para Range (nmi)
% ISA densidad (slug/ft^3) como función anónima (troposfera)
isa_rho = @(h_ft) ( (2116.22 * ( (518.67 - 0.00356616*h_ft)/518.67 )^5.2559) / ...
                    (1716.59 * (518.67 - 0.00356616*h_ft)) );

rhoSL = isa_rho(0);
rho1  = isa_rho(h1);
rho2  = isa_rho(h2);

% ======= Forzar rho1/rho2 a lo que el usuario eligió en los prompts ISA =======
rho1 = rho1_in;
rho2 = rho2_in;

% (Opcional) guardar velocidades del sonido ISA editadas
a1_ftps = a1_in;
a2_ftps = a2_in;

k = 1/(pi*AR*e);

%% ===================== TEORÍA / THEORY (ES/EN) =====================
% Polar: C_D = C_D0 + k*C_L^2,  k = 1/(pi*AR*e)
% ES: Máxima eficiencia L/D_max ocurre en C_L* = sqrt(C_D0/k).
% EN: Maximum efficiency occurs at C_L* = sqrt(C_D0/k).
%
% ES: Mínima potencia requerida (≈ mejor razón de ascenso para motor casi-constante):
%     C_L|minP = sqrt(3)*C_L*, y en ese punto C_D = 4*C_D0.
% EN: Minimum power required (≈ best rate-of-climb for near-constant power):
%     C_L|minP = sqrt(3)*C_L*, and there C_D = 4*C_D0.

%% ===================== ÓPTIMOS / OPTIMA =====================
CL_star   = sqrt(CD0/k);                       % C_L @ L/D_max
LD_max    = 0.5*sqrt(pi*e*AR/CD0);
V_star_SL = sqrt( 2*W0/(rhoSL*S*CL_star) );
V_star_h2 = sqrt( 2*W0/(rho2 *S*CL_star) );

% Mínima potencia (≈ mejor RC / best ROC)
CL_minP = sqrt(3)*CL_star;
V_y_SL  = sqrt( 2*W0/(rhoSL*S*CL_minP) );
V_y_h2  = sqrt( 2*W0/(rho2 *S*CL_minP) );

%% ===================== POTENCIAS y RC / POWER and ROC =====================
P_av_SL_hp = eta_p * P_SL;                        % available power @SL
P_av_h2_hp = eta_p * P_SL * (rho2/rhoSL);         % available power @h2

% Requerida en minP (C_D = 4*C_D0) / Required in minP
P_req_SL_hp = (0.5*rhoSL*V_y_SL^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);
P_req_h2_hp = (0.5*rho2 *V_y_h2^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);

RC_SL_ftps = (P_av_SL_hp - P_req_SL_hp)*hp2ftlbfps / W0;
RC_h2_ftps = (P_av_h2_hp - P_req_h2_hp)*hp2ftlbfps / W0;

%% ===================== TIEMPO ASCENSO / CLIMB TIME =====================
% Modelo: RC(h) lineal entre h1 y h2  / linear model between h1 and h2
if h2 == h1
    a = 0; dt_min = 0;
else
    a = (RC_h2_ftps - RC_SL_ftps) / (h2 - h1);      % [1/s]
    dt_sec = (1/a) * log(RC_h2_ftps/RC_SL_ftps);    % s
    dt_min = dt_sec/60;
end

%% ===================== RANGE (BREGUET, PROP) =====================
Wi = W0;
Wf = W0 - 0.95*Wfuel;              % ES/EN: 95% del fuel como en el ejemplo
lnW = log(Wi/Wf);
LD_at_star = CL_star/(2*CD0);      % L/D @ L/D_max
R_nmi = (eta_p/TSFC) * LD_at_star * lnW * nmi_per_hp_hr_per_lbf;

%% ===================== CONVERSIONES PRÁCTICAS / PRACTICAL CONVERSIONS =====================
ftps2kt = 0.592484;
V_star_SL_kt = V_star_SL*ftps2kt;  V_star_h2_kt = V_star_h2*ftps2kt;
V_y_SL_kt    = V_y_SL*ftps2kt;     V_y_h2_kt    = V_y_h2*ftps2kt;
RC_SL_fpm    = RC_SL_ftps*60;      RC_h2_fpm    = RC_h2_ftps*60;






%% ======= PRE-CÁLCULO DE SALIDAS ADICIONALES PARA TABLA 2 =======
CD_minP = 4*CD0;                            % C_D en mínima potencia
q_SL    = 0.5*rhoSL*V_y_SL^2;               % lbf/ft^2  @SL
D_SL    = q_SL * S * CD_minP;               % lbf       @SL
P_req_SL_hp_exact = (D_SL * V_y_SL)/(hp2ftlbfps*eta_p);  % hp (eje) @SL

q_h2    = 0.5*rho2*V_y_h2^2;                % lbf/ft^2  @h2
D_h2    = q_h2 * S * CD_minP;               % lbf       @h2
P_req_h2_hp_exact = (D_h2 * V_y_h2)/(hp2ftlbfps*eta_p);  % hp (eje) @h2

% Potencia disponible (modelo pistón NA: P ~ rho)
P_av_SL_hp_exact = eta_p * P_SL;                             % hp @SL
P_av_h2_hp_exact = eta_p * P_SL * (rho2/rhoSL);              % hp @h2

% Exceso de potencia y RC (consistentes con tu definición)
P_ex_SL_hp  = P_av_SL_hp_exact  - P_req_SL_hp_exact;         % hp
P_ex_h2_hp  = P_av_h2_hp_exact  - P_req_h2_hp_exact;         % hp
RC_SL_ftps  = (P_ex_SL_hp*hp2ftlbfps)/W0;                    % ft/s
RC_SL_fpm   = RC_SL_ftps*60;                                 % ft/min
RC_h2_ftps  = (P_ex_h2_hp*hp2ftlbfps)/W0;                    % ft/s
RC_h2_fpm   = RC_h2_ftps*60;                                 % ft/min

% Breguet (para mostrar también L/D y ln(Wi/Wf))
LD_at_star_show = CL_star/(2*CD0);
lnW_show        = lnW; % ya calculado arriba
















%% ===================== REPORTE (FORMATO EJEMPLO CLARO) =====================
% Este bloque imprime Inputs/Outputs con el mismo estilo y texto de tu ejemplo.
% Usa las variables ya calculadas en tu script.

% Asegurar variables de tiempo si h1==h2
if ~exist('dt_min','var'); dt_min = 0; end
if ~exist('dt_sec','var'); dt_sec = 0; end

fprintf('\n================= ENTRADAS / INPUTS =================\n');
fprintf('S (wing area)        = %8.3f ft^2   -> Área alar de referencia\n', S);
fprintf('b (wingspan)         = %8.3f ft     -> Envergadura\n', b);
fprintf('W0 (gross weight)    = %8.3f lbf    -> Peso de despegue/actual\n', W0);
fprintf('W_fuel (fuel weight) = %8.3f lbf    -> Peso de combustible\n', Wfuel);
fprintf('CD0 (zero-lift drag) = %8.4f  -     -> Arrastre parásito\n', CD0);
fprintf('e (Oswald eff.)      = %8.3f  -     -> Eficiencia de Oswald\n', e);
fprintf('AR (aspect ratio)    = %8.3f  -     -> Relación de aspecto (b^2/S si no se ingresó)\n', AR);
fprintf('P_SL (engine power)  = %8.3f hp     -> Potencia a nivel del mar\n', P_SL);
fprintf('eta_p (prop. eff.)   = %8.3f  -     -> Eficiencia propulsiva\n', eta_p);
fprintf('TSFC                 = %8.3f lb/(hp·h) -> Consumo específico (hélice)\n', TSFC);
fprintf('Altitudes h1→h2      = [%g, %g] ft  -> Tramo para tiempo de ascenso\n', h1, h2);

% (Opcional) Mostrar ISA usada (si activaste los prompts ISA manuales)
if exist('rho1_in','var') && exist('rho2_in','var')
    fprintf('\n--- ISA usada (opcional) ---\n');
    fprintf('h1=%g ft: rho1=%.6f slug/ft^3, a1=%.1f ft/s, sigma1=%.6f\n', h1, rho1, a1_ftps, sigma1_in);
    fprintf('h2=%g ft: rho2=%.6f slug/ft^3, a2=%.1f ft/s, sigma2=%.6f\n', h2, rho2, a2_ftps, sigma2_in);
end

fprintf('\n================= SALIDAS / OUTPUTS =================\n');

% Aerodinámica base
fprintf('k (induced factor)   = %.5f   -> 1/(pi·AR·e): escala el arrastre inducido / scales induced drag\n', k);
fprintf('CL* (at L/D_max)     = %.4f   -> Coef. óptimo de sustentación / optimum lift coeff.\n', CL_star);
fprintf('L/D_max              = %.2f    -> Máxima eficiencia aerodinámica / max aero efficiency\n', LD_max);

% Velocidad óptima de alcance (usa CL*)
fprintf('V* @SL (range speed) = %.1f ft/s  (%.1f kt) -> Velocidad óptima para alcance / best range speed\n', V_star_SL, V_star_SL_kt);
fprintf('V* @%g ft            = %.1f ft/s  (%.1f kt) -> Óptima para alcance en altitud / best range at altitude\n', h2, V_star_h2, V_star_h2_kt);

% Mínima potencia (≈Vy) y dinámica asociada
fprintf('CL|min power (≈Vy)   = %.4f   -> CL para mínima potencia / min power CL (≈best ROC)\n', CL_minP);

fprintf('Vy @SL               = %.1f ft/s  (%.1f kt) -> Velocidad de mejor ROC a SL / best ROC speed at SL\n', V_y_SL, V_y_SL_kt);
fprintf('q  @SL               = %.4f lbf/ft^2        -> Presión dinámica en Vy (SL)\n', q_SL);
fprintf('D  @SL               = %.2f  lbf            -> Arrastre total (C_D=4·C_D0) en Vy (SL)\n', D_SL);
fprintf('P_req @SL (eje)      = %.2f  hp             -> Potencia requerida aerodinámica/propulsiva en Vy (SL)\n', P_req_SL_hp_exact);
fprintf('P_avail @SL          = %.2f  hp             -> Potencia disponible (η_p·P_SL)\n', P_av_SL_hp_exact);
fprintf('P_exceso @SL         = %.2f  hp             -> P_avail - P_req en SL\n', P_ex_SL_hp);
fprintf('RC @SL               = %.2f ft/s  (%.0f fpm)-> Mejor razón de ascenso / best rate of climb\n', RC_SL_ftps, RC_SL_fpm);

fprintf('Vy @%g ft            = %.1f ft/s  (%.1f kt) -> Mejor ROC a altitud / best ROC speed at altitude\n', h2, V_y_h2, V_y_h2_kt);
fprintf('q  @%g ft            = %.4f lbf/ft^2        -> Presión dinámica en Vy (h2)\n', h2, q_h2);
fprintf('D  @%g ft            = %.2f  lbf            -> Arrastre total (C_D=4·C_D0) en Vy (h2)\n', h2, D_h2);
fprintf('P_req @%g ft (eje)   = %.2f  hp             -> Potencia requerida en Vy (h2)\n', h2, P_req_h2_hp_exact);
fprintf('P_avail @%g ft       = %.2f  hp             -> Potencia disponible (escala con ρ)\n', h2, P_av_h2_hp_exact);
fprintf('P_exceso @%g ft      = %.2f  hp             -> P_avail - P_req en h2\n', h2, P_ex_h2_hp);
fprintf('RC @%g ft            = %.2f ft/s  (%.0f fpm)-> Rate of climb en altitud / ROC at altitude\n', h2, RC_h2_ftps, RC_h2_fpm);

% Tiempo de ascenso formateado
if h2 == h1
    fprintf('Δt climb h1→h2       = %.2f min              -> Tiempo de ascenso / climb time (linear RC)\n', 0);
else
    fprintf('Δt climb h1→h2       = %.2f min              -> Tiempo de ascenso / climb time (linear RC)\n', dt_min);
end

% Breguet: muestra también L/D @star y ln(Wi/Wf)
fprintf('Range (95%% fuel)     = %.0f nmi             -> Alcance Breguet en L/D_max / Breguet range at L/D_max\n', R_nmi);
fprintf('  L/D usado (Breguet) = %.3f  (-)            -> C_L*/(2·C_D0)\n', LD_at_star_show);
fprintf('  ln(Wi/Wf)           = %.4f  (-)            -> ln(%g/%g)\n', lnW_show, Wi, Wf);
