%% perf_prop_exam_run.m  —  Script interactivo (MATLAB Online)
% Unidades: ft, s, lbf, slug, hp (imperial). Sin funciones (no hay "end").
% Corre:  run('perf_prop_exam_run.m')

clc;

%% -------- Defaults (ejemplo del profesor) ----------
S_def     = 160;           % ft^2
b_def     = 33 + 4/12;     % ft
W0_def    = 1670;          % lbf
Wfuel_def = 177;           % lbf
CD0_def   = 0.03;          % -
e_def     = 0.70;          % -
P_SL_def  = 110;           % hp
eta_p_def = 0.875;         % -
TSFC_def  = 0.40;          % lb/(hp·h)
h1_def    = 0;             % ft
h2_def    = 5000;          % ft

%% -------- Prompts con default (Enter = default) ----------
S     = str2double(input(sprintf('Wing area S [ft^2] (default %.3g): ',S_def),'s'));     if isnan(S),     S=S_def;     end
b     = str2double(input(sprintf('Wingspan b [ft] (default %.3g): ',b_def),'s'));        if isnan(b),     b=b_def;     end
W0    = str2double(input(sprintf('Gross weight W0 [lbf] (default %.3g): ',W0_def),'s'));if isnan(W0),    W0=W0_def;   end
Wfuel = str2double(input(sprintf('Fuel weight W_fuel [lbf] (default %.3g): ',Wfuel_def),'s')); if isnan(Wfuel), Wfuel=Wfuel_def; end
CD0   = str2double(input(sprintf('Zero-lift drag CD0 [-] (default %.3g): ',CD0_def),'s'));     if isnan(CD0),   CD0=CD0_def;   end
e     = str2double(input(sprintf('Oswald e [-] (default %.3g): ',e_def),'s'));                 if isnan(e),     e=e_def;       end
AR_in = str2double(input(       'Aspect ratio AR [-] (Enter = usar b^2/S): ','s'));           % puede quedar NaN
P_SL  = str2double(input(sprintf('Engine power SL [hp] (default %.3g): ',P_SL_def),'s'));      if isnan(P_SL),  P_SL=P_SL_def; end
eta_p = str2double(input(sprintf('Propulsive efficiency eta_p [-] (default %.3g): ',eta_p_def),'s')); if isnan(eta_p), eta_p=eta_p_def; end
TSFC  = str2double(input(sprintf('TSFC [lb/(hp·h)] (default %.3g): ',TSFC_def),'s'));          if isnan(TSFC),  TSFC=TSFC_def; end
h1    = str2double(input(sprintf('Initial altitude h1 [ft] (default %.3g): ',h1_def),'s'));   if isnan(h1),    h1=h1_def;     end
h2    = str2double(input(sprintf('Final altitude h2 [ft] (default %.3g): ',h2_def),'s'));     if isnan(h2),    h2=h2_def;     end

if isnan(AR_in)
    AR = b^2/S;
else
    AR = AR_in;
end

%% -------- Constantes y funciones anónimas ----------
hp2ftlbfps = 550;                       % 1 hp = 550 ft·lbf/s
nmi_per_hp_hr_per_lbf = 325.6;          % (hp·h)/lbf -> nmi (factor del ejemplo)
% ISA simple (slug/ft^3) como función anónima (sin "end"):
isa_rho = @(h_ft) ( ...
    (2116.22 * ( (518.67 - 0.00356616*h_ft)/518.67 )^5.2559) / ...
    (1716.59 * (518.67 - 0.00356616*h_ft)) );

rhoSL = isa_rho(0);
rho1  = isa_rho(h1);
rho2  = isa_rho(h2);

k = 1/(pi*AR*e);

%% -------- Óptimos aerodinámicos ----------
CL_star   = sqrt(CD0/k);                       % CL @ L/Dmax
LD_max    = 0.5*sqrt(pi*e*AR/CD0);
V_star_SL = sqrt( 2*W0/(rhoSL*S*CL_star) );
V_star_h2 = sqrt( 2*W0/(rho2 *S*CL_star) );

% Condición de mínima potencia requerida (≈ mejor RC)
CL_minP = sqrt(3)*CL_star;                     % y en ese punto C_D = 4·C_D0
V_y_SL  = sqrt( 2*W0/(rhoSL*S*CL_minP) );
V_y_h2  = sqrt( 2*W0/(rho2 *S*CL_minP) );

%% -------- Potencias y RC ----------
P_av_SL_hp = eta_p * P_SL;
P_av_h2_hp = eta_p * P_SL * (rho2/rhoSL);

P_req_SL_hp = (0.5*rhoSL*V_y_SL^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);
P_req_h2_hp = (0.5*rho2 *V_y_h2^3*S*(4*CD0)) / (hp2ftlbfps*eta_p);

RC_SL_ftps = (P_av_SL_hp - P_req_SL_hp)*hp2ftlbfps / W0;
RC_h2_ftps = (P_av_h2_hp - P_req_h2_hp)*hp2ftlbfps / W0;

%% -------- Tiempo de ascenso (RC lineal) ----------
if h2 == h1
    a = 0; dt_min = 0;
else
    a = (RC_h2_ftps - RC_SL_ftps) / (h2 - h1);      % [1/s]
    dt_sec = (1/a) * log(RC_h2_ftps/RC_SL_ftps);    % s
    dt_min = dt_sec/60;
end

%% -------- Range (Breguet prop) con factor del ejemplo ----------
Wi = W0;
Wf = W0 - 0.95*Wfuel;      % 95% del fuel
lnW = log(Wi/Wf);
LD_at_star = CL_star/(2*CD0);                    % L/D en L/Dmax
R_nmi = (eta_p/TSFC) * LD_at_star * lnW * nmi_per_hp_hr_per_lbf;

%% -------- Conversión práctica ----------
ftps2kt = 0.592484;
V_star_SL_kt = V_star_SL*ftps2kt;  V_star_h2_kt = V_star_h2*ftps2kt;
V_y_SL_kt    = V_y_SL*ftps2kt;     V_y_h2_kt    = V_y_h2*ftps2kt;
RC_SL_fpm    = RC_SL_ftps*60;      RC_h2_fpm    = RC_h2_ftps*60;

%% -------- Reporte (entradas + salidas explicadas) ----------
fprintf('\n================= ENTRADAS (eco) =================\n');
fprintf('Wing area (S)         = %8.3f ft^2     -> Área alar de referencia\n', S);
fprintf('Wingspan  (b)         = %8.3f ft       -> Envergadura\n', b);
fprintf('Weight (W0)           = %8.3f lbf      -> Peso de despegue/actual\n', W0);
fprintf('Fuel weight (W_fuel)  = %8.3f lbf      -> Peso total de combustible\n', Wfuel);
fprintf('CD0 (zero-lift)       = %8.4f  -       -> Arrastre parásito\n', CD0);
fprintf('Oswald efficiency (e) = %8.3f  -       -> Calidad de distribución de sustentación\n', e);
fprintf('Aspect ratio (AR)     = %8.3f  -       -> Relación de aspecto (b^2/S si no se ingresó)\n', AR);
fprintf('Engine power SL       = %8.3f hp       -> Potencia en el eje (nivel del mar)\n', P_SL);
fprintf('Prop. efficiency      = %8.3f  -       -> Eficiencia propulsiva\n', eta_p);
fprintf('TSFC                  = %8.3f lb/(hp·h)-> Consumo específico de potencia\n', TSFC);
fprintf('Altitudes             = [%g, %g] ft -> Tramo de ascenso para Δt\n', h1, h2);

fprintf('\n================= SALIDAS (explicadas) ================\n');
fprintf('Induced factor   (k)  = %.5f   -> 1/(pi·AR·e), escala arrastre inducido\n', k);
fprintf('CL @ L/Dmax    (CL*)  = %.4f   -> Coeficiente de sustentación para máxima eficiencia\n', CL_star);
fprintf('L/D máximo  (L/Dmax)  = %.2f    -> Eficiencia aerodinámica tope\n', LD_max);
fprintf('V @ L/Dmax (SL)       = %.1f ft/s  (%.1f kt) -> Velocidad óptima p/range a SL\n', V_star_SL, V_star_SL_kt);
fprintf('V @ L/Dmax (h2)       = %.1f ft/s  (%.1f kt) -> Velocidad óptima p/range a %g ft\n', V_star_h2, V_star_h2_kt, h2);
fprintf('CL @ Pmin  (≈ Vy)     = %.4f   -> CL para mínima potencia requerida (C_D=4·C_D0)\n', CL_minP);
fprintf('Vy (SL)               = %.1f ft/s  (%.1f kt) -> Velocidad aprox. de mejor rate a SL\n', V_y_SL, V_y_SL_kt);
fprintf('RC (SL)               = %.2f ft/s  (%.0f fpm)-> Mejor razón de ascenso a SL\n', RC_SL_ftps, RC_SL_fpm);
fprintf('Vy (h2=%g ft)         = %.1f ft/s  (%.1f kt) -> Velocidad aprox. de mejor rate a h2\n', h2, V_y_h2, V_y_h2_kt);
fprintf('RC (h2)               = %.2f ft/s  (%.0f fpm)-> Mejor razón de ascenso a h2\n', RC_h2_ftps, RC_h2_fpm);
fprintf('Climb time Δt         = %.2f min          -> Tiempo  h1→h2 (RC lineal)\n', dt_min);
fprintf('Range (95%% fuel)      = %.0f nmi           -> Breguet en L/Dmax con 95%% del fuel\n', R_nmi);
fprintf('\nNotas: en CL_minP se cumple C_D=4·C_D0. P_avail(h)≈η_p·P_SL·(ρ/ρ_SL) sin turbo.\n');
fprintf('       Endurance ~ mínimo P_req; Range ~ L/Dmax. Validar límites de operación.\n');
fprintf('=========================================================\n');
