#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AsuPStore
import SharedMem
import Smash
from collections import defaultdict

class NatPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self,
                 natStatus,
                 dynamicConnectionStatus,
                 dynamicConnectionHwStatus ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.natStatus = natStatus
      self.dynamicConnectionStatus = dynamicConnectionStatus
      self.dynamicConnectionHwStatus = dynamicConnectionHwStatus

   def getDynamicConnections( self ):
      def _createConnection():
         rv = {}
         rv[ 'globalAddress' ] = ''
         rv[ 'target' ] = ''
         rv[ 'fullCone' ] = ''
         rv[ 'addrOnly' ] = ''
         rv[ 'twiceNat' ] = ''
         rv[ 'established' ] = ''
         rv[ 'cmrIntfName' ] = ''
         rv[ 'cmrAclName' ] = ''
         rv[ 'cmrGroup' ] = ''
         return rv

      def _setGlobalAddress( key, globalAddress ):
         dynamicConnection[ key ][ 'globalAddress' ] = globalAddress

      def _setTarget( key, target ):
         dynamicConnection[ key ][ 'target' ] = target

      def _setFullCone( key, fullCone ):
         dynamicConnection[ key ][ 'fullCone' ] = fullCone

      def _setAddrOnly( key, addrOnly ):
         dynamicConnection[ key ][ 'addrOnly' ] = addrOnly

      def _setTwiceNat( key, twiceNat ):
         dynamicConnection[ key ][ 'twiceNat' ] = twiceNat

      def _setEstablished( key, established ):
         dynamicConnection[ key ][ 'established' ] = established

      def _setConnMarkRule( key, connMark ):
         assert connMark in self.natStatus.connMarkRule
         cmr = self.natStatus.connMarkRule[ connMark ]
         cmrKey = cmr.cmrKey
         dynamicConnection[ key ][ 'cmrIntfName' ] = cmrKey.intfName
         dynamicConnection[ key ][ 'cmrAclName' ] = cmrKey.aclName
         try:
            dynamicConnection[ key ][ 'cmrGroup' ] = cmrKey.group
         except AttributeError:
            dynamicConnection[ key ][ 'cmrGroup' ] = cmr.group

      dynamicConnection = defaultdict( _createConnection )

      for k in self.dynamicConnectionStatus.dynamicConnection:
         status = self.dynamicConnectionStatus.dynamicConnection[ k ]
         key = k.natIntf + k.connTuple.stringValue()
         _setGlobalAddress( key, status.globalAddress.stringValue() )
         _setTarget( key, status.target() )
         _setFullCone( key, status.fullCone() )
         _setAddrOnly( key, status.addrOnly() )
         _setTwiceNat( key, status.twiceNat() )
         est = status.established() if hasattr( status, 'established' ) else True
         _setEstablished( key, est )
         _setConnMarkRule( key, status.connMark )

      return dynamicConnection

   def save( self, pStoreIO ):
      pStoreIO.set( 'dynamicConnections', self.getDynamicConnections() )

   def getSupportedKeys( self ):
      return [ 'dynamicConnections' ]

   def getKeys( self ):
      return [ 'dynamicConnections' ]


def Plugin( ctx ):
   featureName = 'Nat'

   if ctx.opcode() == 'GetSupportedKeys':
      natStatus = None
      dynamicConnectionStatus = None
      dynamicConnectionHwStatus = None
   else:

      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()

      natStatus = mg.mount( 'ip/nat/status', 'Ip::Nat::Status', 'r' )

      # We are using "shadow" instead of "reader" to avoid mismatch with NatCli.py
      smashEm = SharedMem.entityManager( sysdbEm=entityManager )
      mountInfo = Smash.mountInfo( 'shadow' )

      dynamicConnectionStatus = \
            smashEm.doMount( "ip/nat/status/dynamicConnection",
               "Ip::Nat::DynamicConnectionStatus", mountInfo )

      dynamicConnectionHwStatus = \
            smashEm.doMount( "ip/nat/hwStatus/dynamicConnection",
               "Ip::Nat::DynamicConnectionHwStatus", mountInfo )


      mg.close( blocking=True )

   ctx.registerAsuPStoreEventHandler(
         featureName,
         NatPStoreEventHandler( natStatus,
                                dynamicConnectionStatus,
                                dynamicConnectionHwStatus ) )
