# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# N.B. If you edit this file you must regenerate and replace
# the associated AsuPatch patch RPM. For more information see
# /src/AsuPatch/packages/README.

# This plugin will be shipped as AsuPatch that gets installed
# from EOS version 4.24.1 onwards.
# - The same source should work for both python2 and python3
# - There are various changes that has been done to AfiSafi state
#   across releases. This plugin should work across all those
#   changes and hence it supports legacy types and mount paths.
from __future__ import absolute_import, division, print_function
import os
import Tracing
import AsuPStore
import SharedMem
import Smash
import Tac
import Cell

__defaultTraceHandle__ = Tracing.Handle( 'ArBgpAsuPStore' )
t0 = Tracing.trace0
t1 = Tracing.trace1

pStoreItem = {}

class ArBgpPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self, afiSafiStateTableDir, em, peerStatusInSmash=False ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.afiSafiStateTableDir = afiSafiStateTableDir
      self.em = em
      self.peerStatusInSmash = peerStatusInSmash
      self.funcs_ = {
         'bgpPeerAfiSafiStateTableDir' : self.storeBgpPeerAfiSafiStateTable
      }
      self.keys = [ 'bgpPeerAfiSafiStateTableDir' ]
      self.setAfiSafiIndexEnumType()
      self.setIsLegacy()

   def setAfiSafiIndexEnumType( self ):
      # The AfiSafiIndexEnum type has moved across releases. Use the one that works
      try:
         if os.environ.get( 'ARBGP_ASU_PSTORE_TEST_LEGACY_AFISAFI_TYPE' ):
            t0( 'TestEnv: Raise TypeError to test legacy ',
                'Smash::Routing::AfiSafiIndexEnum' )
            raise TypeError
         self.afiSafiIndexEnumType = Tac.Type( 'Routing::Bgp::AfiSafiIndexEnum' )
      except TypeError:
         self.afiSafiIndexEnumType = Tac.Type( 'Smash::Routing::AfiSafiIndexEnum' )

   def setIsLegacy( self ):
      if not self.afiSafiStateTableDir:
         return
      entities = self.afiSafiStateTableDir.values()
      if entities:
         # If dir has entities, check the type of first entity.
         self.isLegacy = ( entities[ 0 ].tacType.fullTypeName !=
                           'Routing::Bgp::BgpPeerAfiSafiStateTable' )
         t0( 'Legacy environment is', self.isLegacy )
      else:
         # If dir is empty, we have nothing to save and does not matter.
         self.isLegacy = None

   def getSmashPeers( self, vrf ):
      smashEm = SharedMem.entityManager( sysdbEm=self.em )
      smi = Smash.mountInfo( 'reader' )
      if os.environ.get( 'ARBGP_ASU_PSTORE_TEST_LEGACY_SMASH_PATH' ):
         t0( 'TestEnv: Test legacy smash type' )
         smashType = 'Smash::Routing::BgpPeerInfoStatusLegacy'
      else:
         smashType = 'Smash::Routing::BgpPeerInfoStatus'
      smashVrf = smashEm.doMount(
         'routing/bgp/bgpPeerInfoStatus/%s' % vrf.name,
         smashType, smi )
      ret = smashVrf.bgpPeerStatusEntry.items()
      t0( 'Get peer info from smash', vrf.name )
      smashEm.doUnmount( 'routing/bgp/bgpPeerInfoStatus/%s' % vrf.name )
      return ret

   def getPeers( self, vrf ):
      if self.isLegacy:
         if self.peerStatusInSmash:
            return self.getSmashPeers( vrf )
         else:
            return vrf.bgpPeerInfoStatusEntry.items()
      else:
         return vrf.bgpPeerAfiSafiState.items()

   def getIntfId( self, peer ):
      return peer.intfId

   def getState( self, peer ):
      return peer.bgpState if self.isLegacy else peer.state

   def getAfiSafiState( self, peer ):
      return peer.bgpAfiSafiState.values() if self.isLegacy\
             else peer.afiSafiState.values()

   def storeBgpPeerAfiSafiStateTable( self ):
      t0( 'Store afiSafiStateTableDir' )
      numPeersStored = 0
      pyCol = {}
      hasEnumName = 'enumName' in dir( Tac )
      for vrfName, vrf in self.afiSafiStateTableDir.items():
         vrfInfo = pyCol.setdefault( vrfName, {} )
         for peerIp, peer in self.getPeers( vrf ):
            peerInfo = vrfInfo.setdefault( peerIp.stringValue, {} )
            peerInfo[ 'intfId' ] = self.getIntfId( peer )
            peerInfo[ 'state' ] = self.getState( peer )
            peerInfo[ 'afiSafiState' ] = {}
            for i, v in enumerate( self.getAfiSafiState( peer ) ):
               # The enumName is not available in all previous versions
               if hasEnumName:
                  afiSafiName = Tac.enumName( self.afiSafiIndexEnumType, i )
               else:
                  afiSafiName = self.afiSafiIndexEnumType.attributes[ i ]
               peerInfo[ 'afiSafiState' ][ afiSafiName ] = v
            numPeersStored = numPeersStored + 1
            t1( 'ASU PStore store' +
                ': VRF ' + vrfName +
                ', peer ' + peerIp.stringValue +
                ', intfId ' + peerInfo[ 'intfId' ] +
                ', state ' + peerInfo[ 'state' ] +
                ', afiSafiState ' + str( peerInfo[ 'afiSafiState' ] ) )
      t0( 'ASU PStore store: number of peers stored:', numPeersStored )
      return pyCol

   def save( self, pStoreIO ):
      global pStoreItem
      for k in self.keys:
         pStoreIO.set( k, pStoreItem[ k ] )
      t0( 'Reset pre-save state after save' )
      pStoreItem = {}

   def preSave( self ):
      t0( 'pre-save afiSafiStateTableDir' )
      for k in self.keys:
         pStoreItem[ k ] = self.funcs_[ k ]()

   def getSupportedKeys( self ):
      return self.keys

   def getKeys( self ):
      return self.keys

   def hitlessReloadSupported( self ):
      # The "save" method is called only after shutdown stages are
      # complete in ASU shutdown. The ArBgp agent registers for
      # "NetworkAgentShutdown" stage and puts all the peers in "Idle"
      # state when it handles this stage. Now the plugin "save" method
      # saves the "Idle" state, which is not going to be useful for
      # restore path.
      # As a workaround, pre-save the state when the plugin is invoked
      # with opcode "GetKeys". The "GetKeys" is called before the
      # "NetworkAgentShutdown" stage starts and hence we save the
      # pre-NetworkAgentShutdown state.
      self.preSave()
      return ( None, None )

def Plugin( ctx ):
   featureName = 'ArBgp'

   t0( 'Opcode is', ctx.opcode() )
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
         featureName,
         ArBgpPStoreEventHandler( None, None ) )
      return

   em = ctx.entityManager()
   # This patch applies from EOS version starting from 4.24.1
   # The mount path and type that stores the peer afiSafiState
   # has changed. So we try mounting different paths.
   mg = em.mountGroup()
   exportDir = mg.mount( 'routing/bgp/export', 'Tac::Dir', 'r' )
   mg.close( blocking=True )
   peerStatusInSmash = False
   if 'vrfBgpPeerAfiSafiStateTable' in exportDir.entryState:
      # The 'routing/bgp/export/vrfBgpPeerAfiSafiStateTable' is the latest path
      # its table type is 'Routing::Bgp::BgpPeerAfiSafiStateTable'
      # cl/27040842 did this change from cell path of vrfBgpPeerInfoStatusEntryTable
      mg2 = em.mountGroup()
      afiSafiStateTableDir = mg2.mount(
         'routing/bgp/export/vrfBgpPeerAfiSafiStateTable',
         'Tac::Dir', 'ri' )
      mg2.close( blocking=True )
      t0( 'Mounted routing/bgp/export/vrfBgpPeerAfiSafiStateTable' )
   elif 'vrfBgpPeerInfoStatusEntryTable' in exportDir.entryState:
      # The 'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' is the
      # next path its table type is 'Routing::Bgp::BgpPeerInfoStatusEntryTable'.
      # The cl/25996352 did this change from smash path. This
      # was later moved to cell path (handled in else block).
      mg2 = em.mountGroup()
      afiSafiStateTableDir = mg2.mount(
         'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable',
         'Tac::Dir', 'ri' )
      mg2.close( blocking=True )
      t0( 'Mounted legacy routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' )

      if os.environ.get( 'ARBGP_ASU_PSTORE_TEST_LEGACY_NON_CELL_EXPORT_PATH' ):
         t0( 'TestEnv: test legacy non cell export path ',
             'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' )
      else:
         BgpPeerInfoStatusEntry = Tac.Type( 'Routing::Bgp::BgpPeerInfoStatusEntry' )
         if 'bgpAfiSafiState' not in BgpPeerInfoStatusEntry.attributes:
            # This oldest path where this state is available is
            # in smash path 'routing/bgp/bgpPeerInfoStatus'. The
            # cl/14228794 introdued bgpAfiSafiState in this path.
            # We will mount per-Vrf path from smash in the plugin
            # code for all the VRFs in vrfBgpPeerInfoStatusEntryTable.
            t0( 'The peer status is in legacy Smash table '
                'routing/bgp/bgpPeerInfoStatus/<vrf>' )
            peerStatusInSmash = True
   else:
      # The cl/26523265 did cellification of vrfBgpPeerInfoStatusEntryTable. So, the
      # vrfBgpPeerInfoStatusEntryTable is either found in exportDir or in the below
      # cell path.
      mg2 = em.mountGroup()
      afiSafiStateTableDir = mg2.mount(
         Cell.path( 'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' ),
         'Tac::Dir', 'ri' )
      mg2.close( blocking=True )
      t0( 'Mounted legacy cell:routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' )

   ctx.registerAsuPStoreEventHandler(
      featureName, ArBgpPStoreEventHandler( afiSafiStateTableDir,
                                            em, peerStatusInSmash ) )
