#!/bin/bash

#
# Script to be called before logrotate starts processing files under
# /var/log/agents and /var/core. 
#

agent_logs_dir="/var/log/agents"
core_files_dir="/var/core"

# When we have bunch of empty log files for an agent(s) (repeated agent crash due to, 
# say, lack of space), logrotate can spend lot of time processing these empty files.
# In such scenario, we can delete all empty logs of an agent except the 
# one that is opened by agent. This cannot be done with a section in logrotate.conf 
delete_empty_agent_logs()
{
    # for each agent (which has empty logs)
    for f in `find $agent_logs_dir -size 0 -regex "$agent_logs_dir/[A-Z].*\-[0-9]+$" | rev | cut -d '-' -f2- | rev | sort | uniq`
    do
       an=$(basename $f)
       # delete all empty files but log of latest file (maybe opened by agent)
       fcount=`find $agent_logs_dir -size 0 -regex "$agent_logs_dir/${an}\-[0-9]+$" | wc -l`
       if [ $fcount -gt 1 ]
       then
           # Keep most recent file
           find $agent_logs_dir -size 0 -regex "$agent_logs_dir/${an}\-[0-9]+.*" |\
                         xargs ls -1t | tail -n +2 | xargs rm
       fi
    done
}

delete_empty_cores()
{
    # Delete empty core files that are older than a min
    find $core_files_dir -size 0 -mmin +1 -exec rm {} \;
}

# When logrotate fails to rotate due to lack of space, it leaves empty or partially
# copied logs around. When logrotate starts next time, even though space is available
# it fails to rotate as it sees a file with rotated (copy) name already exists.
delete_rotate_failed_logs()
{
    find $agent_logs_dir -regex "$agent_logs_dir/[A-Z].*\-[0-9]+\.[0-9]+$" -exec rm {} \;
}

# An EOS agent may have restarted several times. We only want to keep
# agent logs of the first instance and the last few instances
cleanup_agent_logs()
{
    if [ -d "$agent_logs_dir" ]; then
       /usr/bin/arista-python /usr/bin/archivecleanup.py --src "$agent_logs_dir"
    fi
}

delete_rotate_failed_logs
delete_empty_agent_logs
delete_empty_cores
cleanup_agent_logs

exit 0
