#! /bin/bash

. /etc/rc.d/init.d/functions

### BEGIN INIT INFO
# Required-Stop: Post
### END INIT INFO

# This script waits for EOS to initialize to a reasonable point (right now, 
# Aaa and Ssh warmup so that EOS users can login through console and SSH to 
# get a CLI session). User can press ESC to skip this phase and login as root 
# through the console if root login is enabled.
#
# This script should be run at the last stage of initialization so that
# it does not block any other boot scripts. Currently it is set to S94,
# just before EosStage3 so the version information is printed last.
lockfile=/var/lock/subsys/Aaa
eosmsg="Completing EOS initialization"
eosinitlog=/var/log/EosInit
datefmt='+%Y-%m-%d %T'

INTERVAL=0.5

# BUG997778: Disable AAA warmup for SWAG workers.
python -c 'import Swag;import sys; sys.exit(0 if Swag.role() == "worker" else 1)'
if [ $? -eq 0 ]; then
    AAA_WARMUP_CMD="true"
fi

# The env variable can be used to supersede the wfw cmd, even on SWAG worker devices.
if [ -z "$AAA_WARMUP_CMD" ]; then
    AAA_WARMUP_CMD="wfw -v -t 3600 Aaa Ssh ConfigAgent"
fi

tty_path() {
    # find a tty associated with fd 0, 1 or 2.
    for i in 0 1 2
    do
        if [ -t $i ]; then
            echo "/proc/self/fd/$i"
            break
        fi
    done
}

start() {
    touch $lockfile
    echo "$(date "$datefmt") Aaa.sh: start" >> $eosinitlog

    grep 'card\|manufacturing' /etc/celltype >> /dev/null 2>&1

    if [ $? -eq 0 ]; then 
        # We're on a card ( ie not a supervisor or generic pc ), 
        # so Aaa doesn't run and we don't rely on the admin 
        # account being created.
        # In the case of manufacturing, we aren't running EOS.
        echo -n "$eosmsg: Card or manufacturing cell detected. Skipping. "
        success
        echo
        exit 0
    fi

    # set password/sshkey for root. Although Aaa agent will do that later on,
    # we do it here, so that in case Aaa fails to come up for some reason,
    # we still have root account set appropriately.
    /usr/bin/AaaSetRootPassword

    ulimit -c 0

    # Run wfw in background
    # wfw has a default timeout of 10 minutes - just pick a large value.
    # On standby supervisor, Aaa might be warm before ConfigAgent.
    echo "$(date "$datefmt") Aaa.sh: AAA_WARMUP_CMD='${AAA_WARMUP_CMD}'" >> $eosinitlog
    ( $AAA_WARMUP_CMD ) >> $eosinitlog 2>&1 &
    PID=$!
    SYSDB_PID=$(pidof Sysdb)
    attempts=1
    echo "$(date "$datefmt") Aaa.sh: warmup command started" >> $eosinitlog

    # read from TTY associated with stdin/stdout/stderr
    # in default systemd we have this:
    #
    # lr-x------ 1 root root 64 Oct 15 14:12 0 -> /dev/null
    # l-wx------ 1 root root 64 Oct 15 14:12 1 -> pipe:[10872]
    # l-wx------ 1 root root 64 Oct 15 14:12 2 -> /dev/console
    #
    # Somehow setting StandardInput to tty messes up output on console,
    # so we just directly read from TTY.
    TTY="$(tty_path)"
    if [ "$TTY" != "" ]; then
        exec 99<$TTY
        # Display the message here after spawning wfw so Esc can work
        echo -n $"$eosmsg (press ESC to skip): "
    else
        echo -n $"$eosmsg: "
    fi

    cancelled=0

    while [ 1 ]; do
        if [ "$TTY" != "" ]; then
            read -s -r -d "" -N 1 -t $INTERVAL -u 99 CHAR || true
        else
            sleep $INTERVAL
        fi
        if [ "$CHAR" = $'\x1B' ]; then
            cancelled=1
            echo "$(date "$datefmt") Aaa.sh: <ESC>" >> $eosinitlog
            kill -9 $PID >> $eosinitlog 2>&1
            # fall through to wait for process to exit
        fi

        STATE="`ps -p $PID -o state=`"
        if [ "$STATE" = ""  ]; then
            # has exited
            echo "$(date "$datefmt") Aaa.sh: warmup process has exited" >> $eosinitlog
            wait $PID >> $eosinitlog 2>&1
            WFWRET=$?
            echo "$(date "$datefmt") Aaa.sh: wait complete" >> $eosinitlog
            if [ $WFWRET -eq -1 ]; then
                # BUG408306: sometimes wait returns -1 even though agent is warm.
                # Not sure why, but deal with it here.
                grep 'Agents are warm' $eosinitlog >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    WFWRET=0
                fi
            fi

            if [ $WFWRET -eq 0 ]; then
                echo "$(date "$datefmt") Aaa.sh: success" >> $eosinitlog
                success
                echo
                exit 0
            else
                echo "$(date "$datefmt") Aaa.sh: failure:" >> $eosinitlog
                echo "wfw failed with exit code $WFWRET" >> $eosinitlog
                # Sysdb could have been restarted by ElectionMgr in parallel.
                # The new Sysdb may not have started yet, but wfw will wait.
                NEW_SYSDB_PID=$(pidof Sysdb)
                if [[ "$SYSDB_PID" != "$NEW_SYSDB_PID"
                      && $attempts -lt 2
                      && $cancelled -eq 0 ]]; then
                    echo "Sysdb's pid changed from $SYSDB_PID to $NEW_SYSDB_PID" >> \
                           $eosinitlog
                    ( $AAA_WARMUP_CMD ) >> $eosinitlog 2>&1 &
                    PID=$!
                    echo "retrying wfw, PID =" $PID >> $eosinitlog
                    attempts=$(expr $attempts + 1)
                else
                    failure
                    echo "
EOS will continue to boot without waiting for full initialization.
You may not be able to login using normal accounts, but you may be
able to login as root."
                    echo
                    exit 1
                fi
            fi
        fi
    done
}

stop() {
    rm -f $lockfile
    echo -n "Flushing AAA accounting queue: "
    /usr/bin/AaaFlushAccounting
    # EOS18: If we are rebooting or shutting down,
    # shutdown sshd at this point, so that any clients
    # connected to this gracefully close the connection.
    reboot=`systemctl list-units --type=target | grep reboot`
    shutdown=`systemctl list-units --type=target | grep shutdown`
    halt=`systemctl list-units --type=target | grep halt`
    poweroff=`systemctl list-units --type=target | grep poweroff`
    if [ "$reboot" != "" -o "$halt" != "" -o "$poweroff" != "" -o \
    "$shutdown" != "" ];
    then
       logger "Shutting down sshd"
       killall sshd 2> /dev/null || true
       logger "Shutting down nginx"
       service nginx stop_all >& /dev/null || true
       logger "Shutting down OpenConfig"
       killall OpenConfig 2> /dev/null || true
    fi

    if [ $? -eq 0 ]; then
        success
        echo
        exit 0
    else
        failure
        echo
        exit 1
    fi
}

case "$1" in
start)
    start
    ;;
stop)
    stop
    ;;
*)
    echo $"Usage: $0 {start}"
    exit 1
esac

