#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
   Print SystemMemoryPlusLoadReport for the memory usage of the system.

   Includes:
   --------
   * Overall System Memory usage information tracked by Linux and reported
     in /proc/meminfo
   * Display the size of the crash kernel memory
   * Display the number of OOM Killed Processes on this system since boot
   * Top twenty memory using Linux Kernel slabs so we can track slab usage
     tracked by Linux and reported in /proc/slabinfo
   * Print slabinfo for slabs that are using 0.1% of memory to limit the
     number of slab info entries we print.
   * System Load Averqge Statistics
   * Pressure Stall Information (PSI)
   * Display the transparent huge page statistics as well as related
     memory compaction and khugepage statistics

   Author: Edward Chron <echron@arista.com>

   Date: 2023/07/03
"""

import argparse
from collections import deque
from dataclasses import dataclass
from datetime import datetime
import json
import operator
import os
import subprocess
from subprocess import Popen
import sys
import time
from typing import Type
import CliPrint
from CliPrint import Printer
from CliPrint import FloatAttrInfo
from CliPrint import IntAttrInfo
from CliPrint import StringAttrInfo

PROC_MEMINFO_PATH = '/proc/meminfo'
LOCAL_OOM_CTR_PATH = "/proc/oom_kills"
GENERIC_OOM_CTR_PATH = "/proc/vmstat"
GENERIC_THP_STATS_PATH = "/proc/vmstat"
KERNEL_PROC_IOMEM_PATH = "/proc/iomem"
CRASH_KERNEL_MEMORY = " : Crash kernel"
PROC_LOADAVG_PATH = "/proc/loadavg"
PROC_PSI_PATH = "/proc/pressure"
PROC_PSI_CPU_PATH = "/proc/pressure/cpu"
PROC_PSI_IOS_PATH = "/proc/pressure/io"
PROC_PSI_IRQ_PATH = "/proc/pressure/irq"
PROC_PSI_MEM_PATH = "/proc/pressure/memory"
PROC_SYS_UPTIME = "/proc/uptime"

OUTPUT_FORMAT = CliPrint.JSON
PRINTER = Printer( OUTPUT_FORMAT )
PRETTY_PRINT = False
jsonData = {}

@dataclass
class MemSwapValues:
   """.
   Initialize the Memory and Swap size our system to zero
   Using a dataclass to avoid pylint grousing about using global variables
   """
   totalStorage = 0
   totalMemory = 0
   totalSwap = 0

# This gives access to global values for storage, memory, swap totals
memSwapSize = MemSwapValues()

class PrintAll:
   """
   PrintAll class set to globally accessible variable that indicates
   whether we need to print all values, even if they would otherwise
   be excluded from being printed (for example they have a value of 0).

   The default value is to print all variables so the class is set
   _enabled as True.
   Setting the value of the class to False means that the class
   specifies that prtAll is set to False, so printing all variables
   is False unless the variable(s) to be printed have at least one
   non-zero value is to be printed. This option can be used to reduce
   the output for lines that have a zero setting or usage.

   The value can be changed anywhere but it is typically set in the
   main routine of the main program to define the setting based on a
   an optional command line argument that user my specify.

   An optional command line argument is used to enable printing all
   variables. The argument specifies whether we want to print all
   entries or just the entries that have greater than zero values in
   them.
   """
   _enabled = True

   def __new__( cls ):
      """ creates singleton object if no object exists else it
          returns the previous singleton object """
      if not hasattr( cls, 'instance' ):
         cls.instance = super( PrintAll, cls ).__new__( cls )
      return cls.instance

   def prtAllEnabled( self ) -> None:
      """ Enable print all, we want to print all variables """
      self._enabled = True

   def prtAllDisabled( self ) -> None:
      """ Disable print all, we do not want to print all variables """
      self._enabled = False

   def prtAll( self ) -> bool:
      """ Return print all status, Enabled (True) or Disabled (False) """
      return self._enabled

def memUnits( memValue: int, inUnits: str ) -> tuple[ float, str ]:
   """
   Convert value in inital units to a float with corresponding
   units to make value scaled for print.

   inUnits are either KiB = 1024 or B for bytes.
   """
   byteCt: int = memValue
   units: str = "B"

   if inUnits == "KiB":
      byteCt = memValue * 1024
   fvalue = float( byteCt )

   if byteCt >= int( 1125899906842624 ):
      fvalue = fvalue / float( 1125899906842624 )
      units = "PiB"
   elif byteCt >= int( 1099511627776 ):
      fvalue = fvalue / float( 1099511627776 )
      units = "TiB"
   elif byteCt >= int( 1073741824 ):
      fvalue = fvalue / float( 1073741824 )
      units = "GiB"
   elif byteCt >= int( 1048576 ):
      fvalue = fvalue / float( 1048576 )
      units = "MiB"
   elif byteCt >= int( 1024 ):
      fvalue = fvalue / float( 1024 )
      units = "KiB"
   return fvalue, units

def unitValue( units: str ) -> int:
   """
   Return the unit size in KiB
   """
   kibs = 0
   if units == 'KiB':
      kibs = 1
   elif units == 'MiB':
      kibs = 1024
   elif units == 'GiB':
      kibs = 1048576
   elif units == 'TiB':
      kibs = 1073741824
   elif units == 'PiB':
      kibs = 1125899906842624
   return kibs

def pctPrt( value: float, units: str ) -> str:
   """
   Given the value and units of memory return the percentage of
   memory and swap that represents.
   Total storage size available from global totalStorage.
   """
   unitsize = unitValue( units )
   sizebytes = float( value * unitsize )
   pct = float( sizebytes / float( memSwapSize. totalStorage ) ) * 100
   pctstr = f"({pct:9.2f}%)"
   return pctstr

def dictPrt( minfoDict: dict[ str, str ], keys: list[ str ] ) -> None:
   """
   Print list of specified keys, values are all in KiB
   """
   prt = PrintAll()
   if len( keys ) == 3:
      val0, uni0 = int( minfoDict[ keys[ 0 ] ] ), "KiB"
      val1, uni1 = int( minfoDict[ keys[ 1 ] ] ), "KiB"
      val2, uni2 = int( minfoDict[ keys[ 2 ] ] ), "KiB"
      if val0 == 0 and val1 == 0 and val2 == 0 and not prt.prtAll():
         return
      print( f"{keys[0]:17}: {val0:14} {uni0:3} {pctPrt(val0, uni0)}  "
             f"{keys[1]:17}: {val1:14} {uni1:3} {pctPrt(val1, uni1)}  "
             f"{keys[2]:17}: {val2:14} {uni2:3} {pctPrt(val2, uni2)}" )
   elif len( keys ) == 2:
      val0, uni0 = int( minfoDict[ keys[ 0 ] ] ), "KiB"
      val1, uni1 = int( minfoDict[ keys[ 1 ] ] ), "KiB"
      if val0 == 0 and val1 == 0 and not prt.prtAll():
         return
      print( f"{keys[0]:17}: {val0:14} {uni0:3} {pctPrt(val0, uni0)}  "
             f"{keys[1]:17}: {val1:14} {uni1:3} {pctPrt(val1, uni1)}" )
   elif len( keys ) == 1:
      val0, uni0 = int( minfoDict[ keys[ 0 ] ] ), "KiB"
      if val0 == 0 and not prt.prtAll():
         return
      print( f"{keys[0]:17}: {val0:14} {uni0:3} {pctPrt(val0, uni0)}" )

def memInfoPrint( memInfoDict: dict[ str, str ] ) -> None:
   """
   Print the /proc/meminfo values for each entry
   """
   totStg, totStgUnits = memSwapSize.totalStorage, "KiB"
   titleStr = \
    f"Memory Usage Statistics: Total Storage (memory + swap): {totStg} {totStgUnits}"
   if PRETTY_PRINT:
      print( titleStr )
      print( "---------------------------------------------------"
             "----------------------" )
      dictPrt( memInfoDict, [ "MemTotal", "MemFree", "MemAvailable" ] )
      dictPrt( memInfoDict, [ "Buffers", "Cached", "SwapCached" ] )
      if os.path.isfile( os.path.join( PROC_MEMINFO_PATH, 'SecPageTables' ) ):
         dictPrt( memInfoDict, [ "KernelStack", "PageTables", "SecPageTables" ] )
      else:
         dictPrt( memInfoDict, [ "KernelStack", "PageTables" ] )
      dictPrt( memInfoDict, [ "Mlocked", "Percpu", "Unevictable" ] )
      dictPrt( memInfoDict, [ "AnonPages", "Mapped", "Shmem" ] )
      dictPrt( memInfoDict, [ "Dirty", "Writeback", "WritebackTmp" ] )
      dictPrt( memInfoDict, [ "Slab", "SUnreclaim", "SReclaimable" ] )
      if "KReclaimable" in memInfoDict:
         dictPrt( memInfoDict, [ "CommitLimit", "Committed_AS", "KReclaimable" ] )
      else:
         dictPrt( memInfoDict, [ "CommitLimit", "Committed_AS" ] )
      dictPrt( memInfoDict, [ "Active", "Active(anon)", "Active(file)" ] )
      dictPrt( memInfoDict, [ "Inactive", "Inactive(anon)", "Inactive(file)" ] )
      if os.path.isfile( os.path.join( PROC_MEMINFO_PATH, 'CmaTotal' ) ):
         dictPrt( memInfoDict, [ "CmaTotal", "CmaFree", "Bounce" ] )
      dictPrt( memInfoDict, [ "HugePages_Total", "HugePages_Free",
                                "HugePages_Rsvd" ] )
      dictPrt( memInfoDict, [ "Hugetlb", "Hugepagesize", "HugePages_Surp" ] )
      if os.path.isfile( os.path.join( PROC_MEMINFO_PATH, "AnonHugePages" ) ):
         dictPrt( memInfoDict, [ "AnonHugePages", "FileHugePages",
                                   "FilePmdMapped" ] )
      if os.path.isfile( os.path.join( PROC_MEMINFO_PATH, "ShmemHugePages" ) ):
         dictPrt( memInfoDict, [ "ShmemHugePages", "ShmemPmdMapped",
                                   "HardwareCorrupted" ] )
      else:
         dictPrt( memInfoDict, [ "HardwareCorrupted" ] )
      dictPrt( memInfoDict, [ "DirectMap4k", "DirectMap2M", "DirectMap1G" ] )
      dictPrt( memInfoDict, [ "SwapTotal", "SwapFree" ] )
      if os.path.isfile( os.path.join( PROC_MEMINFO_PATH, 'Zswap' ) ):
         dictPrt( memInfoDict, [ "Zswap", "Zswapped", "NFS_Unstable" ] )
      else:
         dictPrt( memInfoDict, [ "NFS_Unstable" ] )
      dictPrt( memInfoDict, [ "VmallocTotal", "VmallocUsed", "VmallocChunk" ] )
      print()
      return

   strAttrs = []
   for k in memInfoDict.keys():
      strAttrInfo = StringAttrInfo( PRINTER, k )
      strAttrs.append( strAttrInfo )
   with PRINTER.dict( titleStr ):
      prt = PrintAll()
      for strAttr in strAttrs:
         val = memInfoDict[ strAttr.name ]
         if int( val ) == 0 and not prt.prtAll():
            continue
         #val += ' KiB' + pctPrt( val, 'KiB' ).replace( ' ', '' )
         strAttr.setValue( val )
         PRINTER.addAttributes( strAttr.name, strAttr.getArg() )

def memInfValue( memInfoDict: dict[ str, str ], key: str ) -> str:
   """
   Return the value for the given /proc/meminfo key
   """
   try:
      value = memInfoDict[ key ]
   except KeyError:
      # Return zero string if key does not exist
      return "0"
   return value

def memInfValues( minfoDict: dict[ str, str ], keys: list[ str ] ) -> list[ str ]:
   """
   For a given list of /proc/meminfo entry keys return a list of values
   corresponding to the keys.
   """
   values: list[ str ] = []
   for key in keys:
      value = memInfValue( minfoDict, key )
      values.append( value )
   return values

def memInfLineParse( line: str ) -> tuple[ str, str ]:
   """
   Parse meminfo entries that are each on a separate line:
   ------------------------------------------------------
   """
   values: list[ str ] = line.split()
   return values[ 0 ].rstrip( ':' ), values[ 1 ]

def memInfRead() -> dict[ str, str ]:
   """
   Read the proc meminfo entries into our meminfo dict
   """
   memInfoDict = {}

   with open( PROC_MEMINFO_PATH, "rt", encoding='ascii' ) as file:
      lines = file.readlines()
      for line in lines:
         key, value = memInfLineParse( line )
         memInfoDict[ key ] = value
         if key == "MemTotal":
            memSwapSize.totalMemory = int( value )
         if key == "SwapTotal":
            memSwapSize.totalSwap = int( value )

   return memInfoDict

def memInfProcess() -> dict[ str, str ]:
   """
   Process /proc/meminfo values and place them into a dictionary
   """
   memInfoDict: dict[ str, str ] = memInfRead()
   memSwapSize.totalStorage = memSwapSize.totalMemory + memSwapSize.totalSwap
   return memInfoDict

def memInfProcessFromJson() -> dict[ str, str ]:
   """
   Get /proc/meminfo values from Json and place them into a separate dictionary
   """
   keyword = 'Memory Usage Statistics'
   # Get corresponding dict inside jsonData
   memInfoDict = next( v for k, v in jsonData.items() if keyword in k )
   memSwapSize.totalMemory = int( memInfoDict[ "MemTotal" ] )
   memSwapSize.totalSwap = 0 if 'SwapTotal' not in memInfoDict else int(
         memInfoDict[ "SwapTotal" ] )
   memSwapSize.totalStorage = memSwapSize.totalMemory + memSwapSize.totalSwap
   return memInfoDict

def memInfoProcessAndPrint() -> None:
   """
   Process /proc/meminfo values and place them into a dictionary
   """
   memInfoDict = memInfProcess() if not jsonData else memInfProcessFromJson()
   memInfoPrint( memInfoDict )

class SlabInfo:
   """
   Linux slab info struct
   ----------------------
   name is slab name
   objSize is size of each object and is specified in bytes
   objsPerSlab is the number objects in a single slabe.
   pagePerSlab is the number of pages in a slab
   numSlabs is the number of slabs allocated
   actSlabs is the number of slabs allocated that are active (in use)
   numObjs is the number of objects allocated
   actObjs is the number of objects allocated that are actuce (in use)
   """
   # pylint: disable=too-many-instance-attributes
   # Nine is reasonable, these are kernel slab attributes.
   _name: str
   _pages: int
   _objSize: str
   _objsPerSlab: str
   _pagesPerSlab: str
   _numSlabs: str
   _actSlabs: str
   _numObjs: str
   _actObjs: str

   def __init__( self, constants, pages, slabs, objects ) -> None:
      """ Initialize SlabInfo object:
          Constants: name, size, objects per slab, pages per slab
      """
      self._name = constants[ 0 ]
      self._objSize = constants[ 1 ]
      self._objsPerSlab = constants[ 2 ]
      self._pagesPerSlab = constants[ 3 ]
      self._pages = pages
      self._numSlabs = slabs[ 0 ]
      self._actSlabs = slabs[ 1 ]
      self._numObjs = objects[ 0 ]
      self._actObjs = objects[ 1 ]

   @property
   def name( self ) -> str:
      """ Return the slab name """
      return self._name

   @property
   def pages( self ) -> int:
      """ Return the number pages in use """
      return self._pages

   @pages.setter
   def pages( self, numpages ) -> None:
      """ Set the number of pages in use """
      self._pages = numpages

   @property
   def objSize( self ) -> str:
      """ Return the size of the object in bytes """
      return self._objSize

   @property
   def objsPerSlab( self ) -> str:
      """ Return the number of objects created with each slab entry """
      return self._objsPerSlab

   @property
   def pagesPerSlab( self ) -> str:
      """ Return the number of memory pages allocated with each slab entry """
      return self._pagesPerSlab

   @property
   def numSlabs( self ) -> str:
      """ Return the number slabs defined """
      return self._numSlabs

   @numSlabs.setter
   def numSlabs( self, numSlabs ) -> None:
      """ Set the number of slabs defined """
      self._numSlabs = numSlabs

   @property
   def actSlabs( self ) -> str:
      """ Return the number slabs in use """
      return self._actSlabs

   @actSlabs.setter
   def actSlabs( self, actslabs ) -> None:
      """ Set the number of active slabs """
      self._actSlabs = actslabs

   @property
   def numObjs( self ) -> str:
      """ Return the number objects defined """
      return self._numObjs

   @numObjs.setter
   def numObjs( self, numobjs ) -> None:
      """ Set the number of objects defined """
      self._numObjs = numobjs

   @property
   def actObjs( self ) -> str:
      """ Return the number objects in use """
      return self._actObjs

   @actObjs.setter
   def actObjs( self, actobjs ) -> None:
      """ Set the number of active object """
      self._actObjs = actobjs

def slabsInfoPrint( slabs: list[ SlabInfo ], limit: int = 25 ) -> None:
   """
   Print the slab info for some (limit) or all of the slabs
   Default is to just print the top 25 values since there can be hundreds
   of entries and normally we only care about the largest memory slabs

   Convert number of pages in use to scaled memory size
   """
   kibPerPg = 4
   titleStr = "Kernel Slabs Memory Usage (top 25)"
   if PRETTY_PRINT:
      print( titleStr )
      print( "----------------------------------" )
      print( f"{'Slab Name':25}          Memory Used   Mem-%  Slab Total / Active  "
             "   Pgs/Slab ObjSize Objs/Slab   Objs Total / Active" )
      print( "----------------------------  ----------------  ------  "
             "----------------------  "
             "-------- ------- ----------  -----------------------" )
   else:
      slabName = StringAttrInfo( PRINTER, "Slab Name" )
      memUsed = StringAttrInfo( PRINTER, "Memory Used" )
      memPct = StringAttrInfo( PRINTER, "Mem-%" )
      slabTot = IntAttrInfo( PRINTER, "Slab Total" )
      slabActive = IntAttrInfo( PRINTER, "Slab Active" )
      pgsSlab = IntAttrInfo( PRINTER, "Pgs/Slab" )
      objSize = IntAttrInfo( PRINTER, "ObjSize" )
      objSlab = IntAttrInfo( PRINTER, "Objs/Slab" )
      objTotal = IntAttrInfo( PRINTER, "Objs Total" )
      objActive = IntAttrInfo( PRINTER, "Objs Active" )
   count = 0
   if PRETTY_PRINT:
      for slab in slabs:
         count += 1
         if count > limit:
            break
         # numActSlabs = f"{slab.numSlabs} / {slab.actSlabs}"
         # numActObjs = f"{slab.numObjs} / {slab.actObjs}"
         memSize, units = slab.pages * kibPerPg, "KiB"
         pctMem = ( ( slab.pages * kibPerPg ) / memSwapSize.totalMemory ) * 100
         print( f"{slab.name:29} {memSize:12} {units}  "
                f"{pctMem:5.2f}%  "
                f"{slab.numSlabs:10} / {slab.actSlabs:9}     "
                f"{slab.pagesPerSlab:5} "
                f"{slab.objSize:7} {slab.objsPerSlab:10}  "
                f"{slab.numObjs:10} / {slab.actObjs:10}     " )
      return

   with PRINTER.dict( titleStr ):
      for slab in slabs:
         count += 1
         if count > limit:
            break
         # numActSlabs = f"{slab.numSlabs} / {slab.actSlabs}"
         # numActObjs = f"{slab.numObjs} / {slab.actObjs}"
         memSize, units = slab.pages * kibPerPg, "KiB"
         pctMem = ( ( slab.pages * kibPerPg ) / memSwapSize.totalMemory ) * 100
         slabName.setValue( f"{slab.name}" )
         with PRINTER.dictEntry( slabName.getArg() ):
            memUsed.setValue( f"{memSize} {units}" )
            memPct.setValue( f"{pctMem}" )
            slabTot.setValue( int( slab.numSlabs ) )
            slabActive.setValue( int( slab.actSlabs ) )
            pgsSlab.setValue( int( slab.pagesPerSlab ) )
            objSize.setValue( int( slab.objSize ) )
            objSlab.setValue( int( slab.objsPerSlab ) )
            objTotal.setValue( int( slab.numObjs ) )
            objActive.setValue( int( slab.actObjs ) )
            PRINTER.addAttributes( memUsed.name, memUsed.getArg() )
            PRINTER.addAttributes( memPct.name, memPct.getArg() )
            PRINTER.addAttributes( slabTot.name, slabTot.getArg() )
            PRINTER.addAttributes( slabActive.name, slabActive.getArg() )
            PRINTER.addAttributes( pgsSlab.name, pgsSlab.getArg() )
            PRINTER.addAttributes( objSize.name, objSize.getArg() )
            PRINTER.addAttributes( objSlab.name, objSlab.getArg() )
            PRINTER.addAttributes( objTotal.name, objTotal.getArg() )
            PRINTER.addAttributes( objActive.name, objActive.getArg() )

def slabInfoLineParse( line: str ) -> SlabInfo:
   """
   Parse slabinfo line:
   -------------------
   slabName[0] activeObjects[1] numObjects[2] objSize[3] objsPerSlab[4]
   pgsPerSlab[5] activeSlabs[14] numSlabs[15]
   """
   values: list[ str ] = line.split()
   numPages: int = int( values[ 14 ] ) * int( values[ 5 ] )
   constants = ( values[ 0 ], values[ 3 ], values[ 4 ], values[ 5 ] )
   slabs = ( values[ 14 ], values[ 13 ] )
   objects = ( values[ 2 ], values[ 1 ] )
   slabInfo: SlabInfo = SlabInfo( constants=constants,
                                   pages=numPages,
                                   slabs=slabs,
                                   objects=objects )
   return slabInfo

def slabInfoRead() -> list[ SlabInfo ]:
   """
   Read the proc info into procinfo structure
   """
   slabs: list[ SlabInfo ] = []
   if not jsonData:
      try:
         with open( "/proc/slabinfo", "rt", encoding='ascii' ) as file:
            linenum = 0
            lines = file.readlines()
            for line in lines:
               linenum += 1
               if linenum < 3:
                  continue
               slab = slabInfoLineParse( line )
               slabs.append( slab )

      except PermissionError:
         emsg = "Error: Permission denied /proc/slabinfo requires root privilege"
         print( emsg )
         sys.exit( "Exiting with Permission Error: not root" )
   else:
      keyword = 'Kernel Slabs Memory Usage'
      slabsDict = next( v for k, v in jsonData.items() if keyword in k )
      for slabName, values in slabsDict.items():
         numPages: int = int( values[ 'Slab Active' ] ) * int( values[ 'Pgs/Slab' ] )
         constants = ( slabName, values[ 'ObjSize' ], values[ 'Objs/Slab' ],
                       values[ 'Pgs/Slab' ] )
         slabsArg = ( values[ 'Slab Total' ], values[ 'Slab Active' ] )
         objects = ( values[ 'Objs Total' ], values[ 'Objs Active' ] )
         slabInfo: SlabInfo = SlabInfo( constants=constants,
                                         pages=numPages,
                                         slabs=slabsArg,
                                         objects=objects )
         slabs.append( slabInfo )

   return slabs

def slabInfoProcess() -> None:
   """
   Process the proc
   """
   slabs: list[ SlabInfo ] = slabInfoRead()
   slabs.sort( key=operator.attrgetter( 'pages' ), reverse=True )
   slabsInfoPrint( slabs )

def uptime() -> str:
   """ Return the system uptime """
   with open( PROC_SYS_UPTIME, 'rt', encoding='ascii' ) as pfile:
      line = pfile.readline()
      upsecs = line.split()[ 0 ]
      return upsecs

def numCPUs() -> str:
   """ Return the number of CPUs on the system """
   cmd = [ "getconf", "_NPROCESSORS_ONLN" ]
   with Popen( cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE ) as info:
      out, err = info.communicate()
      out = out.strip().decode( 'utf-8' )
      if err:
         return 0
      return out

@dataclass
class LoadAvgData:
   """ The /proc/loadavg data """
   loadAvgRunIoWait: list[ str ]
   numRunableEntities: str
   numScheduableEntities: str
   lastPid: str

def procLoadAvg() -> LoadAvgData:
   """ Read /proc/avg and return the values """
   ldAvg = LoadAvgData( [ "0.00", "0.00", "0.00" ], "0", "0", "0" )
   with open( PROC_LOADAVG_PATH, 'rt', encoding='ascii' ) as procfile:
      lines = procfile.readlines()
      assert len( lines ) == 1, f"Expecting 1 line found: {len( lines )}"
      line = lines[ 0 ].strip()
      values = line.split( ' ' )
      ldAvg.loadAvgRunIoWait[ 0 ] = values[ 0 ]
      ldAvg.loadAvgRunIoWait[ 1 ] = values[ 1 ]
      ldAvg.loadAvgRunIoWait[ 2 ] = values[ 2 ]
      entities = values[ 3 ].split( '/' )
      ldAvg.numRunableEntities = entities[ 0 ]
      ldAvg.numScheduableEntities = entities[ 1 ]
      ldAvg.lastPid = values[ 4 ]
      return ldAvg

def procLoadAvgPrint() -> None:
   """ Obtain and print the current proc loadavg values """
   if not jsonData:
      now = datetime.now()
      uptm = uptime()
      dtstr = now.strftime( "%Y/%m/%d %H:%M:%S" )
      tstmp = time.time()
      utcoffset = ( datetime.fromtimestamp( tstmp ) -
                    datetime.utcfromtimestamp( tstmp ) ).total_seconds() // 3600
      timezone = time.tzname[ time.daylight ]
      ldAvg = procLoadAvg()
      cpus = numCPUs()
   else:
      keyword = 'System Load Average Statistics'
      procLdAvgDict = next( v for k, v in jsonData.items() if keyword in k )
      cpus = procLdAvgDict[ 'Number of CPUs' ]
      dtstr = procLdAvgDict[ "Date" ].split()[ 0 ] + ' '
      utcoffset = procLdAvgDict[ "Date" ].split()[ 1 ]
      timezone = procLdAvgDict[ "Date" ].split()[ 2 ]
      uptm = float( procLdAvgDict[ "Uptime" ].split()[ 0 ] )
      ldAvg = LoadAvgData( [ "0.00", "0.00", "0.00" ], "0", "0", "0" )
      ldAvg.numRunableEntities = procLdAvgDict[ 'RunnableEntities' ]
      ldAvg.numScheduableEntities = procLdAvgDict[ 'ScheduableEntities' ]
      ldAvg.loadAvgRunIoWait = [ procLdAvgDict[ '1 min Load Average' ],
                                 procLdAvgDict[ '5 min Load Average' ],
                                 procLdAvgDict[ '15 min Load Average' ]
                               ]
      ldAvg.lastPid = procLdAvgDict[ 'Latest pid' ]
   titleStr = f"System Load Average Statistics - Number of CPUs:{cpus:2}"
   if PRETTY_PRINT:
      print( f"\n{titleStr}" )
      print( "--------------------------------------------------" )
      print( f"Date: {dtstr}{utcoffset} {timezone}  Uptime: {uptm} secs  "
             f"Entities: Runnable({ldAvg.numRunableEntities}) "
             f"Scheduable({ldAvg.numScheduableEntities})" )

      print( "      Avg Num Runable/IO-Wait Entities: "
             f"1 min({ldAvg.loadAvgRunIoWait[ 0 ]}) "
             f"5 min({ldAvg.loadAvgRunIoWait[ 1 ]}) "
             f"15 min({ldAvg.loadAvgRunIoWait[ 2 ]})  "
             f"-  Latest pid({ldAvg.lastPid})" )
      return

   with PRINTER.dict( titleStr ):
      cpusAttr = StringAttrInfo( PRINTER, "Number of CPUs" )
      date = StringAttrInfo( PRINTER, "Date" )
      uptimeAttr = StringAttrInfo( PRINTER, "Uptime" )
      runnableEnt = IntAttrInfo( PRINTER, "RunnableEntities" )
      schedEnt = IntAttrInfo( PRINTER, "ScheduableEntities" )
      ldAvg1 = FloatAttrInfo( PRINTER, "1 min Load Average" )
      ldAvg5 = FloatAttrInfo( PRINTER, "5 min Load Average" )
      ldAvg15 = FloatAttrInfo( PRINTER, "15 min Load Average" )
      latestPid = IntAttrInfo( PRINTER, "Latest pid" )
      cpusAttr.setValue( cpus )
      date.setValue( f"{dtstr}{utcoffset} {timezone}" )
      uptimeAttr.setValue( f"{uptm} secs" )
      runnableEnt.setValue( int( ldAvg.numRunableEntities ) )
      schedEnt.setValue( int( ldAvg.numScheduableEntities ) )
      ldAvg1.setValue( float( ldAvg.loadAvgRunIoWait[ 0 ] ) )
      ldAvg5.setValue( float( ldAvg.loadAvgRunIoWait[ 1 ] ) )
      ldAvg15.setValue( float( ldAvg.loadAvgRunIoWait[ 2 ] ) )
      latestPid.setValue( int( ldAvg.lastPid ) )
      PRINTER.addAttributes( cpusAttr.name, cpusAttr.getArg() )
      PRINTER.addAttributes( date.name, date.getArg() )
      PRINTER.addAttributes( uptimeAttr.name, uptimeAttr.getArg() )
      PRINTER.addAttributes( "Entities: Runnable(%d) Scheduable(%d)",
                             runnableEnt.getArg(), schedEnt.getArg() )
      PRINTER.addAttributes( "Avg Num Runable/IO-Wait Entities: 1 min(%f) 5 min(%f) "
                             "15 min(%f) - Latest pid(%d)", ldAvg1.getArg(),
                             ldAvg5.getArg(), ldAvg15.getArg(), latestPid.getArg() )

@dataclass
class PSI:
   """
   Pressure Stall Information data

   svalid: some entry is valid (supported)
   savg:   some entries for the three time values
   stotal: some total stall time
   fvalid: full entry is valid (supported)
   favg:   full entries for the three time values
   ftotal: full total stall time

   some avg10 avg60 avg300 total
   full avg10 avg60 avg300 total
   """
   svalid: bool
   savg: list[ str ]
   stotal: str
   fvalid: bool
   favg: list[ str ]
   ftotal: str

@dataclass
class PSIs:
   """
   Pressure Stall information for each PSI tracked resource

   date: the date time that these entries were recorded
   Entries Currently: cpu, io, irq (6.1), memory PSI values are stored
   """
   date: str
   uptm: str
   cpu: PSI
   ios: PSI
   irq: PSI
   mem: PSI

@dataclass
class PSIsTable:
   """    
   PSIs stored in a table so we can collect these entries over a
   longer period of time, e.g. to produce an hourly report

   psis: Entries are stored here in a deque
   maxent: Maximum entries we store in the deque
   freq: How often we want to add a new entry
   """
   psis: deque[ PSIs ]
   maxent: int
   freq: int

   def __post_init__( self ):
      """ Runs post init for this dataclass """
      print( "schedule the timer driven table update" )

   def addEntry( self, psis: PSIs ) -> None:
      """ Add an entry to the table but restrict table size """
      self.psis.append( psis )
      if len( self.psis ) == self.maxent:
         self.psis.pop()

def psiRead( psiPath: str ) -> PSI:
   """
   Read the specified psi and return the current values
   """
   psi = PSI( False, [ "0.0", "0.0", "0.0" ], "0",
              False, [ "0.0", "0.0", "0.0" ], "0" )
   if not os.path.isfile( psiPath ):
      return psi
   with open( psiPath, 'rt', encoding='ascii' ) as file:
      lines = file.readlines()
      if len( lines ) == 0 or len( lines ) > 2:
         sys.exit( f"Error: Expected 1 or 2 lines, lines={len( lines )}" )
      ctr = 0
      while ctr < len( lines ):
         words = lines[ ctr ].split()
         avg10 = words[ 1 ].split( '=' )[ -1 ] if '=' in words[ 1 ] else words[ 1 ]
         avg60 = words[ 2 ].split( '=' )[ -1 ] if '=' in words[ 2 ] else words[ 2 ]
         avg300 = words[ 3 ].split( '=' )[ -1 ] if '=' in words[ 3 ] else words[ 3 ]
         total = words[ 4 ].split( '=' )[ -1 ] if '=' in words[ 4 ] else words[ 4 ]
         ctr += 1
         if words[ 0 ] == "some":
            psi.svalid = True
            psi.savg = [ avg10, avg60, avg300 ]
            psi.stotal = total
         if words[ 0 ] == "full":
            psi.fvalid = True
            psi.favg = [ avg10, avg60, avg300 ]
            psi.ftotal = total

      return psi

def psisReturn() -> Type[ PSIs ]:
   """
   Return the pressure stall information for each psi category
   """
   psis = PSIs
   now = datetime.now()
   upt = uptime()
   dtstr = now.strftime( "%Y/%m/%d %H:%M:%S" )
   tstmp = time.time()
   utcoffset = ( datetime.fromtimestamp( tstmp ) -
                 datetime.utcfromtimestamp( tstmp ) ).total_seconds() // 3600
   timezone = time.tzname[ time.daylight ]
   psis.date = f"{dtstr}{utcoffset} {timezone}"
   psis.uptm = upt
   psis.cpu = psiRead( PROC_PSI_CPU_PATH )
   psis.ios = psiRead( PROC_PSI_IOS_PATH )
   psis.irq = psiRead( PROC_PSI_IRQ_PATH )
   psis.mem = psiRead( PROC_PSI_MEM_PATH )
   return psis

def psiPrint( name: str, psi: PSI ) -> None:
   """
   Print the specified psi content
   """
   if not psi.svalid and not psi.fvalid:
      return

   nameAttr = StringAttrInfo( PRINTER, "Name" )
   nameAttr.setValue( name )
   avg10 = FloatAttrInfo( PRINTER, "avg10" )
   avg60 = FloatAttrInfo( PRINTER, "avg60" )
   avg300 = FloatAttrInfo( PRINTER, "avg300" )
   total = FloatAttrInfo( PRINTER, "total" )
   if psi.svalid:
      if PRETTY_PRINT:
         print( f"{name} some: "
                f"avg10: {psi.savg[ 0 ]} avg60: {psi.savg[ 1 ]} "
                f"avg300: {psi.savg[ 2 ]} "
                f"total: {psi.stotal}" )
      else:
         with PRINTER.dict( f"{name} some" ):
            avg10.setValue( float( psi.savg[ 0 ] ) )
            avg60.setValue( float( psi.savg[ 1 ] ) )
            avg300.setValue( float( psi.savg[ 2 ] ) )
            total.setValue( float( psi.stotal ) )
            PRINTER.addAttributes( nameAttr.name, nameAttr.getArg() )
            PRINTER.addAttributes( avg10.name, avg10.getArg() )
            PRINTER.addAttributes( avg60.name, avg60.getArg() )
            PRINTER.addAttributes( avg300.name, avg300.getArg() )
            PRINTER.addAttributes( total.name, total.getArg() )
   if psi.fvalid:
      if PRETTY_PRINT:
         print( f"{name} full: "
                f"avg10: {psi.favg[ 0 ]} avg60: {psi.favg[ 1 ]} "
                f"avg300: {psi.favg[ 2 ]} "
                f"total: {psi.ftotal}" )
      else:
         with PRINTER.dict( "{name} full" ):
            avg10.setValue( float( psi.favg[ 0 ] ) )
            avg60.setValue( float( psi.favg[ 1 ] ) )
            avg300.setValue( float( psi.favg[ 2 ] ) )
            total.setValue( float( psi.ftotal ) )
            PRINTER.addAttributes( nameAttr.name, nameAttr.getArg() )
            PRINTER.addAttributes( avg10.name, avg10.getArg() )
            PRINTER.addAttributes( avg60.name, avg60.getArg() )
            PRINTER.addAttributes( avg300.name, avg300.getArg() )
            PRINTER.addAttributes( total.name, total.getArg() )

def crashKernelMem() -> float:
   """ Return memory reserved for the crash kernel or 0 bytes if none. """
   with open( KERNEL_PROC_IOMEM_PATH, 'rt', encoding='ascii' ) as pfile:
      for line in pfile:
         if CRASH_KERNEL_MEMORY in line:
            mrange = line.split( ' ' )[ 2 ]
            values = mrange.split( '-' )
            value = float( int( values[ 1 ], 16 ) ) - \
                    float( int( values[ 0 ], 16 ) ) + 1.0
            valmib = value / 1048576
            return valmib

   return 0.0

def crashKernelMemFromJson() -> float:
   """ Return memory reserved for the crash kernel or 0 bytes if none. """
   keyword = 'Crash Kernel Memory Reserved'
   # Get corresponding dict inside jsonData
   crashKernelMemDict = next( v for k, v in jsonData.items() if keyword in k )
   return float( crashKernelMemDict[ 'size' ].split()[ 0 ] )

def psisPrt( psis: Type[ PSIs ] ) -> None:
   """
   Print the pri values for the specified structure
   """
   if PRETTY_PRINT:
      print( "\nPSI - Pressure Stall Information" )
      print( "---------------------------------" )
      print( f"Date: {psis.date}  Uptime: {psis.uptm} secs" )
      print( "-----------------------------------------------------------" )
      psiPrint( "cpu", psis.cpu )
      psiPrint( "io ", psis.ios )
      psiPrint( "irq", psis.irq )
      psiPrint( "mem", psis.mem )
      print( "-----------------------------------------------------------------"
             "-----------------------" )
      return

   titleStr = "PSI - Pressure Stall Information"
   date = StringAttrInfo( PRINTER, "Date" )
   uptm = StringAttrInfo( PRINTER, "Uptime" )
   date.setValue( psis.date )
   uptm.setValue( psis.uptm )
   with PRINTER.dict( titleStr ):
      PRINTER.addAttributes( "Date: %s Uptime: %s secs", date.getArg(),
                             uptm.getArg() )
      psiPrint( "cpu", psis.cpu )
      psiPrint( "io ", psis.ios )
      psiPrint( "irq", psis.irq )
      psiPrint( "mem", psis.mem )

def psisGetAndPrint() -> None:
   """
   Obtain and print the current psi values.
   """
   if not os.path.isdir( PROC_PSI_PATH ):
      return

   psis = psisReturn()
   psisPrt( psis )

def crashKernelMemPrint() -> None:
   """ Check if memory has been reserved for the crash kernel and print it """
   cmemsize = crashKernelMem() if not jsonData else crashKernelMemFromJson()
   titleStr = "Crash Kernel Memory Reserved"
   if PRETTY_PRINT:
      print( f"{titleStr}: {cmemsize} MiB" )
      return
   attrInfo = StringAttrInfo( PRINTER, "size" )
   with PRINTER.dict( titleStr ):
      attrInfo.setValue( f"{cmemsize} MiB" )
      PRINTER.addAttributes( attrInfo.name, attrInfo.getArg() )

def localOomCheck() -> tuple[ bool, int ]:
   """ Return local OOM Killed Process count, if available """
   try:
      os.stat( LOCAL_OOM_CTR_PATH )
      with open( LOCAL_OOM_CTR_PATH, 'rt', encoding='ascii' ) as procfile:
         lines = procfile.readlines()
         assert len( lines ) == 1, f"Expecting 1 line found: {len( lines )}"
         line = lines[ 0 ].strip()
         killCtr = int( line )
         return True, killCtr
   except FileNotFoundError:
      return False, 0

def genericOomCheck() -> tuple[ bool, int ]:
   """ Return generic OOM Killed Process count, should be available """
   with open( GENERIC_OOM_CTR_PATH, 'rt', encoding='ascii' ) as procfile:
      for line in procfile:
         if line.startswith( 'oom_kill' ):
            line = line.strip()
            value = line.split( ' ' )[ 1 ]
            return True, int( value )
   return False, 0

def oomKillCtrGet() -> int:
   """ Return the OOM Kill count """
   exists, oomkillctr = localOomCheck()
   if exists:
      return oomkillctr

   exists, oomkillctr = genericOomCheck()
   if exists:
      return oomkillctr

   raise FileNotFoundError( 'Could not find the OOM Kill count since boot' )

def oomKilledProcessCount() -> None:
   """ Print the OOM Killed Process count since the system booted """
   titleStr = "OOM Killed Processes Count"
   currTime = StringAttrInfo( PRINTER, "Current Time" )
   upTime = StringAttrInfo( PRINTER, "Uptime" )
   oomKilled = IntAttrInfo( PRINTER, "OOM Killed Processes since boot" )
   if not jsonData:
      oomkillctr = oomKillCtrGet()
      now = datetime.now()
      uptm = uptime()
      dtstr = now.strftime( "%Y/%m/%d %H:%M:%S" )
      tstmp = time.time()
      utcoffset = ( datetime.fromtimestamp( tstmp ) -
                    datetime.utcfromtimestamp( tstmp ) ).total_seconds() // 3600
      timezone = time.tzname[ time.daylight ]
   else:
      oomKillDict = next( v for k, v in jsonData.items() if titleStr in k )
      oomkillctr = oomKillDict[ oomKilled.name ]
      timeStr = oomKillDict[ currTime.name ]
      dtstr = timeStr.split()[ 0 ] + ' '
      utcoffset = timeStr.split()[ 1 ] + timeStr.split()[ 2 ]
      timezone = timeStr.split()[ 3 ]
      uptm = oomKillDict[ upTime.name ].split()[ 0 ]
   if PRETTY_PRINT:
      print( f"Date: {dtstr}{utcoffset} {timezone}  Uptime: {uptm} secs  "
             f"OOM Killed Processes since boot: {oomkillctr}" )
      print()
      return
   with PRINTER.dict( titleStr ):
      currTime.setValue( f"{dtstr} {utcoffset} {timezone}" )
      upTime.setValue( f"{uptm} secs" )
      oomKilled.setValue( oomkillctr )
      PRINTER.addAttributes( currTime.name, currTime.getArg() )
      PRINTER.addAttributes( upTime.name, upTime.getArg() )
      PRINTER.addAttributes( oomKilled.name, oomKilled.getArg() )

def genericThpStats() -> ( list[ str ], list[ str ], list[ str ] ):
   """ Return THP stats from proc vmstat """
   statslist = []
   compactlist = []
   kHugePgsList = []
   hugepgslist = []
   if not jsonData:
      with open( GENERIC_THP_STATS_PATH, 'rt', encoding='ascii' ) as procfile:
         for line in procfile:
            if line.startswith( 'thp_' ):
               statslist.append( line.strip() )
            elif line.startswith( 'compact_' ):
               compactlist.append( line.strip() )
            elif 'khuge' in line:
               kHugePgsList.append( line.strip() )
            elif 'huge' in line:
               hugepgslist.append( line.strip() )
   else:
      def fetchList( keyword ):
         try:
            dictVar = next( v for k, v in jsonData.items() if keyword in k )
         except StopIteration:
            # stats not present in jsonData
            return []
         return [ f'{k} {x}' for k, x in dictVar.items() ]
      keyword = 'Transparent Huge Page Statistics'
      statslist = fetchList( keyword )
      keyword = 'Memory Compaction Statistics'
      compactlist = fetchList( keyword )
      keyword = 'kHugePages Statistics'
      kHugePgsList = fetchList( keyword )
      keyword = 'Hugepages Statistics'
      hugepgslist = fetchList( keyword )
   return statslist, compactlist, kHugePgsList, hugepgslist

def hugePgsListPrint( hugepgslist: list[ str ] ) -> None:
   """ Print the hugepslist if there are entries to print """
   if len( hugepgslist ) == 0:
      return

   # See if actually have any entries
   prt = PrintAll()
   nonZeroEntries = 0
   for hugepg in hugepgslist:
      if hugepg[ -1 ] == '0' and hugepg[ -2 ] == ' ' and not prt.prtAll():
         continue
      nonZeroEntries += 1

   if nonZeroEntries == 0:
      return

   if PRETTY_PRINT:
      print( "\nHugepages Statistics" )
      print( "--------------------" )
      for hugepg in hugepgslist:
         if hugepg[ -1 ] == '0' and hugepg[ -2 ] == ' ' and not prt.prtAll():
            continue
         print( f"{hugepg}" )
      return

   with PRINTER.dict( "Hugepages Statistics" ):
      for hugepg in hugepgslist:
         if hugepg[ -1 ] == '0' and hugepg[ -2 ] == ' ' and not prt.prtAll():
            continue
         attrInfo = IntAttrInfo( PRINTER, hugepg.split()[ 0 ] )
         attrVal = int( hugepg.split()[ 1 ] )
         attrInfo.setValue( attrVal )
         PRINTER.addAttributes( attrInfo.name, attrInfo.getArg() )

def compactListPrint( compactlist: list[ str ] ) -> None:
   """ Print the compactlist if there are entries to print """
   if len( compactlist ) == 0:
      print()
      return

   # See if actually have any entries
   prt = PrintAll()
   nonZeroEntries = 0
   for comp in compactlist:
      if comp[ -1 ] == '0' and comp[ -2 ] == ' ' and not prt.prtAll():
         continue
      nonZeroEntries += 1

   if nonZeroEntries == 0:
      return

   if PRETTY_PRINT:
      print( "\nMemory Compaction Statistics" )
      print( "----------------------------" )
      for comp in compactlist:
         if comp[ -1 ] == '0' and comp[ -2 ] == ' ' and not prt.prtAll():
            continue
         print( f"{comp}" )
      return

   with PRINTER.dict( "Memory Compaction Statistics" ):
      for comp in compactlist:
         if comp[ -1 ] == '0' and comp[ -2 ] == ' ' and not prt.prtAll():
            continue
         attrInfo = IntAttrInfo( PRINTER, comp.split()[ 0 ] )
         attrVal = int( comp.split()[ 1 ] )
         attrInfo.setValue( attrVal )
         PRINTER.addAttributes( attrInfo.name, attrInfo.getArg() )

def thpStatsListPrint( statslist: list[ str ] ) -> None:
   """ Print the statslist if there are entries to print """
   if len( statslist ) == 0:
      print()
      return

   # See if actually have any entries
   prt = PrintAll()
   nonZeroEntries = 0
   for stat in statslist:
      if stat[ -1 ] == '0' and stat[ -2 ] == ' ' and not prt.prtAll():
         continue
      nonZeroEntries += 1

   if nonZeroEntries == 0:
      return

   if PRETTY_PRINT:
      print( "\nTransparent Huge Page Statistics" )
      print( "--------------------------------" )
      for stat in statslist:
         if stat[ -1 ] == '0' and stat[ -2 ] == ' ' and not prt.prtAll():
            continue
         print( f"{stat}" )
      return

   with PRINTER.dict( "Transparent Huge Page Statistics" ):
      for stat in statslist:
         if stat[ -1 ] == '0' and stat[ -2 ] == ' ' and not prt.prtAll():
            continue
         attrInfo = IntAttrInfo( PRINTER, stat.split()[ 0 ] )
         attrVal = int( stat.split()[ 1 ] )
         attrInfo.setValue( attrVal )
         PRINTER.addAttributes( attrInfo.name, attrInfo.getArg() )

def kHugePgsListPrint( kHugePgsList: list[ str ] ) -> None:
   """ Print the khugepslist if there are entries to print """
   if len( kHugePgsList ) == 0:
      return

   # See if actually have any entries
   prt = PrintAll()
   nonZeroEntries = 0
   for khugepg in kHugePgsList:
      if khugepg[ -1 ] == '0' and khugepg[ -2 ] == ' ' and not prt.prtAll():
         continue
      nonZeroEntries += 1

   if nonZeroEntries == 0:
      return

   if PRETTY_PRINT:
      for khugepg in kHugePgsList:
         if khugepg[ -1 ] == '0' and khugepg[ -2 ] == ' ' and not prt.prtAll():
            continue
         print( f"{khugepg}" )
      return

   with PRINTER.dict( "kHugePages Statistics" ):
      for khugepg in kHugePgsList:
         if khugepg[ -1 ] == '0' and khugepg[ -2 ] == ' ' and not prt.prtAll():
            continue
         attrInfo = IntAttrInfo( PRINTER, khugepg.split()[ 0 ] )
         attrVal = int( khugepg.split()[ 1 ] )
         attrInfo.setValue( attrVal )
         PRINTER.addAttributes( attrInfo.name, attrInfo.getArg() )

   print()

def hpStatsPrint() -> None:
   """ Print the HugePages and THP stats from proc vmstat """
   thpStatsList, compactList, kHugePgsList, hugePgsList = genericThpStats()
   hugePgsListPrint( hugePgsList )
   compactListPrint( compactList )
   thpStatsListPrint( thpStatsList )
   kHugePgsListPrint( kHugePgsList )

def runReportingCommands():
   """
   Report system memory and load information we find useful
   """
   PRINTER.start()
   memInfoProcessAndPrint()
   crashKernelMemPrint()
   oomKilledProcessCount()
   slabInfoProcess()
   hpStatsPrint()
   procLoadAvgPrint()
   psisGetAndPrint()
   PRINTER.end()
   print()

def main():
   """
   Drive reporting system (kernel view) memory and load information
   """
   parser = argparse.ArgumentParser( prog=sys.argv[ 0 ] )

   pparser = parser.add_mutually_exclusive_group( required=False )
   pparser.add_argument( '--prtall', action='store_true' )
   pparser.add_argument( '--no-prtall', dest='prtall',
                          action='store_false' )
   parser.add_argument( '--pretty-print', action='store_true' )
   parser.add_argument( '--loadJsonFile', type=str, action='store' )
   parser.set_defaults( prtall=False )

   args = parser.parse_args()

   prtarg = PrintAll()
   if args.prtall:
      prtarg.prtAllEnabled()
   elif not args.prtall:
      prtarg.prtAllDisabled()
   if args.pretty_print:
      global PRETTY_PRINT
      PRETTY_PRINT = True
   if args.loadJsonFile:
      global jsonData
      with open( args.loadJsonFile, "r" ) as f:
         jsonData = json.loads( f.read() )
   runReportingCommands()

if __name__ == "__main__":
   main()
