#!/usr/bin/env python3

# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# archive-init.py
#
# Script that is run once at boot time as part of systemd startup to setup the log
# archiving mechanism for EOS.
#
# Check potential destinations for archiving logs in the following order:
#      - Previous archive configuration
#      - Ext4 ssd drive mounted at /mnt/drive
#      - Ext4 flash drive mounted at /mnt/flash which size is greater than 20GiB
#
# Archive operations are done using the LogMgr/ArchiveLib library.
#
# If the selected destination already contains an archive directory, the latter
# will be rotated to be a new archived instance of the archive, i.e. the directory
# will be renamed to var_archive.<date stamp>.dir and a fresh archive directory will
# be created. This fresh archive directory become the current or active instance of
# archive to hold archived files.
#
# If archiving is enabled (which is the case by default) and we found a valid
# archive destination. We will try to start with at least 50% of the reserved
# space by deleting older archived instances of the archive.
#
# The archiving itself, i.e. copying files from EOS's tmpfs /var/log and /var/core
# done by the cron job LogMgr/etcfiles/archivecheck.sh which is run every minute.

import os
import syslog

import ArchiveLib
import SpaceMgmtLib
import Tac

LOG_PREFIX = 'archive-init'
FLASH_MNTPT = '/mnt/flash'
DRIVE_MNTPT = '/mnt/drive'
SSD_NO_USER_OVERRIDE_FILE = os.path.join( FLASH_MNTPT, 'no_ssd_var' )
LEGACY_ARCHIVE_DISABLED_FILE = os.path.join( FLASH_MNTPT,
                                             '.arista_ssd_archive_disabled' )

def logMsg( *args ):
   syslog.syslog( ' '.join( map( str, args ) ) )

def logErr( *args ):
   syslog.syslog( 'error: ' + ' '.join( map( str, args ) ) )

def ssdNoUseOverride():
   return os.path.isfile( SSD_NO_USER_OVERRIDE_FILE )

def lookupConfiguredArchive():
   if not os.path.isfile( ArchiveLib.Archive.configFilePath ):
      return None

   try:
      return ArchiveLib.Archive.configFileInfo()
   except ( OSError, ValueError ) as e:
      logErr( 'failed to read archive configuration file:', e )
      return None

def selectArchiveDestination():
   configInfo = lookupConfiguredArchive()
   dests = ArchiveLib.Archive.availableDests()

   if configInfo is not None:
      name, path = configInfo[ 'name' ], configInfo[ 'path' ]
      logMsg( 'selecting archive:', 'configured archive found:', name, path )
   elif 'drive' in dests:
      name, path = 'drive', dests[ 'drive' ]
      logMsg( 'selecting archive:', 'ssd drive found' )
   elif 'flash' in dests:
      name, path = 'flash', dests[ 'flash' ]
      logMsg( 'selecting archive:', 'ext4 flash found' )
   else:
      logMsg( 'selecting archive:', 'no valid candidate found' )
      return None

   if path.startswith( DRIVE_MNTPT ) and ssdNoUseOverride():
      logMsg( 'selecting archive: can not use %s since force disabled by %s'
              % ( path, SSD_NO_USER_OVERRIDE_FILE ) )
      return None

   try:
      return ArchiveLib.Archive( name, path )
   except AssertionError as e:
      logErr( 'unable to use selected archive:', e )
      return None

def cleanOldArchiveFormat( archive ):
   """
   Since BUG364128, archive location changed from:
      <ROOTDIR>/archive/var/log
      <ROOTDIR>/archive/var/core
      <ROOTDIR>/var_archive-<TIMESTAMP>.dir
   to:
      <ROOTDIR>/archive/current/var/log
      <ROOTDIR>/archive/current/var/core
      <ROOTDIR>/archive/history/var_archive-<TIMESTAMP>.dir

   If we found the archive setup with the previous locations, we change it to the
   new one.
   """

   try:
      # Move <ROOTDIR>/archive/var to <ROOTDIR>/archive/current/var
      oldPath = os.path.join( archive.baseDirPath, 'var' )
      if os.path.isdir( oldPath ):
         if not os.path.isdir( archive.path ):
            archive.createDir( archive.path )
         archive.replace( oldPath, archive.path )
   except ( OSError, Tac.SystemCommandError ) as e:
      logErr( f'failed to move {oldPath} to {archive.path}:', e )

   try:
      # Move <ROOTDIR>/var_archive* to <ROOTDIR>/archive/history/
      oldPaths = archive.searchArchivedArchivesAt( archive.rootDirPath )
      if oldPaths:
         if not os.path.isdir( archive.historyDirPath ):
            archive.createDir( archive.historyDirPath )
         for path in oldPaths:
            archive.replace( path, archive.historyDirPath )
   except ( OSError, Tac.SystemCommandError ) as e:
      logErr( f'failed to move {oldPaths} to {archive.historyDirPath}:', e )


def rotateArchiveDir( archive ):
   try:
      newDir = archive.rotateArchiveDir()
   except ( OSError, Tac.SystemCommandError ) as e:
      logErr( 'failed to rotate archive directory:', e )
      return None
   else:
      if newDir is None:
         logMsg( 'no archive directory to rotate' )
      else:
         logMsg( 'archive directory rotated to', newDir )
   return None

def setupArchive( archive ):
   if archive.disabled:
      logMsg( 'skipping setup:', 'archive disabled' )
      return

   try:
      archive.setup()
      logMsg( 'archive setup done' )
   except ( OSError, Tac.SystemCommandError,
            SpaceMgmtLib.Quota.QuotaCmdException ) as e:
      logErr( 'archive setup failed:', e )

def saveConfig( archive ):
   try:
      archive.setAsConfig()
      logMsg( 'write archive config done' )
   except ( OSError, ValueError ) as e:
      logErr( 'write archive config failed:', e )

   Tac.run( [ 'sync' ], stdout=Tac.DISCARD, stderr=Tac.DISCARD )

def removeArchivedArchives( archive, pct ):
   if archive.disabled:
      logMsg( 'skipping %s%% free space check:' % pct, 'disabled' )
      return

   logMsg( 'checking that %s%% of the reserved space is free' % pct )

   try:
      quotaPct = archive.quotaPct
      if quotaPct is None:
         quotaPct = 100
      freePct = quotaPct // 2
      reclaimSizeKiB = archive.computeReclaimSpaceSize( freePct )
      size, files = archive.reclaimArchivedArchivesSpace( reclaimSizeKiB )

      if size > 0:
         logMsg( 'deleted archived instances of the archive:',
                 ' '.join( files ),
                 'reclaimedSize=%sKiB' % size )
      else:
         logMsg( 'no archived instances of the archive deleted' )

      Tac.run( [ 'sync' ], stdout=Tac.DISCARD, stderr=Tac.DISCARD )
   except ( OSError, SpaceMgmtLib.Quota.QuotaCmdException,
            Tac.SystemCommandError ) as e:
      logErr( 'deleting archived instances of the archive', e )
      return

def logStatus( archive ):
   status = archive.statusSummary
   statusFmt = (
      # General information
      'name={name}'
      ' path={path}'
      ' exists={exists}'
      ' dirsExist={dirsExist}'
      ' valid={valid}'
      ' disabled={disabled}'
      ' enabled={enabled}'
      # Device information
      ' devName={devName}'
      ' devPresent={devPresent}'
      ' devPhysical={devPhysical}'
      ' devMounted={devMounted}'
      # Archive space usage information
      ' usedKiB={usedKiB}'
      ' usedPct={usedPct}'
      # Filesystem information
      ' fsMntPt={fsMntPt}'
      ' fsSize={fsSize}'
      ' fsUsedKiB={fsUsedKiB}'
      ' fsUsedPct={fsUsedPct}'
      ' fsQuotaMount={fsQuotaMount}'
      # Quota information
      ' quotasOn={quotasOn}'
      ' quotasOff={quotasOff}'
      ' quotaKiB={quotaKiB}'
      ' quotaPct={quotaPct}'
   )

   logMsg( 'archive status:', statusFmt.format( **status ) )

def configureArchive():
   archive = selectArchiveDestination()

   if archive is None:
      return

   cleanOldArchiveFormat( archive )
   rotateArchiveDir( archive )
   setupArchive( archive )
   saveConfig( archive )
   removeArchivedArchives( archive, 50 )
   logStatus( archive )

def main():
   syslog.openlog( LOG_PREFIX )
   configureArchive()

if __name__ == '__main__':
   main()
