#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import os
import errno
from CmisFirmware import CmisMultiportFirmware
import CmisLogs
import Tac

usage = """cmisDfw.py <multiple ports> <commands> <options>

Commands that don't require a firmware image
--version
--abort
--fwfeatures
--run
--commit

Commands that do require a firmware image
--start
--download
--complete

Option to select a firmware image
--bin

Fine-tuning options of the download command
--writeDelay - delay after each write command (1ms, 10ms, 50ms)
--epl-page-delay - delay after EPL page download, sec
--epl-command-delay - delay before sending command CMD104h, sec
--maxBlockSize - override smbus max transaction size, normally read from 01h:163-164
--skip_cdb_block_complete - skip waiting on CDB Block Complete bit ( 00:08 )

Fine-tuning options of the run (switch active image) command
--hitless - attempt a hitless restart to the inactive image, regardless of advertised
  capability in CMD0041, 9Fh:143
Fine-tuning options of the commit (setting default active image) command
--runCommitDelay - configurable delay between --run and --commit CDB calls if used in
  one command

Debugging option
--verbose - print read transactions
--veryverbose - print read and write transactions ( not recommended
                for firmware upgrade )
--trace - print last message from the progress file
--log - redirect prints into a file
--logfile - a prefix for the log and progress files
--tracelog - print messages from the log file
- traceSize - if --trace options is set, defines number of messages to be printed
              from the log file

Supplementary commands:
-dOn - enter diagnostics mode
-dOff - exit diagnostics mode
-j - display output in json format ( only with --version )
--reset - toggle reset
--noTempPolling - force "pure" diagnostics mode, no background temperature polling
-fX - flag the port as an external fabric xcvr

Miscellaneous commands:
--features - run and decode CMD0040 ( Module Features )
--pmfeatures - run and decode CMD0042 ( Performance and Data Monitoring  Features )

Example of use:
./cmisDfw.py 1 2 --start --download --complete --epl-page-delay=0 --bin="image28.bin"
./cmisDfw.py 1 2 --run
./cmisDfw.py 1 2 --commit
./cmisDfw.py 1 2 --version --verbose
./cmisDfw.py 1 2 --fwfeatures
./cmisDfw.py 1 2 --reset
"""

if __name__ == '__main__':
   parser = argparse.ArgumentParser( usage=usage,
                                     description='CMIS Image Download Utility' )
   parser.add_argument( 'ports',
                        nargs='+',
                        help='Front ports: X for fixed, X/Y for modular' )
   parser.add_argument( '-dOn',
                        '--diagsOn',
                        action='store_true',
                        default=False,
                        help='Enter Diags Mode' )
   parser.add_argument( '-dOff',
                        '--diagsOff',
                        action='store_true',
                        default=False,
                        help='Exit Diags Mode' )
   parser.add_argument( '-rt',
                        '--reset',
                        action='store_true',
                        default=False,
                        help='Toggle reset' )
   parser.add_argument( '-b',
                        '--bin',
                        action='store',
                        default=None,
                        help='Downloads the given image file to module' )
   parser.add_argument(
      '-0X',
      '--hex',
      action='store_true',
      default=False,
      help='Interpret the image specified with --bin as Intel Hex format' )
   parser.add_argument( '-s',
                        '--start',
                        action='store_true',
                        default=False,
                        help='Start image download' )
   parser.add_argument( '-d',
                        '--download',
                        action='store_true',
                        default=False,
                        help='Perform image download' )
   parser.add_argument( '--epl-page-delay',
                        type=float,
                        help='Delay after CMD104 EPL page download in sec ' +
                        '(default=0sec)' )
   parser.add_argument( '--epl-command-delay',
                        type=float,
                        help='Delay before CMD104 command in sec ' +
                        '(default=0sec)' )
   parser.add_argument( '--writeDelay',
                        help='write delay after smbus write command in msec ' +
                        '(1ms/10ms/50ms)' )
   parser.add_argument( '-cpl',
                        '--complete',
                        action='store_true',
                        default=False,
                        help='Complete image download' )
   parser.add_argument( '-c',
                        '--commit',
                        action='store_true',
                        default=False,
                        help='Automatically commit the image after downloading' )
   parser.add_argument( '-r',
                        '--run',
                        action='store_true',
                        default=False,
                        help='Run other image' )
   parser.add_argument( '--hitless',
                        type=int,
                        action='store',
                        help='Force hitless or hitfull Run (0|1)' )
   parser.add_argument( '-a',
                        '--abort',
                        action='store_true',
                        default=False,
                        help='Abort image download' )
   parser.add_argument( '-n',
                        '--version',
                        action='store_true',
                        default=False,
                        help='Read version info' )
   parser.add_argument( '-q',
                        '--query',
                        action='store_true',
                        default=False,
                        help='Module query' )
   parser.add_argument( '-f',
                        '--features',
                        action='store_true',
                        default=False,
                        help='Module features' )
   parser.add_argument( '-ff',
                         '--fwfeatures',
                        action='store_true',
                        default=False,
                        help='Module firmware upgrade features' )
   parser.add_argument( '-pmf',
                        '--pmfeatures',
                        action='store_true',
                        default=False,
                        help='PerfMon features' )
   parser.add_argument( '-v',
                         '--verbose',
                        action='store_true',
                        default=False,
                        help='Verbose mode (read transactions)' )
   parser.add_argument( '-vv',
                        '--veryverbose',
                        action='store_true',
                        default=False,
                        help='Very verbose mode (read and write transactions)' )
   parser.add_argument( '-j',
                        '--json',
                        action='store_true',
                        default=False,
                        help='Output in json form' )
   parser.add_argument( '-mbs',
                        '--maxBlockSize',
                        type=int,
                        help='override smbus max block size' )
   parser.add_argument( '--runCommitDelay',
                        type=int,
                        help='delay between run and commit (default is 60 sec)' )
   parser.add_argument( '--log',
                        action='store_true',
                        default=False,
                        help='log status data' )
   parser.add_argument( '--logfile',
                        action='store',
                        default=CmisLogs.defaultLogfilePrefix,
                        help='logfile prefix' )
   parser.add_argument( '--tracelogs',
                        action='store_true',
                        default=False,
                        help='trace status data' )
   parser.add_argument( '--trace',
                        action='store_true',
                        default=False,
                        help='trace upgrade progress' )
   parser.add_argument( '--traceSize',
                        action='store',
                        type=int,
                        default=CmisLogs.defaultTraceSize,
                        help='trace status data' )
   parser.add_argument( '--skip_cdb_block_complete',
                        action='store_true',
                        default=False,
                        help='Skip CDB Block Complete verification' )
   parser.add_argument( '--noTempPolling',
                        action='store_true',
                        default=False,
                        help='skip temperature polling' )
   parser.add_argument( '-fX',
                        '--fabricXcvr',
                        action='store_true',
                        default=False,
                        help='flag the port as an external fabric xcvr' )

   args = parser.parse_args()
   # Validate arguments
   if not args.ports:
      print( "No port specified, use --h for help" )
      exit( 1 ) # pylint: disable=consider-using-sys-exit

   cmisMultiportFirmware = CmisMultiportFirmware()
   if not cmisMultiportFirmware.checkArgs( args ):
      exit( 1 ) # pylint: disable=consider-using-sys-exit

   # If log directory does not exist, create it now
   dirpath = os.path.dirname( args.logfile )
   if not os.path.exists( dirpath ):
      original_umask = os.umask( 0 )
      try:
         os.makedirs( dirpath, mode=0o777 )
      except OSError as exception:
         if exception.errno != errno.EEXIST:
            exit( 1 ) # pylint: disable=consider-using-sys-exit
      os.umask( original_umask )

   # Start parallel execution of per port commands
   cmisMultiportFirmware.runCommands( args )
