#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This program provides a peek/poke interaction with CMIS modules.

import argparse
import sys
import Tac
from CmisSmbus import SmbusBroker, getXcvrNamePrefix
import CmisLogs

class CmisReg:
   def __init__( self, broker ):
      self._b = broker
      # Store the current page
      self.page = self._b.twi_rr( 127 )
      self.ctrlSm = broker.ctrlSm

   def write8( self, page, reg, value ):
      # If the current page is different,
      # reprogram the page before writing the data
      if reg >= 128 and page != self.page:
         self._b.twi_bw( 127, page )
         self.page = page
      self._b.twi_bw( reg, value )

   def writes( self, page, reg, value ):
      self._b.twi_bw( 127, page )
      self._b.twi_msbw( reg, value )

   def read8( self, page, reg, size ):
      # If the current page is different,
      # reprogram the page before reading the data
      if reg >= 128 and page != self.page:
         self._b.twi_bw( 127, page )
         self.page = page
      res = []
      for address in range( reg, reg + size ):
         res.append( self._b.twi_rr( address ) )
      return res

usage = """
cmisReg.py read8 <port> <page> <offset> <size> <options>
cmisReg.py write8 <port> <page> <offset> <data> <options>
cmisReg.py writes <port> <page> <offset> <datas> <options>

   port
      port number for a fixed system
      Linecard/port for a modular system

   page
      Page can be given in decimal or hexadecimal

   address
      Address must be given in decimal or hexadecimal

   data
      Data can be given in decimal or hexadecimal

   datas
      String of data in hex format, separated by commas (max 16 params)

   size
      Size must be given in decimal

   options:
     -dOn  - stay in the diagnostic mode after script termination
     -dOff - return to operational mode before script termination (obsoleted)
     -v - verbose mode
     -fX - flag the port as an external fabric xcvr

Examples:
   1) To read 8 bytes from page 0x10 address 128, the module is installed
      in linecard 6 port 16:
         cmisReg.py read8 6/16 0x10 128 8

   2) To write 0xFF to page 0x10 address 128, the module is installed
      in port 16, fixed system:
         cmisReg.py write8 16 0x10 128 0xFF

   2) To populate page 0x12 registers 200-201 with 0xFC18, the module is installed
      in port 16, fixed system:
         cmisReg.py writes 16 0x12 200 0xFC,0x18

To execute the script, the module is temporarily put into a DIAGNOSTIC mode
Use -dOn option to stay in the DIAGNOSTIC mode when the script is completed
"""

def strToInt( argStr ):
   if argStr.startswith( '0x' ):
      return int( argStr, 16 )
   else:
      return int( argStr, 10 )

def doPrintByteString( output, addr ):
   lanesize = 16
   startAddr = addr // lanesize * lanesize
   # align output buffer to boundary of lanesize
   for offset in range( startAddr, addr ):
      output.append( ".... " )
   for offset in range( len( output ) ):
      if not offset % lanesize:
         addrStr = f"{startAddr + offset:3}: "
         if offset > 0:
            sys.stdout.write( "\n" )
         sys.stdout.write( addrStr )
      sys.stdout.write( str( output.pop() ) )
   sys.stdout.write( "\n" )

def terminate( args, broker ):
   # If option diagsOn is selected, stay in the diagnostic mode
   if not args.diagsOn:
      broker.exitDiag()
   else:
      print( "Note: The system is in diags mode " )

def runCommand( port, args ):
   '''
   Implements methods to access registers of Cmis transceiver

   Arguments:
   port : str
          front port associated with the transceiver
   args : argparse.Namespace
          user-supplied arguments
   '''
   page = strToInt( args.page )
   address = strToInt( args.address )
   data = None
   if 'data' in args:
      data = strToInt( args.data )
   size = None
   if 'size' in args:
      size = strToInt( args.size )

   controller = f"{ getXcvrNamePrefix( args.fabricXcvr ) }{ args.port }"
   broker = SmbusBroker( controller,
                         args.verbose,
                         args.verbose )
   broker.enterDiag()
   cmisReg = CmisReg( broker )
   if args.op == 'write8':
      cmisReg.write8( page, address, data )
      # pylint: disable-next=consider-using-f-string
      print( "{}: {} page 0x{:x} address {} data 0x{:x}".format(
         controller, args.op, page, address, data ) )
   elif args.op == 'writes':
      datas = args.datas
      # Limit length of data string by 16 parameters, 5 bytes each
      if len( datas ) > 80:
         print( "Error: input data string exceeds max size of 16 parameters" )
         terminate( args, broker )
         exit( 1 ) # pylint: disable=consider-using-sys-exit
      # pylint: disable-next=consider-using-f-string
      print( "{}: {} page 0x{:x} address {} datas {}".format(
         controller, args.op, page, address, datas ) )
      hexdatas = [ int( item, 16 ) for item in args.datas.split( ',' ) ]
      cmisReg.writes( page, address, bytearray( hexdatas ) )
   elif args.op == 'read8':
      output = cmisReg.read8( page, address, size )
      for index, value in enumerate( output ):
         output[ index ] = f"0x{value:02x} "
      output.reverse()
      doPrintByteString( output, address )

   terminate( args, broker )

@CmisLogs.filelock
def run( port, args ):
   runCommand( port, args )

if __name__ == '__main__':
   parser = argparse.ArgumentParser( description='CMIS Registers read/write',
                                     usage=usage )

   # Create subparsers for read and write operations
   operationParser = parser.add_subparsers( dest="op",
                                            help="specifies operation type" )
   parserRead8 = operationParser.add_parser( "read8", usage=usage )
   parserWrite8 = operationParser.add_parser( "write8", usage=usage )
   parserWrites = operationParser.add_parser( "writes", usage=usage )

   for op_parser in [ parserRead8, parserWrite8, parserWrites ]:
      op_parser.add_argument( "port",
                              help='Front port: X for fixed, X/Y for modular' )
      op_parser.add_argument( "page",
                              help="specifies page" )
      op_parser.add_argument( "address",
                              help="specifies address" )
      op_parser.add_argument( '-v',
                           '--verbose',
                           action='store_true',
                           default=False,
                           help='Verbose mode' )
      op_parser.add_argument( '-dOff',
                              '--diagsOff',
                              action='store_true',
                              default=False,
                              help='exit diagnostic mode' )
      op_parser.add_argument( '-dOn',
                              '--diagsOn',
                              action='store_true',
                              default=False,
                              help='stay in the diagnostic mode' )
      op_parser.add_argument( '-fX',
                              '--fabricXcvr',
                              action='store_true',
                              default=False,
                              help='flag the port as an external fabric xcvr' )

   for op_parser in [ parserWrite8 ]:
      op_parser.add_argument( "data",
                              help="specifies data" )

   for op_parser in [ parserWrites ]:
      op_parser.add_argument( "datas",
                              help="specifies data array" )

   for op_parser in [ parserRead8 ]:
      op_parser.add_argument( "size",
                              help="specifies read size" )

   arguments = parser.parse_args()

   if not arguments.port:
      print( "No port specified, use --h for help" )
      exit( 1 ) # pylint: disable=consider-using-sys-exit

   run( arguments.port, arguments )
