#!/bin/bash
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Gather relevant system information and start apropriate 'ptp4l' and 'phc2sys'
# services by creating corresponding configuration files which triggers
# 'ptp-restarter.service'.

#***********************************************************************************#
# Declare variables.
is_modular=$( [[ -e /sys/arista/election_manager ]] &&\
              printf "true" || printf "false" ) # 'true' for modular, 'false' fixed;
id=0 # slot id, '0' for fixed system;
is_active="true" # 'false' for standby slot, 'true' for active slot and fixed system;
interface=() # internal interface, management interfaces;
capability=() # time stamping ('sw-ts'/'hw-ts'), matches 'interface' elements;
id_phc=() # phcs' ids for interfaces, matches 'interface' elements;

#***********************************************************************************#
# Retrive system's information.
retrieve_info() {
   local arg=("${@}") # 'is_modular'

   if [[ ${arg[0]} == "true" ]]; then
      id=$(< /sys/arista/election_manager/local_slot_id)
      if [[ $(< /sys/arista/election_manager/local_redundancy_mode) ==\
            "standby" ]]; then
         is_active="false"
      fi
   fi
}

#***********************************************************************************#
# Retrieve internal and management interfaces' names.
retrieve_interface() {
   local arg=("${@}") # 'id' 'is_modular''

   # Retrieve internal interface, add it as the first element of 'interface' array;
   if [[ ${arg[1]} == "true" ]]; then
      interface[0]="internal${arg[0]}_1"
   else
      interface[0]=""
   fi

   # Retrieve management interfaces.
   # Sample string:
   # 3: ma2_2: <NO-CARRIER,BROADCAST,MULTICAST,UP> mtu 1500 qdisc mq state DOWN...'.
   local interface_management=($(ip link |\
           sed -n "s/^[[:digit:]]\+:\s\+\(ma[[:digit:]]\(_[[:digit:]]\)\?\).*/\1/p"))

   interface+=(${interface_management[@]})
}

#***********************************************************************************#
# Retrieve interface properties: SW/HW time stamping and phc's id. If either TX or RX
# does not support HW time stamping, use SW time stamping for both. This will also
# control the need for running 'phc2sys' as for SW timestamping 'ptp4l' adjusts
# system clock itself.
retrieve_property() {
   local output=""

   # Check if capability is supported.
   is_supported_capability() {
      local arg=("${@}") # 'capability', 'value';
      local value=$( printf "${output}" | sed -n "s/^\s\+\(${arg[0]}\).*/\1/p" )

      if [[ ${value} == ${arg[0]} ]]; then
         return 1
      else
         return 0
      fi
   }

   # Read phc's id.
   retrieve_id_phc() {
      local result=$( printf "${output}" |\
                    sed -n "s/^PTP Hardware Clock:\s\+\([[:digit:]]\+\|none\)/\1/p" )

      if [[ ${result} == "none" ]]; then
         result=0
      else
         (( ++result ))
      fi
      return ${result}
   }


   for (( i = 0; i < ${#interface[@]}; ++ i )); do
      if [[ -n ${interface[$i]} ]]; then # process detected interfaces;
         local result=0
         output=$( ethtool -T ${interface[$i]} )

         is_supported_capability "hardware-transmit"
         result=${?}
         is_supported_capability "hardware-receive"
         result=$(( ${?} & ${result} ))

         if (( ${result} == 1 )); then # HW time stamping;
            capability[$i]="hardware"
         else
            result=0
            is_supported_capability "software-transmit"
            result=${?}
            is_supported_capability "software-receive"
            result=$(( ${?} & ${result} ))

            if (( ${result} == 1 )); then # SW time stamping;
               capability[$i]="software"
            else # unable to detect time stamping;
               capability[$i]=""
            fi
         fi
         retrieve_id_phc
         result=${?}
         if (( ${result} == 0 )); then
            id_phc[$i]=""
         else
            (( --result ))
            id_phc[$i]=$( printf "${result}" )
         fi
      else # interface is '', i.e., is not detected;
         capability[$i]=""
         id_phc[$i]=""
      fi
   done
}

#***********************************************************************************#
# Retrieve system info, interfaces and their properties. Create 'ptp4l' and 'phc2sys'
# configuration files accordingly.
# TODO: POTENTIALLY process management interfaces and frontpanel phc for fixed
# system and active supervisor on modular system. Standby supervisor only
# synchronizes time from the active peer.
# Caller will probably need to provide a configuration file for this script for such
# elaborate configuration.

main() {
   local result=0
   local message=""
   # Define configuration files 'ptp4l' and 'phc2sys'. 'ptp4l' uses the configuration
   # file directly. 'phc2sys' uses same file as 'ptp4l'
   # ('ptp4l@<interface>.conf') for global settings, and 'phc2sys'-service uses
   # 'phc2sys@<interface>.conf' as environment file to specify command-line options
   # for 'phc2sys'.
   local configuration_ptp4l=("") # goes to 'ptp4l@.conf' for 'ptp4l' and 'phc2sys';
   local configuration_phc2sys=("") # goes to 'phc2sys@.conf' for 'phc2sys@.service';
   local configuration_restarter="" # goes to 'ptp-restarter.conf';

   retrieve_info ${is_modular}
   retrieve_interface ${id} ${is_modular}
   retrieve_property

   message+="System: "
   if [[ ${is_modular} == "false" ]]; then # fixed system;
      message+="fixed;\n"

   else # modular system;
      configuration_ptp4l[0]+="[global]\n"
      configuration_ptp4l[0]+="logging_level 6\n"
      configuration_ptp4l[0]+="time_stamping ${capability[0]}\n"
      configuration_ptp4l[0]+="verbose 1\n"
      configuration_ptp4l[0]+="uds_address /var/run/ptp4l@${interface[0]}\n"
      configuration_ptp4l[0]+="use_syslog 0\n"

      if [[ "${is_active}" == "true" ]]; then # modular system, active supervisor;
         message+="modular (active);\n"
         # Synchronize time over internal interface TO standby supervisor.
         configuration_ptp4l[0]+="[${interface[0]}]\n"
         configuration_ptp4l[0]+="masterOnly 1\n" # active is M;

         if [[ ${capability[0]} == "hardware" ]]; then # enable 'phc2sys' for HW TS;
            configuration_phc2sys[0]+="OPTION=-w -s clock_realtime\
 -c /dev/ptp${id_phc[0]}\n"
         fi
      else # modular system, standby supervisor;
         message+="modular (standy);\n"
         # Synchronize time over internal interface FROM active supervisor.
         configuration_ptp4l[0]+="slaveOnly 1\n" # '[global]' section, standby is S;
         configuration_ptp4l[0]+="[${interface[0]}]\n"

         if [[ ${capability[0]} == "hardware" ]]; then # enable 'phc2sys' for HW TS;
            configuration_phc2sys[0]+="OPTION=-w -s /dev/ptp${id_phc[0]}\
 -c clock_realtime\n"
         fi
      fi
   fi

   rm -f /etc/ptp-restarter.conf # remove 'ptp-restarter.service' configuration;
   message+="Interface properties (timestamp, phc[, ptp4l[, phc2sys]]):\n"

   for (( i = 0; i < ${#interface[@]}; ++i )); do
      if [[ -n ${interface[$i]} ]]; then
         # Remove old configuration files before possibly writing new one.
         rm -f /etc/ptp4l@${interface[$i]}.conf
         rm -f /etc/phc2sys@${interface[$i]}.conf

         message+="  ${interface[$i]}: "

         if [[ -n ${capability[$i]} ]]; then
            message+="${capability[$i]}, "
         else
            message+="<unknown>, "
         fi

         if [[ -n ${id_phc[$i]} ]]; then
            message+="/dev/ptp${id_phc[$i]}"
         else
            message+="<unknown>"
         fi

         if [[ -n ${capability[$i]} ]] && [[ -n ${id_phc[$i]} ]]; then
            if [[ -n "${configuration_ptp4l[$i]}" ]]; then # create 'ptp4l@.conf';
               message+=", enabled"
               printf "${configuration_ptp4l[$i]}" >\
                  /etc/ptp4l@${interface[$i]}.conf
               configuration_restarter+="ptp4l@${interface[$i]}.service "

               if [[ -n "${configuration_phc2sys[$i]}" ]]; then # 'phc2sys@.conf';
                  message+=", enabled"
                  printf "${configuration_phc2sys[$i]}" >\
                     /etc/phc2sys@${interface[$i]}.conf
                  configuration_restarter+="phc2sys@${interface[$i]}.service "
               else
                  message+=", disabled"
               fi
            else
               message+=", disabled"
            fi
         fi
         message+=";\n"
      fi
   done

   # Trim trailing whitespaces.
   configuration_restarter=$( printf "${configuration_restarter}" | awk '{$1=$1};1' )
   [[ -n ${configuration_restarter} ]] &&\
      printf "SERVICE=${configuration_restarter}\n" > /etc/ptp-restarter.conf

   printf "${message}"

   return "${result}"
}

#***********************************************************************************#
# Run 'main'.
main

exit ${?}
