# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import socket
import traceback

import PyClient
import Tac
import Tracing

traceHandle = Tracing.Handle( 'AaaApiClient' )
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

PYCLIENT_CONNECT_TIMEOUT = 30
PYCLIENT_RECV_TIMEOUT = 30
PyClientErrors = ( socket.error, ValueError )

def parsePrivLevel( result, defaultPrivLevel ):
   privLevel = result.av.get( 'privilegeLevel' )
   if privLevel is not None and privLevel.isdigit():
      try:
         return int( privLevel )
      except ValueError:
         pass
   warn( 'Warning: invalid privilege level:', privLevel )
   return defaultPrivLevel

class AaaApiClient:
   # This class is not threadsafe due to the underlying implementation using
   # PyClient which is not threadsafe.

   def __init__( self, sysname ):
      self.sysname_ = sysname
      self.aaaPc_ = None

   def _aaaPc( self ):
      """Returns a PyClient instance to the Aaa agent."""
      if self.aaaPc_ is None:
         try:
            self.aaaPc_ = PyClient.PyClient(
               self.sysname_, "Aaa",
               execMode=PyClient.Rpc.execModeThreadPerConnection,
               initConnectCmd="import AaaApi",
               connectTimeout=PYCLIENT_CONNECT_TIMEOUT,
               recvTimeout=PYCLIENT_RECV_TIMEOUT )
         except OSError as e:
            warn( "socket.error during PyClient setup:", str( e ) )
            self.aaaPc_ = None
         except PyClient.RpcError as e:
            warn( "PyClient.RpcError during PyClient setup:", str( e ) )
            self.aaaPc_ = None
      return self.aaaPc_

   def _aaaRpc( self, cmd ):
      """Evaluates Python code 'cmd' on the Aaa agent."""
      pc = self._aaaPc()
      if not pc:
         return None
      start = Tac.now()
      try:
         evalResult = pc.eval( cmd )
      except PyClientErrors as e:
         warn( "RPC Aaa Exception:", e.__class__.__name__, str( e ) )
         info( traceback.format_exc() )
         self.aaaPc_ = None
         return None

      # Temp hack for now to remove the "L" from integers, until Aaa is py3 ready.
      # The bad evalResult looks like: (SyntaxError in py3)
      #   "Value('AaaApi::AuthenAndAuthzResult', ** {'status': 'authenFailed', 
      #    'sessionId': 0, 'av': {}, 'uid': 4294967295L, 'gid': 4294967295L})" )"
      if "AaaApi::AuthenAndAuthzResult" in evalResult:
         import re # pylint: disable=import-outside-toplevel
         evalResult = re.sub("4294967295L", "4294967295", evalResult)
      result = Tac.strepToValue( evalResult )
      elapsed = max( Tac.now() - start, 0.0 )
      # pylint: disable-next=consider-using-f-string
      warn( "RPC Aaa:", cmd, "=", result, "(", "%.3fs)" % elapsed )
      return result
  
   def closeSession( self, sessionId ):
      if not sessionId or sessionId == "":
         return None
      # pylint: disable-next=consider-using-f-string
      return self._aaaRpc( "AaaApi.closeSession( %r )" % int( sessionId ) )

   def getSessionRoles( self, sessionId ):
      if not sessionId or sessionId == "":
         return None
      # pylint: disable-next=consider-using-f-string
      return self._aaaRpc( "AaaApi.getSessionRoles( %r )" % int( sessionId ) )

   def authenticateAndAuthorizeSession( self, user, password, service, remoteHost,
         remoteUser=None, tty=None, localAddr=None, localPort=None,
         remotePort=None ):
      # pylint: disable-next=consider-using-f-string
      cmd = ( 'AaaApi.authenticateAndAuthorizeSession( user=%r, password=%r, '
        'service=%r, remoteHost=%r, remoteUser=%r, tty=%r, localAddr=%r, '
        'localPort=%r, remotePort=%r )' %
        ( user, password, service, remoteHost, remoteUser, tty, localAddr,
           localPort, remotePort ) )
      return self._aaaRpc( cmd )

   def createAndAuthorizeSession( self, user, service, authenMethod, remoteHost,
         remoteUser=None, tty=None, validUserOnly=True, localAddr=None,
         localPort=None, remotePort=None ):
      # pylint: disable-next=consider-using-f-string
      cmd = ( 'AaaApi.createAndAuthorizeSession( user=%r, service=%r, '
              'authenMethod=%r, remoteHost=%r, remoteUser=%r, tty=%r, '
              'validUserOnly=%r, localAddr=%r, localPort=%r, '
              'remotePort=%r )' %
              ( user, service, authenMethod, remoteHost, remoteUser, tty,
                 validUserOnly, localAddr, localPort, remotePort ) )
      return self._aaaRpc( cmd )
