# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os, re, bisect
import errno

import AaaPluginLib
from AaaPluginLib import TR_ERROR, TR_AUTHEN, TR_AUTHZ
from AaaPluginLib import TR_SESSION, TR_INFO, TR_DEBUG
from BothTrace import traceX as bt
from BothTrace import Var as bv
import Logging
import SecretCli
import Tac
from Tracing import traceX

localUserConfigReactor_ = None
mergedUsersConfigReactor_ = None

SshOptions = Tac.Type( "Mgmt::Ssh::Options" )

AAA_ROOT_PASSWORD_NOTUPDATED = Logging.LogHandle(
      "AAA_ROOT_PASSWORD_NOTUPDATED",
      severity=Logging.logWarning,
      fmt="Password setting for root not changed due to internal error (%s)",
      explanation="The password for the root user could not be changed "
                  "from its current value. The old password (if any) "
                  "continues to be valid for the account. "
                  "This is a potential security risk.",
      recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

AAA_REMOTE_LOGIN_DENIED_BY_POLICY = Logging.LogHandle(
      "AAA_REMOTE_LOGIN_DENIED_BY_POLICY",
      severity=Logging.logWarning,
      fmt="Remote login from %s denied for user %s due to authentication policy",
      explanation="A remote login attempt was denied due to authentication policy.  "
                  "For example, the configured policy may prohibit remote logins "
                  "for accounts with empty passwords.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

AAA_SSHDIR_FILESYSTEM_ERROR = Logging.LogHandle(
      "AAA_SSHDIR_FILESYSTEM_ERROR",
      severity=Logging.logError,
      fmt="Error generating ssh %s at path %s (%s)",
      explanation="An attempt to create a file in the ssh directory failed "
                  "because of a filesystem error.",
      recommendedAction="Check to see if the filesystem has run out of disk space.  "
                        "(Potentially from large log files) "
                        "Contact TAC Support if this problem persists " )

AAA_INVALID_REGEX_IN_ROLE = Logging.LogHandle(
      "AAA_INVALID_REGEX_IN_ROLE",
      severity=Logging.logError,
      fmt="Rule %d of Role %s has an invalid regular expression",
      explanation="The specified rule has an invalid regular expression,"
                  " and will not take effect until the syntax is fixed.",
      recommendedAction="Fix the role configuration." )

class CompiledRule:
   def __init__( self, rule ):
      self.permit = ( rule.action == 'permit' )
      self.modeKey = rule.modeKey
      self.modeKeyRe = re.compile( rule.modeKey )
      self.regex = re.compile( rule.regex )

class CompiledRole:
   '''A list of compiled rules for the role.'''
   def __init__( self, name ):
      self.name = name
      self.seqs = [] # sorted sequence numbers
      self.rules = dict() # seq -> CompiledRule # pylint: disable=use-dict-literal

# dictionary of roleName -> CompiledRole
compiledRoles = {}

def removeFile( filename ):
   try:
      os.unlink( filename )
   except OSError:
      pass

def setUpSshDir( sshDirPath ):
   try:
      os.mkdir( sshDirPath, 0o700 )
   except OSError as e:
      if e.errno == errno.EEXIST and os.path.isdir( sshDirPath ):
         pass
      else:
         Logging.log( AAA_SSHDIR_FILESYSTEM_ERROR, 'directory', sshDirPath, e )

def writeSshDirFile( filename, content, logString ):
   try:
      with open( filename, 'w' ) as sshDirFile:
         sshDirFile.write( content )
      os.chmod( filename, 0o600 )
   except OSError as e:
      bt( TR_ERROR, "Failed to write file ", bv( filename ), ":", bv( e.strerror ) )
      Logging.log( AAA_SSHDIR_FILESYSTEM_ERROR, logString, filename, e.strerror )

class LocalUserConfigReactor( Tac.Notifiee ):
   '''When root passwd changes, update it in /etc/shadow'''

   notifierTypeName = "LocalUser::Config"
   def __init__( self, cfg, agent ):
      self.cfg_ = cfg
      self.agent_ = agent
      self.localUserAcctReactors_ = {}
      self.roleConfigReactors_ = {}
      self.sshDirPath_ = '/root/.ssh/'
      Tac.Notifiee.__init__( self, cfg )
      self.handleEncryptedRootPasswd()
      for role in self.notifier_.role:
         self.handleRole( role )

   @Tac.handler( 'encryptedRootPasswd' )
   def handleEncryptedRootPasswd( self ):
      bt( TR_INFO, "handle root password", bv( self.cfg_.encryptedRootPasswd ) )
      cmd = [ "/usr/sbin/usermod",
              "-p", self.cfg_.encryptedRootPasswd,
              "root"
            ]
      try:
         Tac.run( cmd )
      except Tac.SystemCommandError as e:
         Logging.log( AAA_ROOT_PASSWORD_NOTUPDATED, str( e ) )

   @Tac.handler( 'role' )
   def handleRole( self, key ):
      if key in self.notifier_.role:
         # new entry
         compiledRoles[ key ] = CompiledRole( key )
      else:
         del compiledRoles[ key ]
      Tac.handleCollectionChange( RoleConfigReactor, key, 
                                  self.roleConfigReactors_, 
                                  self.cfg_.role, 
                                  reactorArgs=() )

   def close( self ):
      for r in self.localUserAcctReactors_.values():
         r.close()
      self.localUserAcctReactors_.clear()
      for r in self.roleConfigReactors_.values():
         r.close()
      self.roleConfigReactors_.clear()
      Tac.Notifiee.close( self )

class RoleConfigReactor( Tac.Notifiee ):
   notifierTypeName = "LocalUser::Role"

   def __init__( self, role ):
      Tac.Notifiee.__init__( self, role )
      self.role_ = compiledRoles[ role.name ]
      for seq in self.notifier_.rule:
         self.handleRule( seq )

   @Tac.handler( 'rule' )
   def handleRule( self, seq ):
      if seq in self.notifier_.rule:
         # new or updated rule
         try:
            rule = self.notifier_.rule[ seq ]
            r = CompiledRule( rule )
            bisect.insort( self.role_.seqs, seq )
            self.role_.rules[ seq ] = r
            return
         except SyntaxError:
            # The CLI should have protected us so it should be impossible,
            # but lets just handle it here.
            Logging.log( AAA_INVALID_REGEX_IN_ROLE, seq, self.notifier_.name )
      # deleted rule
      if seq in self.role_.rules:
         self.role_.seqs.remove( seq )
         del self.role_.rules[ seq ]

class EnableAuthenticator( AaaPluginLib.Authenticator ):
   # state machine values
   needPassword = 1
   askedPassword = 2
   failed = 3
   succeeded = 4

   stateMap = { 1 : "needPassword", 2 : "askedPassword", 3 : "failed",
                4 : "succeeded" }

   # pylint: disable-next=redefined-builtin
   def __init__( self, config, aaaConfig, method, type, service, remoteHost,
                 remoteUser, tty, user, privLevel ):
      self.config = config
      self.privLevel = privLevel
      self.attempts = 0
      self.state = self.needPassword
      AaaPluginLib.Authenticator.__init__( self, aaaConfig, 
                                           method, type, service,
                                           remoteHost, remoteUser, tty, user )
      assert type == 'authnTypeEnable'

   def authenticate( self, *responses ):
      configuredPasswd = self.config.encryptedEnablePasswd
      if self.state == self.needPassword:
         if configuredPasswd == "":
            # There is no 'enable' password set -- allow the user to switch
            # into 'enable' mode.
            return self.transition( self.succeeded, 'success' )
         else:
            return self.transition( self.askedPassword, 'inProgress',
                                    [ self.passwordPrompt() ] )
      elif self.state == self.askedPassword:
         if len( responses ) == 1:
            traceX( TR_AUTHEN, "authenticate: state=askedPassword" )
            self.attempts += 1
            if self.checkPassword( responses[ 0 ] ):
               return self.transition( self.succeeded, 'success' )
            elif self.attempts < 3:
               return self.transition( self.askedPassword, 'inProgress',
                                       [ self.passwordPrompt() ] )
            else:
               failMsg = Tac.Value( "AaaApi::AuthenMessage",
                                    style='error', text='Bad secret' )
               return self.transition( self.failed, 'fail', [ failMsg ] )
      else:
         # pylint: disable-next=undefined-variable
         BT( TR_ERROR, "EnableAuthenticator.authenticate unexpected state:",
             bv( self.state ) )
         # fall through
      return self.transition( self.failed, 'fail' )

   # pylint: disable-next=dangerous-default-value
   def transition( self, state, authenStatus, messages=[] ):
      sm = self.stateMap
      traceX( TR_AUTHEN, "EnableAuthenticator transitioning from", sm[ self.state ],
              "to", sm[ state ] )
      self.state = state
      r = { "status" : authenStatus, "messages" : messages, "user" : self.user,
            "authToken" : "" }
      return r

   def checkPassword( self, password ):
      # Pass in the current encrypted password as the salt for the
      # encryption, so we don't choose a new salt, which will result
      # in the password entered at the CLI to be encrypted to a
      # different string, so even if it matches the configured passwd,
      # we won't be able to tell...
      configuredPasswd = self.config.encryptedEnablePasswd
      if configuredPasswd == '':
         return False
      encryptedPasswd = SecretCli.encrypt( password, configuredPasswd )
      # pylint: disable-next=superfluous-parens
      return ( configuredPasswd == encryptedPasswd )

class LocalUserAuthenticator( AaaPluginLib.BasicUserAuthenticator ):
   # pylint: disable-next=redefined-builtin
   def __init__( self, config, aaaConfig, method, type, service, remoteHost,
                 remoteUser, tty, user, privLevel ):
      self.config = config
      AaaPluginLib.BasicUserAuthenticator.__init__( self, aaaConfig, method, 
                                                    type, service,
                                                    remoteHost, remoteUser, tty,
                                                    user, privLevel )
      # BUG 13784 requires that we return unavailable for unknown user,
      # but we don't log the typical FALLBACK message.
      self.logFallback = False
      
   def checkUser( self, user ):
      traceX( TR_AUTHEN, "checkUser: user=", user )
      return user in self.config.acct

   def checkEmptyPassword( self, user ):
      return self.checkPassword( user, "" )

   def checkPassword( self, user, password ):
      traceX( TR_AUTHEN, "checkPassword: user=", user, "password=*****" )
      acct = self.config.acct.get( user )
      failMsg = Tac.Value( "AaaApi::AuthenMessage",
                           style='error', text='Bad secret' )
      if acct:
         if self.policyDeniesLogin( acct ):
            host = self.remoteHost if self.remoteHost else "(unknown host)"
            Logging.log( AAA_REMOTE_LOGIN_DENIED_BY_POLICY, host, user )
            return { 'state': self.failed, 'authenStatus': 'fail',
                     'messages': [ failMsg ] }
         if acct.encryptedPasswd not in ('', '*'):
            encrypted = SecretCli.encrypt( password, acct.encryptedPasswd )
         elif acct.encryptedPasswd == '*':
            return { 'state': self.failed, 'authenStatus': 'fail',
                     'messages': [ failMsg ] }
         else:
            encrypted = ''
         traceX( TR_AUTHEN, "  encrypted:", encrypted, "in database:",
                 acct.encryptedPasswd )
         if encrypted == acct.encryptedPasswd:
            attrs = dict( roles = [ acct.role ] )
            return { 'state': self.succeeded, 'authenStatus': 'success',
                     'messages': [], 'user': self.user, 'authToken': self.password,
                     'sessionData' : attrs }
      return { 'state': self.failed, 'authenStatus': 'fail', 
               'messages': [ failMsg ], 'user': user, 'authToken': password }

   def policyDeniesLogin( self, acct ):
      # service is set to 'login' for console login, 'remote' for telnet, and
      # 'sshd' for ssh.
      #
      # Ideally, we don't need the "remoteServices" list, but right now not all
      # these services pass in remoteHost, so keep to be safe. But remote services
      # should all pass in remoteHost.
      remoteServices = ( 'sshd', 'remote', 'command-api', 'openconfig' )
      return ( not self.config.allowRemoteLoginWithEmptyPassword and
               acct.encryptedPasswd == '' and
               ( self.remoteHost or self.service in remoteServices ) )

class SshReactorBase:
   '''
      Provides the common functions to create the home directory
      and update he file ownership of the SSH files
   '''
   def __init__( self, user, agent ):
      self.user_ = user
      self.agent_ = agent
      if self.user_.name == 'root':
         self.sshDirPath_ = '/root/.ssh/'
      else:
         self.sshDirPath_ = f'/home/{self.user_.name}/.ssh/'
      self.sshKeyFilePath_ = self.sshDirPath_ + 'authorized_keys'
      self.sshPrincipalFilePath_ = self.sshDirPath_ + 'principals'

   def _adjustFileOwnership( self, filename ):
      pwent = self.agent_.getPwEnt( name=self.user_.name )
      assert pwent is not None
      os.chown( filename, pwent.userId, pwent.groupId )

   def _setUpLocalSshDir( self ):
      '''
         1. Ensure that the home directory exists, create if not
         2. Create the .ssh directory under home directory.
         3. Adjust file ownership.

         Note: 'root' user just needs the /root/.ssh to be created
      '''
      if self.user_.name == 'root':
         setUpSshDir( self.sshDirPath_ )
      else:
         self.agent_.ensureHomeDirExists( self.user_.name )
         setUpSshDir( self.sshDirPath_ )
         self._adjustFileOwnership( self.sshDirPath_ )

   def processSshOptions( self, sshOptions ):
      options = []
      for option in sorted( sshOptions ):
         sshValues = sshOptions[ option ].sshValues
         # Some options needs to be modified while writing to file
         # CLI syntax to OpenSSH format
         # permit-listen     => permitlisten
         # permit-open       => permitopen
         # no-x11-forwarding => no-X11-forwarding
         # x11-forwarding    => X11-forwarding
         if option == SshOptions.permitListen:
            printOption = "permitlisten"
         elif option == SshOptions.permitOpen:
            printOption = "permitopen"
         elif option == SshOptions.noX11Forwarding:
            printOption = "no-X11-forwarding"
         elif option == SshOptions.x11Forwarding:
            printOption = "X11-forwarding"
         else:
            printOption = option

         # Check if the option has a value associated with it
         # Some options don't have values like "agent-forwarding"
         if len( sshValues ) > 0:
            # Some options can have multiple values like
            # environment="A=B",environment="C=D"
            for value in sorted( sshValues ):
               options.append( f'{printOption}=\"{value}\"' )
         else:
            options.append( printOption )
      return options

   def handleSshAuthorizedKey( self ):
      '''
         Writes the SSH authorized keys with options to the authorized_keys file
         in self.sshKeyFilePath_ which can either be
         1. /home/<user>/.ssh/authorized_keys    --> Normal users
         2. /root/.ssh/authorized_keys           --> Root user
      '''
      authKeys = self.user_.sshAuthKeys

      keysStr = ""
      for key in authKeys:
         keyContents = authKeys[ key ].keyContents
         if keyContents:
            # Check if we have options associated with the key
            sshOptions = authKeys[ key ].sshOptions
            keyOptions = self.processSshOptions( sshOptions )
            if keyOptions:
               # Format of the keys is
               # option1,option2,....,option-n <key>
               keysStr += f'{",".join( keyOptions )} {keyContents}\n'
            else:
               keysStr += f'{keyContents}\n'

      # if nothing has to be written (empty key contents), delete the file
      if keysStr == "":
         # All keys have been removed
         removeFile( self.sshKeyFilePath_ )
         return

      # Setup up local SSH directory only if we have to write authorized_keys file
      self._setUpLocalSshDir()
 
      # Write the keys to file
      writeSshDirFile( self.sshKeyFilePath_, keysStr, "key file" )

      if self.user_.name != "root":
         self._adjustFileOwnership( self.sshKeyFilePath_ )

   def handleSshAuthorizedPrincipal( self ):
      '''
         Writes the SSH authorized principals with options to the principal file
         in self.sshPrincipalDirPath_ which can either be
         1. /home/<user>/.ssh/principal    --> Normal users
         2. /root/.ssh/principal           --> Root user
      '''
      authPrincipals = self.user_.sshAuthPrincipals

      principalsStr = ""
      for principal in sorted( authPrincipals ):
         # Check if we have options associated with the principal
         sshOptions = authPrincipals[ principal ].sshOptions
         principalOptions = self.processSshOptions( sshOptions )
         if principalOptions:
            # Format of the principals is
            # option1,option22,....,option-n <principal>
            principalsStr += f'{",".join( principalOptions )} {principal}\n'
         else:
            principalsStr += f'{principal}\n'

      # if nothing has to be written (empty principals), delete the file
      if principalsStr == "":
         # All principals have been removed
         removeFile( self.sshPrincipalFilePath_ )
         return

      # Setup up local SSH directory only if we have to write principal file
      self._setUpLocalSshDir()
 
      # Write the keys to file
      writeSshDirFile( self.sshPrincipalFilePath_, principalsStr, "principal file" )

      if self.user_.name != "root":
         self._adjustFileOwnership( self.sshPrincipalFilePath_ )

class SshUserKeyReactor( Tac.Notifiee, SshReactorBase ):
   '''
      Ssh key reactor
   '''
   notifierTypeName = "Mgmt::Ssh::SshAuthKeys"

   def __init__( self, key, user, agent ):
      Tac.Notifiee.__init__( self, key )
      SshReactorBase.__init__( self, user, agent )
      self.key_ = key
      # Generate the key file based on latest config for the user
      self.handleSshAuthorizedKey()

   @Tac.handler( 'keyContents' )
   def handleKeyContents( self ):
      self.handleSshAuthorizedKey()

   @Tac.handler( 'sshOptions' )
   def handleOptions( self, key ):
      '''
         This handles SSH options and its values
      '''
      self.handleSshAuthorizedKey()

   def close( self ):
      Tac.Notifiee.close( self )

class SshUserPrincipalReactor( Tac.Notifiee, SshReactorBase ):
   '''
      Ssh principal reactor
   '''
   notifierTypeName = "Mgmt::Ssh::SshAuthPrincipals"

   def __init__( self, principal, user, agent ):
      Tac.Notifiee.__init__( self, principal )
      SshReactorBase.__init__( self, user, agent )
      self.principal_ = principal
      # Generate the principal file based on latest config for the user
      self.handleSshAuthorizedPrincipal()

   @Tac.handler( 'sshOptions' )
   def handleOptions( self, principal ):
      '''
         This handles SSH options and its values
      '''
      self.handleSshAuthorizedPrincipal()

   def close( self ):
      Tac.Notifiee.close( self )

class LocalUserPlugin( AaaPluginLib.Plugin ): # pylint: disable=abstract-method
   def __init__( self, config, sshConfig, agent ):
      aaaConfig = agent.config
      aaaStatus = agent.status
      AaaPluginLib.Plugin.__init__( self, aaaConfig, "local" )
      self.aaaStatus = aaaStatus
      self.config = config
      self.agent = agent
      self.sshConfig = sshConfig

   def _userIsKnown( self, user ):
      return ( user in self.config.acct or user in self.sshConfig.user )

   def ready( self ):
      return True

   def logFallback( self ):
      # BUG 13784 requires that we return unavailable for unknown user,
      # but we don't log the typical FALLBACK message.
      return False

   # pylint: disable-next=redefined-builtin
   def createAuthenticator( self, method, type, service, remoteHost,
                            remoteUser, tty, user=None, privLevel=0 ):
      assert method == self.name
      if type == 'authnTypeLogin':
         a = LocalUserAuthenticator( self.config, self.aaaConfig, method, type,
                                     service, remoteHost, remoteUser, tty, user,
                                     privLevel)
      elif type == 'authnTypeEnable':
         a = EnableAuthenticator( self.config, self.aaaConfig, method, type,
                                  service, remoteHost, remoteUser, tty, user,
                                  privLevel )
      else:
         bt( TR_ERROR, "unknown authentication type:", bv( type ) )
         a = None
      return a

   def openSession( self, authenticator ):
      traceX( TR_SESSION, "openSession for user", authenticator.user )
      return authenticator

   def closeSession( self, token ):
      traceX( TR_SESSION, "closeSession for user", token.user )

   def authorizeShell( self, method, user, session ):
      traceX( TR_AUTHZ, "authorizeShell for method", method, "user", user )
      assert method == self.name

      # My policy is to authorize shells for users I know about
      attrs = {}
      acct = self.config.acct.get( user )
      if acct is not None:
         status = 'allowed'
         message = ''
         attrs[ AaaPluginLib.privilegeLevel ] = acct.privilegeLevel
         attrs[ AaaPluginLib.roles ] = [ acct.role ]
      elif self.sshConfig.user.get( user ) is not None:
         status = 'allowed'
         message = ''
      else:
         status = 'authzUnavailable'
         message = "Unknown user"
      return ( status, message, attrs )

   def _ruleMatchByMode( self, rule, mode ):
      modeName, modeKey, longModeKey = mode
      # A rule without a modeKey is applied to all the Cli modes
      if not rule.modeKey:
         return True
      elif modeName == 'Exec':
         return rule.modeKey == 'exec'
      elif rule.modeKey == 'config-all':
         return True
      elif modeName == 'Configure':
         return rule.modeKey == 'config'
      elif rule.modeKey == modeKey:
         return True
      elif rule.modeKeyRe.match( longModeKey ):
         return True

      # Not applicable
      return False

   def authorizeShellCommand( self, method, user, session, mode, privlevel, tokens ):
      traceX( TR_AUTHZ, "authorizeShellCommand for method", method, "user", user,
              "mode", mode, "privlevel", privlevel, "tokens", tokens )
      assert method == self.name

      role = None
      if session:
         # Deny the established session of a deleted local user
         if session.authenMethod == 'local' and not self._userIsKnown( user ):
            return ( "authzUnavailable", "Unknown user", {} )

         traceX( TR_DEBUG, 'user', user, 'sessionId', session.id )
         sessionData = session.property.get( session.authenMethod )
         if sessionData:
            traceX( TR_DEBUG, 'sessionData', sessionData )
            # We may support multiple roles per user in the future
            attr = sessionData.attr.get( 'roles' )
            if attr:
               role = eval( attr )[ 0 ] # pylint: disable=eval-used
         if role is None:
            return ( "authzUnavailable", "Unknown role", {} )
      else:
         return ( "authzUnavailable", "Unknown user", {} )

      # Get the effective role
      if not role or role not in self.config.role:
         role = self.config.defaultRole
         traceX( TR_DEBUG, 'user', user, 'fall back to role', role )

      compiledRole = compiledRoles.get( role )
      if compiledRole is None:
         return ( "denied", "Unknown role", {} )

      # Match the command and mode against rules
      for seq in compiledRole.seqs:
         try:
            rule = compiledRole.rules[ seq ]
         except KeyError:
            # this may be due to modification from another thread - ignore it
            continue
         if not self._ruleMatchByMode( rule, mode ):
            continue
         match = rule.regex.match( ' '.join( tokens ) )
         if match:
            if rule.permit:
               return ( "allowed", '', {} )
            else:
               return ( "denied", '', {} )

      return ( "denied", '', {} )

   def hasUserShell( self ):
      return True

   def getUserShell( self, name ): # pylint: disable=arguments-renamed
      acct = self.config.acct.get( name )
      if acct and acct.shell:
         return acct.shell
      else:
         return None

   def writeUserInfo( self, f ):
      synchronizer = self.agent.userInfoSynchronizer
      for user, acct in self.config.acct.items():
         if user not in self.aaaStatus.account:
            uid = self.agent.generateUid( user )
            synchronizer.writeLine(
               f,
               user,
               uid,
               synchronizer.gid if synchronizer.gid is not None else uid,
               "Eos-%s (local)" % user, # pylint: disable=consider-using-f-string
               user,
               acct.shell or self.aaaConfig.shell )

class UserConfigReactor( Tac.Notifiee, SshReactorBase ):
   notifierTypeName = "Mgmt::Users::UserConfig"

   def __init__( self, user, agent ):
      self.user_ = user
      self.agent_ = agent
      Tac.Notifiee.__init__( self, user )
      SshReactorBase.__init__( self, user, agent )
      self.sshUserKeyReactors_ = {}
      self.sshUserPrincipalReactors_ = {}

      if self.user_.name != 'root':
         # Create an entry in status, this is required
         # for setting up the home directory and permissions
         with self.agent_.mutex:
            userName = user.name
            if userName not in self.agent_.status.account:
               self.agent_.setUpNewUser( userName, "local" )

      # This will ensure we start with a clean slate for the user,
      # deleting the authorized_keys file if there is no config
      self.handleSshAuthorizedKey()

      # Ensure we process principals before UserConfigReactor is created
      self.handleSshAuthorizedPrincipal()

      # For each key created before SshUserAccountReactor is created
      # create SshUserKeyReactor reactor
      for key in self.notifier_.sshAuthKeys:
         self.handleSshKey( key )

      # For each principal created before SshUserAccountReactor is created
      # create SshUserPrincipalReactor reactor
      for principal in self.notifier_.sshAuthPrincipals:
         self.handleSshPrincipal( principal )

   @Tac.handler( 'sshAuthKeys' )
   def handleSshKey( self, key ):
      self.handleSshAuthorizedKey()
      Tac.handleCollectionChange( SshUserKeyReactor, key, 
                                  self.sshUserKeyReactors_, 
                                  self.user_.sshAuthKeys, 
                                  reactorArgs=(self.user_, self.agent_,) )

   @Tac.handler( 'sshAuthPrincipals' )
   def handleSshPrincipal( self, principal ):
      self.handleSshAuthorizedPrincipal()
      Tac.handleCollectionChange( SshUserPrincipalReactor, principal, 
                                  self.sshUserPrincipalReactors_, 
                                  self.user_.sshAuthPrincipals, 
                                  reactorArgs=(self.user_, self.agent_,) )

   @Tac.handler( 'shell' )
   def handleShell( self ):
      self.agent_.userInfoSynchronizer.scheduleSync()

   def close( self ):
      # User deleted, cleanup the ssh files
      removeFile( self.sshKeyFilePath_ )
      removeFile( self.sshPrincipalFilePath_ )
      with self.agent_.mutex:
         self.agent_.cleanUpUser( self.user_.name )
      Tac.Notifiee.close( self )

class MergedUsersConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Users::AllUsers"

   def __init__( self, cfg, agent ):
      self.cfg_ = cfg
      self.agent_ = agent
      self.mergedUsersReactors_ = {}
      Tac.Notifiee.__init__( self, cfg )
      # Process users in the collection before the MergedUsersConfigReactor
      # is created
      for user in self.notifier_.users:
         self.handleUser( user )

   @Tac.handler( 'users' )
   def handleUser( self, user ):
      Tac.handleCollectionChange( UserConfigReactor, user,
            self.mergedUsersReactors_,
            self.cfg_.users,
            reactorArgs=(self.agent_,) )

def Plugin( ctx ):
   mountGroup = ctx.entityManager.mountGroup()
   config = mountGroup.mount( 'security/aaa/local/config', 'LocalUser::Config', 'r' )
   sshConfig = mountGroup.mount( 'mgmt/ssh/config', 'Mgmt::Ssh::Config', 'r' )
   def _finish():
      global localUserConfigReactor_, mergedUsersConfigReactor_
      localUserConfigReactor_ = LocalUserConfigReactor( config, ctx.aaaAgent )
      mergedUsers = Tac.newInstance( "Mgmt::Users::AllUsers" )
      ctx.aaaAgent.userService = Tac.newInstance( 'Mgmt::Users::UserMergeSm',
            sshConfig, config, mergedUsers )
      mergedUsersConfigReactor_ = MergedUsersConfigReactor( mergedUsers,
            ctx.aaaAgent )
   mountGroup.close( _finish )
   return LocalUserPlugin( config, sshConfig, ctx.aaaAgent )
