# Copyright (c) 2008-2010, 2013-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import AaaPluginLib
from AaaPluginLib import TR_ERROR, TR_WARN, TR_AUTHEN, TR_AUTHZ, \
      TR_ACCT, TR_INFO, hostProtocol
from BothTrace import traceX as bt
from BothTrace import Var as bv
import Cell
import Logging
import Radius
from Radius import radiusNasIdType, radsecproxyInternalPort
import Tac
from Tracing import traceX

import freeradiusclient as rc
import re

AAA_INVALID_RADIUS_SERVER = Logging.LogHandle(
              "AAA_INVALID_RADIUS_SERVER",
              severity=Logging.logWarning,
              fmt="RADIUS server '%s' port %d could not be used: %s",
              explanation="The configuration contains a RADIUS server that "
                          "could not be used.  One cause for this type of "
                          "error is a hostname for which DNS resolution "
                          "fails.",
              recommendedAction="Correct the RADIUS server configuration." )

AAA_UNKNOWN_RADIUS_SERVER = Logging.LogHandle(
              "AAA_UNKNOWN_RADIUS_SERVER",
              severity=Logging.logWarning,
              fmt="RADIUS server group '%s' references unknown server '%s'",
              explanation="The RADIUS server group contains a server that "
                          "was not configured using the 'radius-server host' "
                          "command.",
              recommendedAction="Configure the unknown server or remove it "
                                "from the server group." )

AAA_NO_VALID_RADIUS_SERVERS = Logging.LogHandle(
              "AAA_NO_VALID_RADIUS_SERVERS",
              severity=Logging.logError,
              fmt="No valid RADIUS servers for method list '%s'",
              explanation="The configuration contains an authentication "
                          "method list that is not associated with any valid "
                          "RADIUS servers. One common cause for this error is "
                          "a hostname for which DNS resolution fails.",
              recommendedAction="Correct the RADIUS server configuration." )

AAA_INVALID_RADIUS_RULE = Logging.LogHandle(
              "AAA_INVALID_RADIUS_RULE",
              severity=Logging.logError,
              fmt="Invalid rule '%s' for RADIUS user '%s'",
              explanation="The configuration file on RADIUS server "
                          "contains invalid rules.",
              recommendedAction="Correct the invalid rules on RADIUS server." )

# ConfigReactor (below) listens for changes to Radius::Config and clears
# _sesionPool whenever the config changes to avoid the situation where a
# pooled session with outdated configuration data could be used.
_sessionPool = AaaPluginLib.SessionPool( 'radius' )
validRuleRegex = re.compile( "(permit|deny).*command.*" )

def _invalidateSessionPool( hostgroup=None ):
   bt( TR_WARN, "invalidate RADIUS session pool" )
   _sessionPool.clear( hostgroup )

class CompiledRule:
   def __init__( self, rule ):
      if re.match( validRuleRegex, rule ):
         self.error = False
         ruleAction = rule.split( ' ' )[ 0 ]
         if "mode" in rule:
            self.modeKey = ( rule.split( 'mode ' ) )[ 1 ].split( ' command' )[ 0 ]
         else:
            self.modeKey = ""
         ruleRegex = rule.split( 'command ' )[ 1 ]
         self.permit = ( ruleAction == 'permit' )
         self.modeKeyRe = re.compile( self.modeKey )
         self.regex = re.compile( ruleRegex )
         self.ruleStr = rule
      else:
         self.error = True
         self.permit = True
         self.modeKeyRe = re.compile( "" )
         self.regex = re.compile( "" )
         self.ruleStr = rule
      
def _getRoleIndex( rules ):
   return str( hash( str( rules ) ) )

# dictionary of roleIndex -> a list of CompiledRules
compiledRoles = {}

class Authenticator( AaaPluginLib.BasicUserAuthenticator ):
   # pylint: disable-next=redefined-builtin
   def __init__( self, plugin, method, type, service, remoteHost,
                 remoteUser, tty, user, privLevel, radiusConfig, netStatus ):
      self.plugin = plugin
      AaaPluginLib.BasicUserAuthenticator.__init__( self, plugin.aaaConfig, 
                                                    method, type, service,
                                                    remoteHost, remoteUser, tty,
                                                    user, privLevel )
      def _getNasId():
         nasType = radiusConfig.nasIdType
         nasId = ""
         if nasType != radiusNasIdType.disabled:
            if nasType == radiusNasIdType.hostname:
               nasId = netStatus.hostname
            elif nasType == radiusNasIdType.fqdn:
               nasId = netStatus.fqdn
            else:
               nasId = radiusConfig.nasId
            if nasId and nasId == "localhost":
               nasId = ""
         return nasId

      self.nasId = _getNasId()

   def checkPassword( self, user, password ):
      failType = 'unavailable'
      failText = ''
      # actually do the authentication
      session = self.plugin.acquireSession( self.method )
      if session: # pylint: disable=too-many-nested-blocks
         try:
            if self.type == 'authnTypeEnable':
               # Cisco uses this special user name for privilege mode
               # authentication
               # pylint: disable-next=consider-using-f-string
               username = "$enab%s$" % ( self.privLevel )
            else:
               username = self.user

            bt( TR_AUTHEN, "authenticating user", bv( username ) )
            authenState = session.sendAuthenReq( username, password, self.privLevel,
                                                 self.remoteHost, self.tty,
                                                 self.nasId )
            failText = authenState.failText
            if authenState.status == rc.OK_RC:
               if failText:
                  msg = [ Tac.Value( "AaaApi::AuthenMessage",
                                     style='info', text=failText ) ]
               else:
                  msg = []
               self.plugin.releaseSession( session )
               r = { 'state': self.succeeded,
                     'authenStatus': 'success',
                      'messages': msg,
                      'user': self.user,
                      'authToken': self.password }
               if self.type != 'authnTypeEnable':
                  attrs = {}
                  if authenState.privLevel is not None:
                     # server has no privilege level configuration
                     # set it to '' so authorizeShell() could return 'allow'
                     bt( TR_AUTHEN, 'privilege:', bv( authenState.privLevel ) )
                     attrs[ 'priv-lvl' ] = authenState.privLevel
                  if authenState.roles:
                     bt( TR_AUTHEN, 'roles:', bv( authenState.roles ) )
                     attrs[ 'roles' ] = authenState.roles
                  attrs[ 'roleIndex' ] = None
                  if authenState.rules:
                     roleIndex = _getRoleIndex( authenState.rules )
                     attrs[ 'roleIndex' ]  = roleIndex
                     ruleList = []
                     for ruleStr in authenState.rules:
                        cRule = CompiledRule( ruleStr )
                        if cRule.error:
                           Logging.log( AAA_INVALID_RADIUS_RULE,
                                        cRule.ruleStr, username )
                        else:
                           ruleList.append( CompiledRule( ruleStr ) )
                           # this might be too much, put in INFO
                           bt( TR_INFO, 'rule:', bv( ruleStr ) )
                     compiledRoles[ roleIndex ] = ruleList
                  if authenState.classAttr:
                     attrs[ 'classAttr' ] = authenState.classAttr
                  r[ 'sessionData' ] = attrs
               return r
            elif authenState.status == rc.REJECT_RC:
               # only print this error if no message comes from the server
               # (IOS behavior)
               failType = 'fail'
               session.close()
         except Exception as e: # pylint: disable=broad-except
            bt( TR_ERROR, "sendAuthenReq exception", bv( str( e ) ) )
            session.close()
      else:
         failText = 'Error in adding server (DNS issue?)'

      if not failText:
         failText = 'Error in authentication'
      msg = [ Tac.Value( "AaaApi::AuthenMessage", style='error', text=failText ) ]
      return { 'state': self.failed, 'authenStatus': failType, 'messages': msg,
               'user': user, 'authToken': password }

class RadiusPlugin( AaaPluginLib.Plugin ):
   def __init__( self, config, status, aaaConfig, ipStatus, ip6Status, netStatus,
                 allVrfStatusLocal ):
      AaaPluginLib.Plugin.__init__( self, aaaConfig, "radius",
                                    allVrfStatusLocal=allVrfStatusLocal )
      self.config = config
      self.status = status
      self.ipStatus = ipStatus
      self.ip6Status = ip6Status
      self.netStatus = netStatus
      self.allVrfStatusLocal = allVrfStatusLocal

   def ready( self ):
      ready = True
      if len( self.config.host ) == 0:
         bt( TR_ERROR, "Radius not ready: no servers configured" )
         ready = False
      return ready

   def handlesAuthenMethod( self, method ):
      return self._handlesGroupAuthenMethod( method, "radius", "radius" )
 
   # pylint: disable-next=redefined-builtin
   def createAuthenticator( self, method, type, service, remoteHost, remoteUser,
                            tty, user=None, privLevel=0 ):
      return Authenticator( self, method, type, service, remoteHost, remoteUser,
                            tty, user, privLevel, self.config, self.netStatus )

   def openSession( self, authenticator ):
      return authenticator

   def closeSession( self, token ):
      pass

   def _configureReqFromSession( self, req, session ):
      req.authenMethodIs( rc.PW_RADIUS )
      if session is None:
         return
      if session.authenMethod is None or session.authenMethod == "local":
         req.authenMethodIs( rc.PW_LOCAL )
      elif session.authenMethod != "radius":
         req.authenMethodIs( rc.PW_REMOTE )
      # Send Class attr back in accounting request
      classAttr = None
      sessionData = session.property.get( session.authenMethod )
      if sessionData:
         classAttr = sessionData.attrByte.get( "classAttr" )
      if classAttr:
         traceX( TR_ACCT, "Adding Class attribute to acct req" )
         # pylint: disable-next=protected-access
         req._addVar( rc.PW_CLASS, "Class", classAttr )
      if session.remoteHost:
         req.remoteHostIs( session.remoteHost )
      if session.tty:
         req.portIs( session.tty )

   def authorizeShell( self, method, user, session ):
      attrs = { }
      if session is None or session.property.get( method ) is None:
         # We do not have user privilege data, most likely the authentication
         # wasn't handled by RADIUS
         return ( 'authzUnavailable',
                  'RADIUS authorization requires RADIUS authentication', {} )
      else:
         if session.property.get( method ).attr.get( 'priv-lvl' ):
            attrs[ AaaPluginLib.privilegeLevel ] = session.privilegeLevel

      # We do not extract the "roles" attribute and pass it to RunCli. It is
      # kept in "sessionData". In any Aaa plugin, we can look up per-session
      # attributes from the "session" argument.
      return  ( 'allowed', '', attrs )
   
   def _ruleMatchByMode( self, rule, mode ):
      # Compare the current mode and the mode keyword in rule
      # Permit if they are same. Otherwise deny the rule
      modeName, modeKey, longModeKey = mode
      # A rule without a modeKey is applied to all the Cli modes
      if not rule.modeKey:
         return True
      # "Exec" and "Configure" only have modeName, no modeKey
      elif modeName == 'Exec':
         return rule.modeKey == 'exec'
      elif modeName == 'Configure':
         return rule.modeKey == 'config'
      # "config-all" is applied to all the Cli modes
      elif rule.modeKey == 'config-all':
         return True
      elif rule.modeKey == modeKey:
         return True
      elif rule.modeKeyRe.match( longModeKey ):
         return True

      # Not applicable
      return False

   def authorizeShellCommand( self, method, user, session, mode, privlevel, tokens ):
      role = None
      if session is None or session.property.get( method ) is None:
         # We do not have user privilege data, most likely the authentication 
         # wasn't handled by RADIUS
         return ( "authzUnavailable",
                  "RADIUS authorization requires RADIUS authentication", {} )
      else:
         sessionData = session.property.get( session.authenMethod )
         if sessionData:
            role = sessionData.attr.get( 'roleIndex' )
         if role is None:
            return ( "denied", "Unknown rule", {} )

      # Get the effective role
      rules = compiledRoles.get( role )
      if not rules:
         return ( "denied", "Unknown rule", {} )
      for i, rule in enumerate( rules ):
         if not self._ruleMatchByMode( rule, mode ):
            traceX( TR_AUTHZ, "rule", i, "mode mismatch", mode )
            continue
         match = rule.regex.match( ' '.join( tokens ) )
         if match:
            if rule.permit:
               bt( TR_AUTHZ, "rule", bv( i ), "matched" )
               return ( "allowed", '', {} )
            else:
               bt( TR_ERROR, "rule", bv( i ), "matched" )
               return ( "denied", '', {} )

      return ( "denied", '', {} )

   def invalidateSessionPool( self, hostgroup ):
      _invalidateSessionPool( hostgroup )

   def _doAcct( self, method, createReqFunc ):
      """ Helper function for sendShellAcct and sendCommandAcct. """
      r = 'acctError'
      radiusSession = self.acquireSession( method )
      if not radiusSession:
         return ( r, 'Misconfiguration prevents accounting' )      
      acctReq = createReqFunc( radiusSession )
      try:
         status = radiusSession.sendAcctReq( acctReq )
         if status == rc.OK_RC:
            r = 'acctSuccess'
            msg = "Account request succeeded"
         else:
            r = 'acctFail'
            msg = 'Acct request failed'
      except Radius.AccountingError as e:
         msg = str( e )
         bt( TR_ERROR, "RADIUS accounting error", bv( msg ) )
      self.releaseSession( radiusSession )
      return ( r, msg )

   def sendShellAcct( self, method, user, session, action, startTime, 
                      elapsedTime=None ):
      traceX( TR_ACCT, "sendShellAcct method", method, "session", session,
              "action", action, "startTime", startTime, "elapsedTime",
              elapsedTime or 0 )
      def _createShellAcctReq( radiusSession ):
         acctAction = rc.PW_STATUS_START
         taskId = None
         if action == 'stop':
            acctAction = rc.PW_STATUS_STOP
            assert session
            if session.execAcctTaskId:
               taskId = session.execAcctTaskId
         req = radiusSession.createAcctReq( user, acctStatusType=acctAction,
               acctSessionId=taskId )
         self._configureReqFromSession( req, session )
         # store the exec acct taskid needed for the "stop" action
         if action == 'start':
            assert session
            session.execAcctTaskId = req.sessionId()
         elif action == 'stop':
            req.elapsedTimeIs( elapsedTime )
         return req

      return self._doAcct( method, _createShellAcctReq )
 
   def sendCommandAcct( self, method, user, session, privlevel, timestamp, tokens,
                        cmdType=None, **kwargs ):
      traceX( TR_ACCT, "sendCommandAcct for method", method, "session", session,
              "user", user, "privlevel", privlevel, "timestamp", timestamp )
      def _createCommandAcctReq( radiusSession ):
         acctReq = radiusSession.createAcctReq( user, privlevel )
         self._configureReqFromSession( acctReq, session )
         acctReq.commandIs( tokens ) 
         return acctReq
      return self._doAcct( method, _createCommandAcctReq )

   def sendSystemAcct( self, method, event, startTime, reason, action ):
      traceX( TR_ACCT, "sendSystemAcct method", method, "event", event, "action",
              action, "startTime", startTime, "reason", reason )
      user = 'unknown'
      session = None
      def _createSystemAcctReq( radiusSession ):
         acctAction = rc.PW_STATUS_START
         taskId = None
         if action == 'stop':
            acctAction = rc.PW_STATUS_STOP
         req = radiusSession.createAcctReq( user, acctStatusType=acctAction,
               acctSessionId=taskId )
         # pylint: disable-next=protected-access
         req._addVar( rc.PW_SERVICE_TYPE, "SERVICE_TYPE", 'system' )
         #reason is not yet set
         self._configureReqFromSession( req, session )
         return req

      return self._doAcct( method, _createSystemAcctReq )     


   def acquireSession( self, methodName ):
      hg, serverSpecs = AaaPluginLib.serversForMethodName( 
         methodName, self.aaaConfig, 'radius' )
      if hg is not None:
         session = _sessionPool.get( hg )
         if session is not None:
            return session
      radiusConfig = self.config

      session = Radius.Session( hg,
                                timeout=radiusConfig.timeout,
                                retries=radiusConfig.retries,
                                deadtime=radiusConfig.deadtime )

      addedServers = 0
      # If no aaa group servers are found, then follow the order of radius
      # server configuration. Otherwise, honor the aaa group server config
      if len( serverSpecs ) == 0:
         # If the server dict is empty I consider all configured servers to be
         # fair game.  Either methodName indicates that all servers should be
         # included, or it's a server group with no members, but I expect the
         # former because I shouldn't get to this point without any servers
         # since handlesAuthenMethod should have screened the request out.
         hostList = sorted( radiusConfig.host.values(),
                            key=lambda host: host.index )
      else:
         hostList = []
         for gspec in serverSpecs:
            # first look for a fully matched host, i.e., matching the hostspec
            # (hostname, authport, vrf, acctport)
            try:
               host = radiusConfig.host[ gspec ]
               hostList.append( host )
            except KeyError:
               # look for a partially matched host, i.e., matching (hostname,
               # authport, vrf). Ideally, we should not have a host with
               # different acctports. But it might happen due to a mis-config,
               # e.g., user did not configure the acctport in aaa server group,
               # causing the default acctport (different from what configured 
               # in radius-server host) to be used. To handle such scenario,
               # ignore acctport during the search
               for spec, host in radiusConfig.host.items():
                  if ( gspec.hostname, gspec.port, gspec.vrf ) == (
                     spec.hostname, spec.port, spec.vrf ):
                     hostList.append( host )
                     break
               else:
                  # Server group contains members that were not in the 
                  # host collection
                  groupname = AaaPluginLib.extractGroupFromMethod( methodName )
                  # pylint: disable-next=consider-using-f-string
                  serverStr = "%s:%d" % ( gspec.hostname, gspec.port )
                  Logging.log( AAA_UNKNOWN_RADIUS_SERVER, groupname, serverStr )

      for h in hostList: # pylint: disable=too-many-nested-blocks
         assert h.vrf
         ns = self.getNsFromVrf( h.vrf )
         if not ns:
            bt( TR_ERROR, "cannot get namespace for vrf", bv( h.vrf ) )
            continue

         # determine the source ip to be used in this ns
         srcIpAddr, srcIp6Addr = None, None
         if radiusConfig.srcIntfName:
            srcIntf = radiusConfig.srcIntfName.get( h.vrf )
            if srcIntf:
               srcIpIntfStatus = self.ipStatus.ipIntfStatus.get( srcIntf )
               if ( srcIpIntfStatus and
                    srcIpIntfStatus.activeAddrWithMask.address != "0.0.0.0" ):
                  srcIpAddr = srcIpIntfStatus.activeAddrWithMask.address

               srcIp6IntfStatus = self.ip6Status.intf.get( srcIntf )
               if srcIp6IntfStatus:
                  for ip6addr in srcIp6IntfStatus.addr:
                     # pylint: disable-next=no-else-continue
                     if ip6addr.address.isLinkLocal:
                        continue
                     else:
                        srcIp6Addr = ip6addr.address.stringValue
                        break

         # pylint: disable-next=consider-using-ternary
         key = h.useKey and h.key or radiusConfig.key
         clearTextKey = key.getClearText()
         # pylint: disable-next=consider-using-ternary
         timeout = h.useTimeout and h.timeout or radiusConfig.timeout
         if h.useRetries:
            retries = h.retries
         else:
            retries = radiusConfig.retries
         hs = AaaPluginLib.HostSpec( h.hostname, h.port, h.acctPort, h.vrf,
                                     h.protocol )
         cb = Radius.CounterCallback( self.status, hs )

         bt( TR_AUTHEN, 'add server', bv( h.stringValue() ),
             "srcIpAddr", bv( srcIpAddr or "" ),
             "srcIp6Addr", bv( srcIp6Addr or "" ) )
         try:
            # tlsPort used here is the port used between Aaa and radsecproxy
            # running on the same switch for UDP communication.
            # For now we will be using 1812 as the fixed port for this
            # UDP communitcation.
            # tlsPort should be set to 0 if radsecproxy is not used and Aaa
            # communicates to Radius server directly over UDP.
            #
            # per radius-server key config is not allowed if server is tls enabled,
            # but global key is still configurable and we should not use key
            # for communication between AaaPlugin and radsecproxy(tls) as both of
            # them are running on same box. Since key is mandatory for UDP
            # communication between AaaPlugin and radsecproxy, we are hardcoding it
            # to "arastra". Anyone changing this hardcoded secret, please make
            # sure you change the same in RadsecProxy config generator function
            # which is present in /src/Radsec/RadsecLib.py
            tlsPortToBeUsed = 0
            if h.protocol == hostProtocol.protoRadsec:
               tlsPortToBeUsed = radsecproxyInternalPort 
               clearTextKey = "arastra"
            session.addServer( h.hostname, h.port, h.acctPort, tlsPortToBeUsed,
                               clearTextKey, timeout, retries, 
                               ns, srcIpAddr, srcIp6Addr, counterCallback=cb )
            addedServers += 1
         except Radius.BadServerError as e:
            bt( TR_ERROR, "failed to add server", bv( h.stringValue() ),
                ":", bv( str( e ) ) )
            Logging.log( AAA_INVALID_RADIUS_SERVER, h.hostname, h.port, str( e ) )

      if addedServers == 0:
         # No servers indicates is a configuration error
         Logging.log( AAA_NO_VALID_RADIUS_SERVERS, methodName )
         session.close()
         session = None
      return session

   def releaseSession( self, session ):
      if session is not None:
         _sessionPool.put( session )
         # don't need to do anything else

   def hasUnknownUser( self ):
      return True

class ConfigReactor( AaaPluginLib.ConfigReactor ):
   notifierTypeName = "Radius::Config"
   counterTypeName = "Radius::Counters"

   def __init__( self, notifier, status, allVrfStatusLocal, ipStatus, ip6Status ):
      AaaPluginLib.ConfigReactor.__init__( self, notifier,
                                           status, allVrfStatusLocal,
                                           ipStatus, ip6Status )

   def invalidateSessionPool( self ):
      _invalidateSessionPool()

   @Tac.handler( 'host' )
   def handleHost( self, hostspec=None ):
      self.handleHostEntry( hostspec )

   @Tac.handler( 'deadtime' )
   def handleDeadtime( self ):
      self.invalidateSessionPool()

   def handleVrfState( self, vrfName ):
      # Theoretically we could use the vrfName to figure out if we
      # actually care about this change or not, but for simplicity we
      # just hit it with a big hammer and throw out any persistent
      # singleConn connections any time any VRF changes.  Since this
      # is all driven by config change events rather than protocol or
      # packet events, this isn't really a big deal. (copied from Tacacs)
      _invalidateSessionPool()

class CounterConfigReactor( Tac.Notifiee ):
   notifierTypeName = "AaaPlugin::CounterConfig"

   def __init__( self, notifier, inputDir, status ):
      Tac.Notifiee.__init__( self, notifier )
      self.inputDir_ = inputDir
      self.status_ = status

   def setCheckpoint( self ):
      self.status_.checkpoint.clear()
      for h in self.status_.counter:
         self.status_.checkpoint[ h ] = Radius.getHostCounters( h,
                                                                self.status_,
                                                                self.inputDir_ )
      self.status_.lastClearTime = Tac.now()

   @Tac.handler( 'clearCounterRequestTime' )
   def handleClearCounterRequestTime( self ):
      """Handle clear counter request from the Cli"""
      self.setCheckpoint()

_reactors = {}

def Plugin( ctx ):
   mountGroup = ctx.entityManager.mountGroup()
   config = mountGroup.mount( 'security/aaa/radius/config', 'Radius::Config',
                              'r' )
   counterConfig = mountGroup.mount( 'security/aaa/radius/counterConfig',
                                     'AaaPlugin::CounterConfig', 'r' )
   inputDir = mountGroup.mount( Cell.path( 'security/aaa/radius/input/status' ),
                                'Tac::Dir', 'ri' )
   status = mountGroup.mount( Cell.path( 'security/aaa/radius/status' ),
                              'Radius::Status', 'wf' )
   aaaConfig = ctx.aaaAgent.config
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mountGroup.cMg_, True,
                                                          True )
   ipStatus = mountGroup.mount( 'ip/status', 'Ip::Status', 'r' )
   ip6Status = mountGroup.mount( 'ip6/status', 'Ip6::Status', 'r' )
   allVrfStatusLocal = mountGroup.mount( Cell.path( 'ip/vrf/status/local' ),
                                         'Ip::AllVrfStatusLocal', 'r' )
   netStatus = mountGroup.mount( Cell.path( 'sys/net/status' ),
                                            'System::NetStatus', 'r' )
   def _finish():
      _reactors[ "RadiusConfigReactor" ] = ConfigReactor( config,
                                                          status,
                                                          allVrfStatusLocal,
                                                          ipStatus, ip6Status )
      _reactors[ "RadiusCounterReactor" ] = CounterConfigReactor( counterConfig,
                                                                  inputDir,
                                                                  status )
   mountGroup.close( _finish )
   return RadiusPlugin( config, status, aaaConfig, ipStatus, ip6Status, netStatus,
                        allVrfStatusLocal )
