# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import six
from six.moves import range

from AufVersion import Version

ABOOT_HAS_SB_CPU_WP = {
   # For Aboot7, the protection was added with @11693733 and removed with @21336769
   7: range( 11693733, 21336769 ),
   # For Aboot9, the protection was added with @10573215 and removed with @21161482
   9: range( 10573215, 21161482 ),
}

ABOOT_PCR_LOG_PERMISSION_BUG = range( 24959799, 25735798 )

ABOOT_ROLLBACK_SUPPORTED = {
   6  : 33096050,
   11 : 31655772,
   12 : 29756282, # official Aboot12.0.0 release
   13 : 30592336, # official Aboot13.0.1 release
   16 : 36922739 # temporary to be updated with Aboot16 release
}
ABOOT_FACTORY_RESET_SUPPORTED = {
   12 : 29756282, # official Aboot12.0.0 release
   13 : 30592336  # official Aboot13.0.1 release
}
# describes when protection was added to abort rollback when no recovery files are
# found under /mnt/flash/.user_recovery
ABOOT_ABORT_ROLLBACK = 33118043

class AbootVersion( Version ):
   norcalRe = r'Aboot-(norcal(\d+)?)'
   pcieLaneConfigRe = r'-(pcie)?(\d+x\d+(-\d+x\d+)?)'
   coreRe = r'-(\d+)core'
   dimmRe = r'-((u|so)dimm)'
   changeNumRe = r'-(\d{6,})'
   engRe = r'-eng(%s|-)?$' % changeNumRe # pylint: disable=consider-using-f-string
   extraInfoRe = r'Aboot-norcal\d+-\d+\.\d+\.\d+-(\S+)-\d+'

   def __init__( self, versionString ):
      # pylint: disable-next=super-with-arguments
      super( AbootVersion, self ).__init__( versionString )

      self.norcal = None
      self.norcalId = None
      self.pcieLaneConfig = None
      self.core = None
      self.dimm = None
      self.changeNumber = None
      self.extraInfo = None
      self.eng = False

      versionString = six.ensure_str( versionString )
      m = re.match( self.norcalRe, versionString )
      if m:
         self.norcal = m.group( 1 )
         if m.group( 2 ):
            self.norcalId = int( m.group( 2 ) )

      m = re.search( self.pcieLaneConfigRe, versionString )
      if m:
         self.pcieLaneConfig = m.group( 2 )

      m = re.search( self.coreRe, versionString )
      if m:
         self.core = int( m.group( 1 ) )

      m = re.search( self.changeNumRe, versionString )
      if m:
         self.changeNumber = m.group( 1 )

      m = re.search( self.dimmRe, versionString )
      if m:
         self.dimm = m.group( 1 )

      m = re.search( self.extraInfoRe, versionString )
      if m:
         self.extraInfo = m.group( 1 )

      m = re.search( self.engRe, versionString, flags=re.IGNORECASE )
      self.eng = bool( m )

   @staticmethod
   def abootRe():
      # Slightly more restrictive than abootLooseRe, at least trying to match
      # an actual "x.y.z" version number.
      # The "non-/ or start" starting part helps with discarding build paths, which
      # might contain the version of the Aboot spec file (ie "../Aboot-1.0.18/...")

      # "(?:" --> non-matching group
      # "(?<!/" --> negative lookbehind assertion
      # Don't start matching after /
      # So together: start matching if "Aboot" isn't preceded by "/"
      slashPrefixSkipRe = r'(?:(?<!/)|^)'
      # ie: "norcal7"
      versionPrefixRe = r'([0-9a-zA-Z\-]+-)?'
      versionNumRe = r'([0-9]+\.[0-9]+\.[0-9]+)'
      # ie: changenum, "ENG"...
      versionSuffixRe = r'(-[0-9a-zA-Z\-]+)?'

      return r'(' + slashPrefixSkipRe + 'Aboot-' + versionPrefixRe + versionNumRe + \
             versionSuffixRe + ')'

   @staticmethod
   def abootLooseRe():
      # Very loosely match Aboot version for seaching large flash dumps,
      # potentially without actual version.
      return r'(Aboot-[0-9a-zA-Z\-\.]*-)((eng|ENG(-\d+)?)|\d+)'

   @staticmethod
   def matchProduct( versionString ):
      m = re.match( AbootVersion.productRe, versionString )
      if m:
         return m.group( 1 ) == 'Aboot'
      return False

   def __str__( self ):
      # pylint: disable-next=consider-using-f-string
      pcie = 'pcie%s' % ( self.pcieLaneConfig ) if self.pcieLaneConfig else None
      # pylint: disable-next=consider-using-f-string
      core = '%dcore' % ( self.core ) if self.core else None
      eng = "ENG" if self.eng else None
      return '-'.join( [ _f for _f in [ 'Aboot', self.norcal, self.version, pcie,
                                        core, self.dimm, eng, self.changeNumber ]
                                      if _f ] )

   def reconcileMissingFields( self, referenceVersion ):
      # Sometimes there are discrepancies between the running and programmed
      # aboot version formats, but if they have the same change # it's pretty
      # safe to assume that they should be the same
      if ( self.changeNumber == referenceVersion.changeNumber and
           self.norcal == referenceVersion.norcal ):
         if str( self.line ) == '?':
            self.line = referenceVersion.line
         if str( self.major ) == '?':
            self.major = referenceVersion.major
         if str( self.minor ) == '?':
            self.minor = referenceVersion.minor
         if not self.pcieLaneConfig:
            self.pcieLaneConfig = referenceVersion.pcieLaneConfig
         if not self.core:
            self.core = referenceVersion.core

   def hasSecureBootCpuWP( self ):
      norcalIdOrLine = self.norcalId if self.norcalId else \
                       self.line.field

      if norcalIdOrLine and self.changeNumber:
         cl = int( self.changeNumber )
         if norcalIdOrLine in ABOOT_HAS_SB_CPU_WP:
            clRange = ABOOT_HAS_SB_CPU_WP[ norcalIdOrLine ]
            return cl in clRange

      return False

   def hasPcrLogPermissionBug( self ):
      if not self.changeNumber:
         return False

      cl = int( self.changeNumber )
      return cl in ABOOT_PCR_LOG_PERMISSION_BUG

   def isResetTypeSupportedHelper( self, supportedChangeNumber ):
      norcalIdOrLine = self.norcalId if self.norcalId else \
                       self.line.field

      if norcalIdOrLine and self.changeNumber:
         cl = int( self.changeNumber )
         if norcalIdOrLine in supportedChangeNumber:
            clRange = supportedChangeNumber[ norcalIdOrLine ]
            return cl >= clRange

      return False

   def isRollbackSupported( self ):
      return self.isResetTypeSupportedHelper( ABOOT_ROLLBACK_SUPPORTED )
   
   def isFactoryResetSupported( self ):
      return self.isResetTypeSupportedHelper( ABOOT_FACTORY_RESET_SUPPORTED )
   
   def abortRollbackWithEmptyDir( self ):
      return int( self.changeNumber ) >= ABOOT_ABORT_ROLLBACK

def parseVersion( versionString ):
   if versionString is None:
      versionString = ""
   if AbootVersion.matchProduct( versionString ):
      return AbootVersion( versionString )
   return Version( versionString )
