#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import OrderedDict, namedtuple

from Ark import ReversibleDict

CounterFeatureInfo = namedtuple( 'CounterFeatureInfo', [
   # This is the list of tokens used by the CLI command.
   'cliConfigTokens',
   # This is the "legacy" tuple that was in `featureMap`.
   'featureDirectionOption',
   # This is a boolean representing whether or not to auto generate the CLI.
   'autogenCliCfg',
   # This is the human friendly string used to represent the feature in CLI outputs.
   'featureDisplayString',
   # This is the human friendly string used to represent the direction in CLI
   # outputs.
   'featureDisplayDirection',
] )

# This is a map of features configurable from the CLI and that have their CLI
# commands auto-generated. It MUST match the CLI command defined for the feature.
# The key is the featureId, the value is a CounterFeatureInfo.
# PLEASE KEEP THE BELOW MAP IN ALPHABETICAL ORDER!
counterFeatureMap = OrderedDict()
# `hardware counter feature acl out ipv4`
counterFeatureMap[ "AclEgressIpv4" ] = CounterFeatureInfo(
   ( "acl", "out", "ipv4", ),
   ( "acl", "out", "ipv4", ),
   True, "ACL-IPv4", "out" )
# `hardware counter feature acl out ipv6`
counterFeatureMap[ "AclEgressIpv6" ] = CounterFeatureInfo(
   ( "acl", "out", "ipv6", ),
   ( "acl", "out", "ipv6", ),
   True, "ACL-IPv6", "out" )
# `hardware counter feature acl out mac`
counterFeatureMap[ "AclEgressMac" ] = CounterFeatureInfo(
   ( "acl", "out", "mac", ),
   ( "acl", "out", "mac", ),
   True, "ACL-MAC", "out" )
# `hardware counter feature acl in`
counterFeatureMap[ "AclIngress" ] = CounterFeatureInfo(
   ( "acl", "in" ),
   ( "acl", "in", None, ),
   True, "ACL", "in" )
# `hardware counter feature traffic-policy cpu`
counterFeatureMap[ "CpuTrafficPolicyIngress" ] = CounterFeatureInfo(
   ( "traffic-policy", "cpu", ),
   ( "traffic-policy", None, "cpu", ),
   True, "Traffic policy CPU", "n/a" )
# `hardware counter feature decap-group`
counterFeatureMap[ "DecapGroup" ] = CounterFeatureInfo(
   ( "decap-group", ),
   ( "decap-group", None, None, ),
   True, "Decap group", "n/a" )
# `hardware counter feature directflow in`
counterFeatureMap[ "DirectFlow" ] = CounterFeatureInfo(
   ( "directflow", "in" ),
   ( "directflow", "in", None, ),
   True, "DirectFlow", "in" )
# `hardware counter feature subinterface out drop-precedence`
counterFeatureMap[ "DpSubInterfaceEgress" ] = CounterFeatureInfo(
   ( "subinterface", "out", 'drop-precedence', ),
   ( "subinterface", "out", 'drop-precedence', ),
   True, "Subinterface drop precedence", "out" )
# `hardware counter feature subinterface in drop-precedence`
counterFeatureMap[ "DpSubInterfaceIngress" ] = CounterFeatureInfo(
   ( "subinterface", "in", 'drop-precedence', ),
   ( "subinterface", "in", 'drop-precedence', ),
   True, "Subinterface drop precedence", "in" )
# `hardware counter feature ecn out`
counterFeatureMap[ "Ecn" ] = CounterFeatureInfo(
   ( "ecn", "out", ),
   ( "ecn", "out", None, ),
   True, "ECN", "out" )
# EncryptedPfcIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "EncryptedPfcIngress" ] = CounterFeatureInfo(
   None,
   None,
   False, "Encrypted PFC", "in" )
# `hardware counter feature flow-spec in`
counterFeatureMap[ "Flowspec" ] = CounterFeatureInfo(
   ( "flow-spec", "in", ),
   ( "flow-spec", "in", None, ),
   True, "Flowspec", "in" )
# `hardware counter feature flow-spec two-color in`
counterFeatureMap[ "FlowspecTwoColor" ] = CounterFeatureInfo(
   ( "flow-spec", "two-color", "in", ),
   ( "flow-spec", "two-color", "in", ),
   True, "Flowspec two-color", "in" )
# `hardware counter feature gre tunnel interface out`
counterFeatureMap[ "GreTunnelInterfaceEgress" ] = CounterFeatureInfo(
   ( "gre", "tunnel", "interface", "out", ),
   ( "gre", "out", "interface", ),
   True, "GRE tunnel interface", "out" )
# `hardware counter feature gre tunnel interface in`
counterFeatureMap[ "GreTunnelInterfaceIngress" ] = CounterFeatureInfo(
   ( "gre", "tunnel", "interface", "in", ),
   ( "gre", "in", "interface", ),
   True, "GRE tunnel interface", "in" )
# `hardware counter feature policing interface in`
counterFeatureMap[ "IntfPolicingIngress" ] = CounterFeatureInfo(
   ( "policing", "interface", "in", ),
   ( "policing", "in", "interface", ),
   True, "Interface policing", "in" )
# IntfPolicingMeterIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "IntfPolicingMeterIngress" ] = CounterFeatureInfo(
   None,
   ( "interface-policing-policer", "in", None, ),
   False, "Interface policing policer", "in" )
# `hardware counter feature traffic-policy out`
counterFeatureMap[ "IntfTrafficPolicyEgress" ] = CounterFeatureInfo(
   ( "traffic-policy", "out", ),
   ( "traffic-policy", "out", None, ),
   True, "Traffic policy", "out" )
# `hardware counter feature traffic-policy in`
counterFeatureMap[ "IntfTrafficPolicyIngress" ] = CounterFeatureInfo(
   ( "traffic-policy", "in", ),
   ( "traffic-policy", "in", None, ),
   True, "Traffic policy", "in" )
# `hardware counter feature traffic-policy vlan-interface in`
counterFeatureMap[ "IntfTrafficPolicySviIngress" ] = CounterFeatureInfo(
   ( "traffic-policy", "vlan-interface", "in", ),
   ( "traffic-policy", "in", "vlan-interface", ),
   True, "Traffic policy VLAN interface", "in" )
# `hardware counter feature traffic-policy two-color in`
counterFeatureMap[ "IntfTrafficPolicyTwoColorIngress" ] = CounterFeatureInfo(
   ( "traffic-policy", "two-color", "in", ),
   ( "traffic-policy", "two-color", "in", ),
   True, "Traffic-policy two-color", "in" )
# `hardware counter feature routed-port in`
counterFeatureMap[ "IpIngress" ] = CounterFeatureInfo(
   ( "routed-port", "in", ),
   ( "routed-port", "in", None, ),
   True, "Routed-port", "in" )
# `hardware counter feature ip-in-ip tunnel`
counterFeatureMap[ "IpInIpEncap" ] = CounterFeatureInfo(
   ( "ip-in-ip", "tunnel", ),
   ( "ip-in-ip", "tunnel", None ),
   True, "IP-in-IP tunnel", "out" )
# `hardware counter feature ip out`
counterFeatureMap[ "Ipv4v6Egress" ] = CounterFeatureInfo(
   ( "ip", "out", ),
   ( "ip", "out", None, ),
   True, "IPv4, IPv6", "out" )
# `hardware counter feature ip in`
counterFeatureMap[ "Ipv4v6Ingress" ] = CounterFeatureInfo(
   ( "ip", "in", ),
   ( "ip", "in", None, ),
   True, "IPv4, IPv6", "in" )
# `hardware counter feature subinterface out layer2`
counterFeatureMap[ "L2SubInterfaceEgress" ] = CounterFeatureInfo(
   ( "subinterface", "out", "layer2", ),
   ( "subinterface", "out", "layer2", ),
   True, "Subinterface layer2", "out" )
# `hardware counter feature subinterface in layer2`
counterFeatureMap[ "L2SubInterfaceIngress" ] = CounterFeatureInfo(
   ( "subinterface", "in", "layer2", ),
   ( "subinterface", "in", "layer2", ),
   True, "Subinterface layer2", "in" )
# L2SubInterfaceRoutedEgress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "L2SubInterfaceRoutedEgress" ] = CounterFeatureInfo(
   None,
   None,
   False, "Subinterface layer2 routed", "out" )
# `hardware counter feature ip out layer3`
counterFeatureMap[ "L3Ipv4v6Egress" ] = CounterFeatureInfo(
   ( "ip", "out", "layer3", ),
   ( "ip", "out", "layer3", ),
   True, "IPv4, IPv6 layer3", "out" )
# `hardware counter feature ip in layer3`
counterFeatureMap[ "L3Ipv4v6Ingress" ] = CounterFeatureInfo(
   ( "ip", "in", "layer3", ),
   ( "ip", "in", "layer3", ),
   True, "IPv4, IPv6 layer3", "in" )
# LatencyEndToEndHistogram doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "LatencyEndToEndHistogram" ] = CounterFeatureInfo(
   None,
   None,
   False, "Latency end-to-end histogram", "out" )
# LatencyEndToEndMax doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "LatencyEndToEndMax" ] = CounterFeatureInfo(
   None,
   None,
   False, "Latency end-to-end maximum", "out" )
# MaxQueueSize doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "MaxQueueSize" ] = CounterFeatureInfo(
   None,
   ( "lanz", "in", None, ),
   False, "LANZ", "in" )
# `hardware counter feature multicast ipv4`
counterFeatureMap[ "Mcast" ] = CounterFeatureInfo(
   ( "multicast", "ipv4", ),
   ( "multicast", None, "ipv4", ),
   True, "Multicast-IPv4", "in" )
# `hardware counter feature mpls interface in`
counterFeatureMap[ "MplsInterfaceIngress" ] = CounterFeatureInfo(
   ( "mpls", "interface", "in", ),
   ( "mpls", "in", "interface", ),
   True, "MPLS interface", "in" )
# `hardware counter feature mpls lfib`
counterFeatureMap[ "MplsLfib" ] = CounterFeatureInfo(
   ( "mpls", "lfib", ),
   ( "mpls", "in", "lfib", ),
   True, "MPLS LFIB", "in" )
# `hardware counter feature mpls tunnel`
counterFeatureMap[ "MplsTunnel" ] = CounterFeatureInfo(
   ( "mpls", "tunnel", ),
   ( "mpls", "out", "tunnel", ),
   True, "MPLS tunnel", "out" )
# `hardware counter feature multicast ipv6`
counterFeatureMap[ "MulticastIpv6" ] = CounterFeatureInfo(
   ( "multicast", "ipv6", ),
   ( "multicast", None, "ipv6", ),
   True, "Multicast-IPv6", "in" )
# `hardware counter feature nexthop`
counterFeatureMap[ "Nexthop" ] = CounterFeatureInfo(
   ( "nexthop", ),
   ( "nexthop", None, None, ),
   True, "Nexthop", "n/a" )
# `hardware counter feature nexthop cbf`
counterFeatureMap[ "NexthopCbf" ] = CounterFeatureInfo(
   ( "nexthop", "cbf", ),
   ( "nexthop", "cbf", None, ),
   True, "Nexthop CBF", "n/a" )
# OamLmEgress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "OamLmEgress" ] = CounterFeatureInfo(
   None,
   ( "ethernet-oam", "loss-measurement", "out", None ),
   False, "Ethernet OAM loss measurement", "out" )
# OamLmIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "OamLmIngress" ] = CounterFeatureInfo(
   None,
   ( "ethernet-oam", "loss-measurement", "in", None ),
   False, "Ethernet OAM loss measurement", "in" )
# OamSyntheticLm doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "OamSyntheticLm" ] = CounterFeatureInfo(
   None,
   ( "ethernet-oam", "loss-measurement", "synthetic" ),
   False, "Ethernet OAM synthetic loss measurement", "n/a" )
# `hardware counter feature pbr in`
counterFeatureMap[ "PbrIngress" ] = CounterFeatureInfo(
   ( "pbr", "in" ),
   ( "pbr", "in", None, ),
   True, "PBR", "in" )
# `hardware counter feature qos in`
counterFeatureMap[ "QosIngress" ] = CounterFeatureInfo(
   ( "qos", "in", ),
   ( "qos", "in", None, ),
   True, "QoS", "in" )
# 'hardware counter feature qos dual-rate-policer in'
counterFeatureMap[ "QosTrTcmIngress" ] = CounterFeatureInfo(
   ( "qos", "dual-rate-policer", "in" ),
   ( "qos", "in", "trtcm" ),
   True, "QoS dual-rate-policer", "in" )
# QosMeterIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "QosMeterIngress" ] = CounterFeatureInfo(
   None,
   ( "qos-meter", "in", None, ),
   False, "QoS policer", "in" )
# QueueEgress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "QueueEgress" ] = CounterFeatureInfo(
   None,
   ( 'queue', "out", None, ),
   False, "Queue", "out" )
# RealTimeTrafficBasedWfq doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "RealTimeTrafficBasedWfq" ] = CounterFeatureInfo(
   None,
   ( "real-time-traffic-based-wfq", "in", None, ),
   False, "Real-time traffic-based WFQ", "in" )
# `hardware counter feature rfc2544`
counterFeatureMap[ "Rfc2544NetworkBenchmarks" ] = CounterFeatureInfo(
   ( "rfc2544", ),
   ( "rfc2544", None ),
   True, "RFC2544", "out" )
# `hardware counter feature route` CLI is hand-written in AleCountersCli.
counterFeatureMap[ "Route" ] = CounterFeatureInfo(
   # Keep tokens of this command here so we can at least return the configuration
   # command from `counterFeatureConfigCommand`, but the command is not autogenerated
   # since autogenCliCfg flag is False.
   ( "route", ),
   ( "route", None, None, ),
   False, "Route", "n/a" )
# `hardware counter feature route ipv4`
counterFeatureMap[ "RouteIpv4" ] = CounterFeatureInfo(
   ( "route", "ipv4", ),
   ( "route", "in", "ipv4", ),
   True, "Route IPv4", "in" )
# SampledFlowTracking doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "SampledFlowTracking" ] = CounterFeatureInfo(
   None,
   ( "sampled-flow-tracking", "in", None, ),
   False, "Sampled flow tracking", "in" )
# `hardware counter feature segment-security`
counterFeatureMap[ "SegmentSecurity" ] = CounterFeatureInfo(
   ( "segment-security", "in", ),
   ( "segment-security", "in", None, ),
   True, "Segment-security", "in" )
# SflowAccel doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "SflowAccel" ] = CounterFeatureInfo(
   None,
   ( "sflow-accel", "in", None, ),
   False, "sFlow accel", "in" )
# `hardware counter feature storm-control`
counterFeatureMap[ "StormControlIngress" ] = CounterFeatureInfo(
   ( "storm-control", ),
   ( "storm-control", "in", None, ),
   True, "Storm control", "in" )
# StormControlMeterIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "StormControlMeterIngress" ] = CounterFeatureInfo(
   None,
   ( "storm-control-meter", "in", None, ),
   False, "Storm control policer", "in" )
# `hardware counter feature subinterface out`
counterFeatureMap[ "SubInterfaceEgress" ] = CounterFeatureInfo(
   ( "subinterface", "out", ),
   ( "subinterface", "out", None, ),
   True, "Subinterface", "out" )
# `hardware counter feature subinterface in`
counterFeatureMap[ "SubInterfaceIngress" ] = CounterFeatureInfo(
   ( "subinterface", "in", ),
   ( "subinterface", "in", None, ),
   True, "Subinterface", "in" )
# SubInterfaceStormControlMeterIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "SubInterfaceStormControlMeterIngress" ] = CounterFeatureInfo(
   None,
   ( "subinterface-storm-control-meter", "in", None, ),
   False, "Subinterface storm control policer", "in" )
# `hardware counter feature vlan-interface out`
counterFeatureMap[ "SviEgress" ] = CounterFeatureInfo(
   ( "vlan-interface", "out", ),
   ( "vlan-interface", "out", None, ),
   True, "VLAN interface", "out" )
# `hardware counter feature vlan-interface in`
counterFeatureMap[ "SviIngress" ] = CounterFeatureInfo(
   ( "vlan-interface", "in", ),
   ( "vlan-interface", "in", None, ),
   True, "VLAN interface", "in" )
# `hardware counter feature tapagg in`
counterFeatureMap[ "TapAggIngress" ] = CounterFeatureInfo(
   ( "tapagg", "in", ),
   ( "tapagg", "in", None, ),
   True, "Tap Aggregation", "in" )
# `hardware counter feature traffic-class in`
counterFeatureMap[ "TcIngress" ] = CounterFeatureInfo(
   ( "traffic-class", "in", ),
   ( "traffic-class", "in", None, ),
   True, "Traffic-class", "in" )
# `hardware counter feature vlan out`
counterFeatureMap[ "VlanEgress" ] = CounterFeatureInfo(
   ( "vlan", "out", ),
   ( "vlan", "out", None, ),
   True, "VLAN", "out" )
# `hardware counter feature vlan in`
counterFeatureMap[ "VlanIngress" ] = CounterFeatureInfo(
   ( "vlan", "in", ),
   ( "vlan", "in", None, ),
   True, "VLAN", "in" )
# `hardware counter feature vni decap`
counterFeatureMap[ "VniDecap" ] = CounterFeatureInfo(
   ( "vni", "decap", ),
   ( "vni", "decap", None, ),
   True, "VNI", "decap" )
# `hardware counter feature vni encap`
counterFeatureMap[ "VniEncap" ] = CounterFeatureInfo(
   ( "vni", "encap", ),
   ( "vni", "encap", None, ),
   True, "VNI", "encap" )
# `hardware counter feature policing vni decap`
counterFeatureMap[ "VniPolicingDecap" ] = CounterFeatureInfo(
   ( "policing", "vni", "decap", ),
   ( "policing", "decap", "vni", ),
   True, "VNI policing", "decap" )
# `hardware counter feature policing vni encap`
counterFeatureMap[ "VniPolicingEncap" ] = CounterFeatureInfo(
   ( "policing", "vni", "encap", ),
   ( "policing", "encap", "vni", ),
   True, "VNI policing", "encap" )
# VoqIngress doesn't have a `hardware counter feature` CLI.
counterFeatureMap[ "VoqIngress" ] = CounterFeatureInfo(
   None,
   ( 'voq', 'in', None, ),
   False, "VOQ", "in" )
# `hardware counter feature vtep decap`
counterFeatureMap[ "VtepDecap" ] = CounterFeatureInfo(
   ( "vtep", "decap", ),
   ( "vtep", "decap", None, ),
   True, "VTEP", "decap" )
# `hardware counter feature vtep encap`
counterFeatureMap[ "VtepEncap" ] = CounterFeatureInfo(
   ( "vtep", "encap", ),
   ( "vtep", "encap", None, ),
   True, "VTEP", "encap" )
# PLEASE KEEP THE ABOVE MAP IN ALPHABETICAL ORDER!

featureMap = OrderedDict()
featureIdToTuple = {}

def buildLegacyMaps():
   for featureId, featureInfo in counterFeatureMap.items():
      # Sanity check the map.
      autogenCliCfg = featureInfo.autogenCliCfg
      cliConfigTokens = featureInfo.cliConfigTokens
      if autogenCliCfg:
         assert cliConfigTokens

      # Build other "legacy" maps from the main one.
      tupleForFeatureMap = featureInfo.featureDirectionOption
      if autogenCliCfg or featureId in ( "MaxQueueSize", "Route", ):
         featureMap[ tupleForFeatureMap ] = featureId
         featureIdToTuple[ featureId ] = tupleForFeatureMap

buildLegacyMaps()

def counterFeatureConfigCommand( featureId ):
   featureInfo = counterFeatureMap.get( featureId )
   assert featureInfo is not None, f"No tokens exist for {featureId}"
   assert featureInfo.cliConfigTokens, f"No tokens exist for {featureId}"

   return 'hardware counter feature ' + ' '.join( featureInfo.cliConfigTokens )

featureTokenToHelpdescMap = {
   "in" : "Ingress",
   "out" : "Egress",
   "encap" : "Encapsulation path",
   "decap" : "Decapsulation path",
   "interface" : "Interface",
   "lfib" : "LFIB counters",
   "tunnel" : "Tunnel counters",
   "ipv4" : "IPv4 related",
   "ipv6" : "IPv6 related",
   "mac" : "Ethernet address",
   "layer2" : "Layer2",
   "drop-precedence" : "Drop precedence",
   "cpu" : "CPU traffic-policy counters",
   "acl" : "ACL",
   "pbr" : "PBR",
   "vlan" : "VLAN",
   "lanz" : "LANZ",
   "ip" : "IPv4, IPv6",
   "ip-in-ip" : "IP-in-IP tunnel",
   "mpls" : "MPLS",
   "decap-group" : "Decap group",
   "vtep" : "VTEP",
   "vni" : "VNI",
   "qos" : "QoS",
   "flow-spec" : "Flowspec",
   "gre" : "GRE tunnel",
   "voq" : "VOQ",
   "route" : "Route",
   "segment-security" : "Segment security",
   "tapagg" : "Tap Aggregation",
   "vlan-interface" : "Logical interface into a VLAN",
}

def tokenHelpdesc( token ):
   """Builds help description string based on feature token."""

   # Treatment for acronyms.
   return featureTokenToHelpdescMap.get( token, token.capitalize() )

def featureNameAndDirection( featureId ):
   """Builds feature name and direction for display based on featureId."""

   featureInfo = counterFeatureMap[ featureId ]
   featureName = featureInfo.featureDisplayString
   direction = featureInfo.featureDisplayDirection
   assert featureName
   assert direction

   return featureName, direction

unitsTokenToCountModeMap = {
   None : 'featureCountModePlatformDefault',
   'packets' : 'featureCountModePktOnly',
}

unitsCountModeToTokenMap = ReversibleDict( unitsTokenToCountModeMap ).reverse()
