# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections.abc import Iterable

import json
import Tracing

traceHandle = Tracing.Handle( 'JsonUwsgiServer' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

ATTRNAME = object()

class Field:
   __slots__ = (
      'name',
      'apiName',
      'tacName',
      'versions',
      'inputOk',
      'outputOk',
      'description',
      '__class__',
   )
   __field_type__ = None

   def __init__( self, *,
                 apiName=ATTRNAME,
                 tacName=ATTRNAME,
                 versions=(),
                 inputOk=True,
                 outputOk=True,
                 description='',
                 fieldType=None
                ):
      self.name = ''
      self.apiName = apiName
      self.tacName = tacName
      self.versions = versions
      self.inputOk = inputOk
      self.outputOk = outputOk
      self.description = description

   def __set_name__( self, owner, name ):
      self.name = name
      if self.apiName is ATTRNAME:
         self.apiName = name
      if self.tacName is ATTRNAME:
         self.tacName = name
      owner.__attributes__[ self.apiName ] = self

   def __get__( self, obj, objtype=None ):
      return obj.__values__.get( self.apiName )

   def __set__( self, obj, value ):
      # pylint: disable=isinstance-second-argument-not-valid-type
      if self.__field_type__ and not isinstance( value, self.__field_type__ ):
         msg = f'Type {type(value)} does not match API type {self.__field_type__}'
         warn( msg )
         raise AttributeError( msg )

      # TODO: versions support
      obj.__values__[ self.apiName ] = value

   @property
   def default( self ):
      return None

class Str( Field ):
   __field_type__ = str
   __slots__ = ()

class Int( Field ):
   __field_type__ = int
   __slots__ = ()

class Float( Field ):
   __field_type__ = ( float, int )
   __slots__ = ()

class Bool( Field ):
   __field_type__ = bool
   __slots__ = ()

class Dict( Field ):
   __field_type__ = dict
   __slots__ = ()

class ModelField( Dict ):
   __slots__ = ( '__value_type__', )

   def __init__( self, *,
                 apiName=ATTRNAME,
                 tacName=ATTRNAME,
                 versions=(),
                 inputOk=True,
                 outputOk=True,
                 description='',
                 valueType=None ):
      super().__init__(
         apiName=apiName,
         tacName=tacName,
         versions=versions,
         inputOk=inputOk,
         outputOk=outputOk,
         description=description
      )

      if valueType:
         self.__value_type__ = valueType

   def __set__( self, obj, value ):
      # pylint: disable=isinstance-second-argument-not-valid-type
      if ( self.__field_type__ is not self.__value_type__ and
              isinstance( value, self.__field_type__ )
            ):
         m = self.__value_type__()
         m.populateModelFromJson( value )
         value = m

      if self.__value_type__ and not isinstance( value, self.__value_type__ ):
         if not isinstance( value, self.__field_type__ ):
            msg = ( f'Type {type(value)} does not match dict or API type '
                    f'{self.__field_type__}' )
            warn( msg )
            raise AttributeError( msg )

      obj.__values__[ self.apiName ] = value

class ListType( list ):
   slots = ( '__field_type__', )

   def __init__( self, iterable=(), *, field_type=None ):
      self.__field_type__ = field_type
      list.__init__( self )
      self.extend( iterable )

   def __checkOne( self, value ):
      if self.__field_type__ and not isinstance( value, self.__field_type__ ):
         msg = f'Type {type(value)} does not match API type {self.__field_type__}'
         warn( msg )
         raise ValueError( msg )
      return value

   def __checkMany( self, iterable ):
      for i in iterable:
         if self.__field_type__ and not isinstance( i, self.__field_type__ ):
            msg = f'Type {type(i)} does not match API type {self.__field_type__}'
            warn( msg )
            raise ValueError( msg )
         yield i

   def append( self, iterable ):
      list.append( self, self.__checkMany( iterable ) )

   def extend( self, iterable ):
      list.extend( self, self.__checkMany( iterable ) )

   def __iadd__( self, other ):
      list.extend( self, self.__checkMany( other ) )

   def __add__( self, other ):
      retval = self.__class__( self, field_type=self.__field_type__ )
      retval.extend( other )
      return retval

   def __setitem__( self, key, item ):
      if isinstance( key, slice ):
         self[ key ] = self.__checkMany( item )
      else:
         self[ key ] = self.__checkOne( item )

   def __getitem__( self, key ):
      if isinstance( key, slice ):
         return self.__class__(
            list.__getitem__( self, key ),
            field_type=self.__field_type__
         )
      else:
         return list.__getitem__( self, key )

   def insert( self, i, item ):
      list.insert( self, i, self.__checkOne( item ) )

   def copy( self, *args, **kwargs ):
      return self.__class__( self, field_type=self.__field_type__ )

class List( Field ):
   __slots__ = ( '__field_type__', )

   def __init__( self, *,
                 apiName=ATTRNAME,
                 tacName=ATTRNAME,
                 versions=(),
                 inputOk=True,
                 outputOk=True,
                 description='',
                 valueType=None ):
      super().__init__(
         apiName=apiName,
         tacName=tacName,
         versions=versions,
         inputOk=inputOk,
         outputOk=outputOk,
         description=description
      )

      self.__field_type__ = valueType

   def __set__( self, obj, value ):
      if not isinstance( value, Iterable ):
         msg = f'Type {type(value)} does not match API type list'
         warn( msg )
         raise AttributeError( msg )

      if self.__field_type__:
         obj.__values__[ self.apiName ] = ListType(
            value,
            field_type=self.__field_type__
         )
      else:
         obj.__values__[ self.apiName ] = value

   @property
   def default( self ):
      return []

   def fromSysdb( self, obj, collection ):
      if self.__field_type__ and issubclass( self.__field_type__, BaseModel ):
         items = []
         for i in collection.values():
            m = self.__field_type__()  # pylint: disable=not-callable
            m.fromSysdb( i )
            items.append( m )
      else:
         items = collection
      self.__set__( obj, items )

   def populateModelFromJson( self, obj, collection ):
      if self.__field_type__ and issubclass( self.__field_type__, BaseModel ):
         items = []
         for i in collection:
            m = self.__field_type__()  # pylint: disable=not-callable
            m.populateModelFromJson( i )
            items.append( m )
      else:
         items = collection
      self.__set__( obj, items )

class BaseModelMeta( type ):
   def __new__( cls, name, bases, namespace ):
      attrs = {}
      for b in reversed( bases ):
         if issubclass( b, BaseModel ):
            attrs.update( b.__attributes__ )

      assert '__attributes__' not in namespace
      namespace[ '__attributes__' ] = attrs
      if '__slots__' not in namespace:
         namespace[ '__slots__' ] = ()
      newcls = type.__new__( cls, name, bases, namespace )
      newcls.__apinames__ = { a.apiName: a for a in newcls.__attributes__.values() }
      return newcls


class BaseModel( metaclass=BaseModelMeta ):
   __slots__ = ( '__values__', )

   def __init__( self, **kwargs ):
      self.__values__ = {}
      for k, v in kwargs.items():
         try:
            setattr( self, k, v )
         except AttributeError:
            raise TypeError(
               f'{k} is an invalid keyword argument for {self.__class__.__name__!r}'
            ) from None

   def fromSysdb( self, t ):
      for attr in self.__attributes__.values():  # pylint: disable=no-member
         if attr.tacName:
            if attr.outputOk:
               if isinstance( attr, List ):
                  attr.fromSysdb( self, getattr( t, attr.tacName ) )
               else:
                  setattr( self, attr.name, getattr( t, attr.tacName ) )
            else:
               raise AttributeError( f'{attr.tacName} not allowed as output' )

   def getPopulatedModelFields( self, key=None ):
      return (
         ( attr.tacName if key == 'tacName' else attr.apiName, value )
         for apiName, value in self.__values__.items()
         for attr in self.__attributes__.values()  # pylint: disable=no-member
         if attr.tacName and attr.apiName == apiName
      )

   def setModelField( self, name, value, version=1 ):
      # ignore version for now
      # pylint: disable-next=no-member
      self.__attributes__[ name ].__set__( self, value )

   def populateModelFromJson( self, jsondata ):
      for k, v in jsondata.items():
         # pylint: disable-next=no-member
         a = self.__attributes__.get( k )
         if not a:
            # Since the API is meant to handle unkown params we continue without
            # an error
            warn( f'Unkown attribute {k} in {jsondata}' )
            continue
         if not a.inputOk:
            msg = f'Tried to write non-writable attribute {a.apiName}'
            warn( msg )
            raise AttributeError( msg )
         if isinstance( a, List ):
            a.populateModelFromJson( self, v )
         else:
            a.__set__( self, v )

   def toSysdb( self, entity ):
      """
      Subclasses should override this method to serialize
      the model to Sysdb. The entity parameter must be the
      root of anything that needs to be written to Sysdb.
      """
      pass # pylint: disable=unnecessary-pass

   @property
   def apiDict( self ):
      return {
         a.apiName: self.__values__.get( a.apiName, a.default )
         for a in self.__attributes__.values()  # pylint: disable=no-member
      }

   def __repr__( self ):
      return repr( self.apiDict )

class ModelJsonSerializer( json.JSONEncoder ):
   def default( self, o ):
      if isinstance( o, BaseModel ):
         return o.apiDict
      return json.JSONEncoder.default( self, o )
