#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from contextlib import contextmanager
import os

@contextmanager
def disableSubprocessAsanLeakcheck():
   existingAsanOptions = os.environ.get( 'ASAN_OPTIONS', None )
   asanOptions = ""
   if existingAsanOptions:
      asanOptions = existingAsanOptions + ":detect_leaks=false"
   else:
      asanOptions = "detect_leaks=false"
   os.environ[ 'ASAN_OPTIONS' ] = asanOptions

   yield

   if not existingAsanOptions:
      os.environ.pop( 'ASAN_OPTIONS' )
   else:
      os.environ[ 'ASAN_OPTIONS' ] = existingAsanOptions

@contextmanager
def disableSubprocessAsanPreload():
   """
   Some processes don't run "well" when libasan is preloaded e.g. sysctl hangs
   waiting on a futex call. As EOS devleopers are not interested in ASan-checking
   these subprocesses, this context manager can be used to temporarily remove the
   LD_PRELOAD setting.
   """
   origLdPreload = os.environ.get( 'LD_PRELOAD', '' )
   ASAN_LIB = 'libasan.so.6'
   cleanup = False
   if ASAN_LIB in origLdPreload:
      delimiter = ':' if ':' in origLdPreload else ' '
      preloads = origLdPreload.split( delimiter )
      preloads.remove( ASAN_LIB )
      os.environ[ 'LD_PRELOAD' ] = delimiter.join( preloads )
      cleanup = True

   yield

   if cleanup:
      os.environ[ 'LD_PRELOAD' ] = origLdPreload
