# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import glob
import os
import sys
import sysconfig
from itertools import chain

PY2_LIBDIR = os.environ.get( 'PYTHONPATH' ) or '/usr/lib/python2.7/site-packages'
PY3_LIBDIR = sysconfig.get_path( 'purelib' )

installedFinders_ = []

def uninstallCheatLoader():
   '''
   Remove all finders we added to meta path.
   '''
   while installedFinders_:
      sys.meta_path.remove( installedFinders_.pop() )

def installCheatLoader():
   '''
   Install our cheat loader in `sys.meta_path`.
   See the dosctring for `TPF` for more details.
   '''
   try:
      # pylint: disable-next=import-outside-toplevel
      from importlib.machinery import PathFinder # Py3 only.
   except ImportError:
      return []

   class TPF( PathFinder ):
      '''
      Hacker's delight!
      When attempting to load a module, if the module does not exist,
      look for the module in the python2.7 directory.
      The motivation behind this is to see what plugins we are unable to load.
      '''
      @staticmethod
      def find_spec( fullname, path=None, target=None ):
         if path:
            module = path[ 0 ].replace( PY3_LIBDIR, PY2_LIBDIR )
         else:
            module = PY2_LIBDIR + '/'
         return PathFinder.find_spec( fullname, [ module ], target )

   # Append our finder to the default list, so we defer to it last.
   sys.meta_path.append( TPF )
   installedFinders_.append( TPF )
   sys.dont_write_bytecode = True # Leave no trail (of pyc files).

def findPlugins( pluginType ):
   # If the caller didn't explicitly speficy plugins, load all.
   # But the caveat is that a py2-only leaf plugin won't get
   # referenced by anything and then loaded via TPF, so we leave it out.
   # And we want all available plugins loaded.
   py2path = os.path.join( PY2_LIBDIR, '%s/*.py*' )
   py3path = os.path.join( PY3_LIBDIR, '%s/*.py*' )
   return { os.path.basename( f ).replace( '.pyc', '' ).replace( '.py', '' )
            for f in chain( glob.glob( py2path % pluginType ),
                            glob.glob( py3path % pluginType ) ) }
