# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

class ReadOnlyDictionaryWithPrefixKeys:
   """Act like a normal dictionary, but if the supplied key startswith a key
   which has a matching dictionary entry ending in '*', then return that item
   if there is no exact match on the key.
   For instance, in the data below we have entries for 'Lyonsville' and
   'Lyonsville*'. if 'Lyonsville' is requested, the 'Lyonsville' entry is
   returned, but if 'LyonsvilleDD', 'LyonsvilleBK', or any other variation is
   requested, the 'Lyonsville*' entry is returned.
   This allows us to use the same dictionary entry for several variants of the
   same board, so it doesn't need to be updated for every new variant, whilst
   still allowing us to have specific cases if needed.
   Be wary of extending the use of this class - it intentionally
   omits standard dictionary attributes such as keys(), items(), values(),
   and iterators, because such semantics aren't readily reconciled with
   prefix matching."""
   def __init__( self, dictionary ):
      self.prefixList_ = list() # pylint: disable=use-list-literal
      self.dictionary_ = {}
      for key, value in dictionary.items():
         self[ key ] = value

   def __contains__( self, key ):
      if key in self.dictionary_:
         return True
      for prefix in self.prefixList_:
         if key.startswith( prefix ):
            return True
      return False

   def __getitem__( self, key ):
      if key in self.dictionary_:
         return self.dictionary_[ key ]
      for prefix in self.prefixList_:
         if key.startswith( prefix ):
            return self.dictionary_[ prefix + "*" ]
      return None

   def __setitem__( self, key, value ):
      if key.endswith( "*" ):
         self.prefixList_.append( key[ : -1 ] )
      self.dictionary_[ key ] = value

   def get( self, key, default=None ):
      if key in self.dictionary_:
         return self.dictionary_.get( key, default )
      for prefix in self.prefixList_:
         if key.startswith( prefix ):
            return self.dictionary_.get( prefix + "*", default )
      return default
