#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import fcntl
import os

class LockingFile:
   """ A context manager object (a.k.a 'with helper') that provides a
   new or existing readable and writable file that is exclusively
   locked for this process."""
   
   def __init__( self, filename, mode ):
      self.filename_ = filename
      self.mode_ = mode
      self.file_ = None
      self.fd_ = None
      
   def __enter__( self ):
      try:
         oldumask = os.umask( 0 ) # So that the mode arg in the next call
            # will have the intended effect.
         self.fd_ = os.open( self.filename_, os.O_RDWR | os.O_CREAT, self.mode_ )
         os.umask( oldumask )
         fcntl.flock( self.fd_, fcntl.LOCK_EX )
         self.file_ = os.fdopen( self.fd_, "r+" )
      except Exception as e:
         print( type( e ), str( e ) )
         # Make sure the file is closed/unlocked.
         if self.fd_:
            os.close( self.fd_ )
         raise
      return self

   def write( self, s ):
      self.file_.seek( 0 )
      self.file_.write( s )
      self.file_.truncate()
      
   def __exit__( self, _type, _value, _tb ):
      self.file_.close() # Will unlock.
      return False

