# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import contextlib
import errno
import fcntl
import functools
import grp
import json
import os
import pwd
import re
import tempfile
import time

import CEosHelper
from ArchiveLib import Filesystem
import SpaceMgmtLib
import SpaceMgmtLib.Quota
import SpaceMgmtLib.Utils
from SpaceMgmtLib.Utils import (
   ONE_MIB,
   ONE_GIB,
   EOS_CORE_DIR_PATH
)
import Tac
from Toggles.LogArchiverToggleLib import toggleDisableLowStorageLifetimeEnabled

# Mountpoints
FLASH_MNT_PT = '/mnt/flash'
SSD_MNT_PT = '/mnt/drive'

# Constants
DRIVE_LIFETIME_CUTOFF = 20

class Archive:
   """
   EOS Archive, .i.e persistent location on an ext4 filesystem that stores copies
   of EOS logs. These logs are initially created in the tmpfs filesystem /var.

   Archived logs are stored in a directory named 'archive'. The latter contains
   two folders named 'var' and 'core' for storing copies of the tmpfs directories
   /var/log and /var/core. The 'archive' folder is rotated at each boot to a folder
   named 'var_archive-<date>-<time>.dir'.

   This class provides an interface to the archive. It provides attributes and
   properties to query the archive state as well as methods to configure archiving
   and managing the archive space usage.

   Notes
   -----
   This class only provide management over an archive. Copying logs from the tmpfs
   filesystem to the archive is done by the cron job archivecheck.sh every minute.
   archivecheck.sh itself uses archivetree.py to do the actual copy workload.
   See /src/LogMgr/etcfiles/archivecheck.sh and /src/LogMgr/archivetree.py for
   more details.

   Historically, EOS archived logs were stored on the ssd under /mnt/drive/archive.
   Now an archive can be on any ext4 filesystem. At boot time EOS check for possible
   archive destination in this order:
      * Existing configuration
      * ext4 ssd drive
      * ext4 flash
   This is done by the archive-init systemd service during the boot process.
   See /src/EosInit/scripts/archive-init.sh for more details.

   The archive space usage is controlled and limited using linux quotas (hence the
   need of an ext4 filesystem). Quota limits are set for the 'archive' user which
   is the uid used when archiving logs.

   If the archive fills up, we try to reclaim some space to be able to continue
   archiving. This is done by deleting older version of log files (either older
   rotations created by logrotated or, for agents, logs of an older instance of the
   agent (.i.e with a different pid than the one currently running). Core files will
   only be deleted if the associated agent log are selected for deletion.

   Note that for a minimal sanity, a least 500MiB should be free in the archive.
   This should ensure that an archiving iteration (running the copy cron job once)
   can run without the need to free more space.
   This is enforced by the cron job archivecheck which will remove archived archives
   and older rotations of log files if we run out of space.

   Attributes
   ----------
   name: str
      Nickname of the archive.
   path: str
      Absolute path to the archive destination.
   dev: Device
      Represent the filesystem device on which resides the destination path.
   rootDirPath: str
      Absolute path of the root directory, which contains the 'archive' directory
      as well as archived archives.
   coreDirPath: str
      Absolute path of the folder containing archived core files.
   logDirPath: str
      Absolute path of the directory containing archived logs.
   agentLogDirPath: str
      Absolute path of the directory containing archived agents logs.
   subdirsPaths: list of str
      List of paths of common archive subdirectories.

   Properties
   ----------
   fs

   exists
   valid
   disabled
   enabled
   statusSummary

   spaceUsageSize
   spaceUsagePct
   defaultRequiredFreeSize
   defaultRequiredFreePct
   neededReclaimSpaceSize

   quotasOn
   quotasOff
   graceTimes
   quotaStats
   quotaKiB
   quotaPct
   defaultQuotaPct
   """

   # Configuration and marker files
   configFilename = '.arista_archive_config'
   configFilePath = os.path.join( FLASH_MNT_PT, configFilename )
   markerFilename = '.arista_archive'

   # Maximum/minimum number of iterations (files) for a specific log.
   minMaxFilesCount = 2
   maxMaxFilesCount = 16
   maxArchivedArchivesCount = 12

   # Symbolink link pointing to the archive directory. The default value is /archive
   # Can be overridden with environment variable EOS_ARCHIVE_LINKPATH
   # This is used for btests to avoid tests running in parallel manipulating the
   # same link.
   linkPathEnvVar = 'EOS_ARCHIVE_LINKPATH'
   quickAccesLink = '/archive'

   # name and group of the archive user
   archiveUserName = 'archive'
   archiveGroupName = 'eosadmin'

   # Minimum free space in archive destination for sanity.
   defaultRequiredFreeSizeKiB = SpaceMgmtLib.Utils.toKiB( 500, ONE_MIB )

   # Default quota percentages
   defaultQuotaPctSdd = 20
   defaultQuotaPctFlash = 5
   defaultQuotaPctOther = 10

   # Directories
   baseDir = 'archive'
   currentSubDir = 'current'
   historySubDir = 'history'

   # Archived archives name prefix
   archivedArchivePrefix = 'var_archive.'

   # Lock files
   archiveLockFilename = 'LCK..arista_archive'
   archiveLockFilePath = os.path.join( '/var/lock', archiveLockFilename )
   archiveLockTimeoutEnvVar = 'ARCHIVE_LOCK_TIMEOUT'

   #################################################################################
   # Class methods
   #################################################################################

   _archiveUsrId = None

   @classmethod
   def archiveUsrId( cls ):
      """Numeric uid of the archive user."""

      if cls._archiveUsrId is None:
         cls._archiveUsrId = pwd.getpwnam( cls.archiveUserName ).pw_uid
      return cls._archiveUsrId

   _archiveGrpId = None

   @classmethod
   def archiveGrpId( cls ):
      """Numeric gid of the archive group."""

      if cls._archiveGrpId is None:
         cls._archiveGrpId = grp.getgrnam( cls.archiveGroupName ).gr_gid
      return cls._archiveGrpId

   @classmethod
   def configFileInfo( cls ):
      """
      Info from configuration file

      Returns
      -------
      dict
         'name': str
            Name of the archive.
         'path': str
            Path of the archive.
         'quotaPct': int
            Quota percentage.
         'enabled': bool
            True if archiving is enabled, False otherwise.
         'storageLifetime': int
            Percent wear of drive used for archive
      """

      with open( cls.configFilePath ) as f:
         content = f.read()

      info = json.loads( content )

      values = {
         str( key ) : str( val ) if isinstance( val, str ) else val
         for key, val in info.items()
      }

      # it is possible for storageLifetime to not be present in the config file
      # since it was added later
      if values.get( 'storageLifetime' ) is None:
         values[ 'storageLifetime' ] = 100

      return values

   @classmethod
   def writeConfig( cls, name, path, quotaPct, enabled, storageLifetime ):
      """
      Write the name, path, quotaPct enabled, and storageLifetime name / value
      pairs stored in the info object to the json format archive configuration file.
      """
      tmpPath = ""

      info = {
         'name' : name,
         'path' : path,
         'quotaPct' : quotaPct,
         'enabled' : enabled,
      }

      if toggleDisableLowStorageLifetimeEnabled():
         info[ 'storageLifetime' ] = storageLifetime

      # Write the new configuration to a temporary file and then rename the
      # latter to the real archive config file.
      # If we write to the config file directly, and another process try to read
      # it at the same time, the content read can be invalid.

      try:
         fh, tmpPath = tempfile.mkstemp( suffix='.new_arista_archive_config',
                                         dir=FLASH_MNT_PT )
         os.chmod( tmpPath, 0o0664 )

         with open( tmpPath, 'w' ) as tmpFile:
            tmpFile.write( json.dumps( info ) )

         os.close( fh )
         os.rename( tmpPath, cls.configFilePath )
      except OSError:
         if tmpPath:
            os.remove( tmpPath )

   @classmethod
   def updateConfig( cls, name=None, path=None, quotaPct=-1,
                     enabled=None, storageLifetime=None ):
      """
      Update the config file.
      Only change the value of fields where the corresponding named argument
      has a different value than the the default one.
      """

      if not os.path.isfile( cls.configFilePath ):
         info = {
            'name' : '',
            'path' : '',
            'quotaPct' : None,
            'enabled' : True,
            'storageLifetime' : 100
         }
      else:
         info = cls.configFileInfo()

      if name is None:
         name = info[ 'name' ]
      if path is None:
         path = info[ 'path' ]
      if quotaPct == -1:
         quotaPct = info[ 'quotaPct' ]
      if enabled is None:
         enabled = info[ 'enabled' ]
      if storageLifetime is None:
         storageLifetime = info[ 'storageLifetime' ]

      cls.writeConfig( name, path, quotaPct, enabled, storageLifetime )

   @classmethod
   def clearConfig( cls ):
      """Remove archive configuration file if it exists."""

      # Need Tac.run for root privileges
      cmd = [ 'rm', '-f', Archive.configFilePath ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   @classmethod
   def currentArchive( cls ):
      """
      Return an instance of Archive corresponding to the archive destination
      referenced in the configuration file if any.
      """

      if not os.path.isfile( cls.configFilePath ):
         return False

      try:
         configInfo = cls.configFileInfo()
         if not configInfo[ 'name' ] and not configInfo[ 'path' ]:
            return False
         return cls( configInfo[ 'name' ], configInfo[ 'path' ] )
      except ( OSError, ValueError ):
         return None

   @classmethod
   def availableDests( cls ):
      """
      Check the available archiving destinations on the system.
      Looks for:
         * ext4 mounted filesystem at /mnt/flash
         * ext4 mounted filesystem at /mnt/drive
         * Currently configured archive if there is any.

      Returns
      -------
      dict of str: str
         key: Destination name.
         value: Associated path.
      """

      def _isExt4Mount( mntPt ):
         if not os.path.ismount( mntPt ):
            return False
         mntInfo = SpaceMgmtLib.Utils.mountInfo( mntPt=mntPt )
         return mntInfo is not None and mntInfo[ 'fsType' ] == 'ext4'

      dests = {}

      if _isExt4Mount( SSD_MNT_PT ):
         dests[ 'drive' ] = SSD_MNT_PT

      if _isExt4Mount( FLASH_MNT_PT ):
         fsInfo = SpaceMgmtLib.Utils.fileSystemInfo( FLASH_MNT_PT )
         # cEOS-lab is likely to have the container host's filesystem
         # mounted at /mnt/flash and it's likely to be ext4.
         # However, we ignore this case because we don't manage
         # quotas on cEOS at all, which causes a single container
         # to be able to completely fill the host's disk with
         # core files.
         if ( not CEosHelper.isCeosLab()
              and fsInfo is not None
              and fsInfo[ 'size' ] > SpaceMgmtLib.Utils.toKiB( 20, ONE_GIB ) ):
            dests[ 'flash' ] = FLASH_MNT_PT

      return dests

   #################################################################################
   # Init
   #################################################################################

   def __init__( self, name, rootDirPath, rotate=False ):
      """
      Parameters
      ----------
      name: str
         Nickname of the archive.
      rootDirPath: str
         Absolute path to the archive destination. This is the path of the
         directory that will contain the 'archive' dir as well as rotated archive.
      rotate: bool, optional.
         If True, rotate any existing 'archive' directory in 'rootDirPath'.
      """

      self.name = name
      self.path = os.path.join( rootDirPath, Archive.baseDir, Archive.currentSubDir )
      self.dev = Filesystem.Filesystem( rootDirPath ).dev

      msg = 'Destination device not mounted: archive path in root filesystem'
      assert self.dev.mntPt != '/', msg

      # Common paths
      self.rootDirPath = rootDirPath
      self.baseDirPath = os.path.join( self.rootDirPath, Archive.baseDir )
      self.historyDirPath = os.path.join( self.baseDirPath, Archive.historySubDir )
      self.coreDirPath = os.path.join( self.path, 'var/core' )
      self.logDirPath = os.path.join( self.path, 'var/log' )
      self.agentLogDirPath = os.path.join( self.logDirPath, 'agents' )
      self.markerFilePath = os.path.join( self.rootDirPath, Archive.markerFilename )
      self.subdirsPaths = [ self.coreDirPath, self.logDirPath, self.agentLogDirPath ]

      linkPathEnv = os.getenv( Archive.linkPathEnvVar )
      if linkPathEnv is not None:
         Archive.quickAccesLink = linkPathEnv

   #################################################################################
   # Locks
   #################################################################################

   def lockAccess( self, blocking=False, timeout=300 ):
      envTimeout = os.getenv( Archive.archiveLockTimeoutEnvVar )
      if envTimeout is not None:
         timeout = int( envTimeout )

      retry = timeout is not None

      # Create the file as root and set permissions so the lock can be grabbed
      # from the Cli process
      if not os.path.exists( Archive.archiveLockFilePath ):
         cmds = [
            [ 'touch', Archive.archiveLockFilePath ],
            [ 'chown', 'root:eosadmin', Archive.archiveLockFilePath ],
            [ 'chmod', '660', Archive.archiveLockFilePath ]
         ]
         for cmd in cmds:
            # Need Tac.run for root privileges
            Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

      # pylint: disable-next=consider-using-with
      lockFile = open( Archive.archiveLockFilePath, 'w+' )

      def tryAcquireLock():
         try:
            fcntl.flock( lockFile,
                         fcntl.LOCK_EX | ( 0 if blocking else fcntl.LOCK_NB ) )
            return True
         except OSError as e:
            if e.errno in ( errno.EACCES, errno.EAGAIN ):
               return False
            raise

      lockAcquired = False
      try:
         if blocking or not retry:
            if not tryAcquireLock():
               return None
         else:
            Tac.waitFor( tryAcquireLock,
                         timeout=timeout,
                         description='Archive lock to be available',
                         warnAfter=None,
                         sleep=True )

         lockFile.write( str( os.getpid() ) )
         lockFile.flush()
         lockAcquired = True
         return lockFile
      finally:
         if not lockAcquired:
            lockFile.close()

   def unlockAccess( self, lockFile ):
      fcntl.flock( lockFile, fcntl.LOCK_UN )
      lockFile.close()

   #################################################################################
   # Context managers
   #################################################################################

   @contextlib.contextmanager
   def runWithDisabledArchive( self ):
      """Make sure archive is disabled."""

      disabled = self.disabled
      if not disabled:
         self.disable()
      try:
         yield
      finally:
         if not disabled:
            self.enable()

   @contextlib.contextmanager
   def runWithQuotaOff( self ):
      """Make sure quota is turned off."""

      quotaOff = self.quotasOff
      if not quotaOff:
         self.disableQuotas()
      try:
         yield
      finally:
         if not quotaOff:
            self.enableQuotas()

   @contextlib.contextmanager
   def runWithLockedAccess( self, blocking=False, timeout=300 ):
      """Make sure archive is locked."""

      lockFile = self.lockAccess( blocking=blocking, timeout=timeout )
      try:
         yield
      finally:
         self.unlockAccess( lockFile=lockFile )

   #################################################################################
   # Decorators
   #################################################################################

   class _Decorators:
      @classmethod
      def _runInsideContextManager( cls,
                                    decoratedMethod,
                                    getContextManager,
                                    *ctxMgrExtraArgs,
                                    **ctxMgrExtraKwargs ):

         @functools.wraps( decoratedMethod )
         def wrapper( *methodArgs, **methodKwargs ):
            archive = methodArgs[ 0 ] # method's 'self' argument
            with getContextManager()( archive,
                                      *ctxMgrExtraArgs,
                                      **ctxMgrExtraKwargs ):
               return decoratedMethod( *methodArgs, **methodKwargs )

         return wrapper

      @classmethod
      def runWithDisabledArchive( cls, decoratedMethod ):
         """Make sure archive is disabled."""

         getContextManager = lambda: Archive.runWithDisabledArchive
         return cls._runInsideContextManager( decoratedMethod, getContextManager )

      @classmethod
      def runWithQuotaOff( cls, decoratedMethod ):
         """Make sure quota is turned off."""

         getContextManager = lambda: Archive.runWithQuotaOff
         return cls._runInsideContextManager( decoratedMethod, getContextManager )

      @classmethod
      def runWithLockedAccess( cls, blocking=False, timeout=300 ):
         """Make sure archive is locked."""

         def noArgDecorator( decoratedMethod ):
            getContextManager = lambda: Archive.runWithLockedAccess
            return cls._runInsideContextManager( decoratedMethod,
                                                 getContextManager,
                                                 blocking=blocking,
                                                 timeout=timeout )
         return noArgDecorator

   #################################################################################
   # Filesystem property
   #################################################################################

   @property
   def fs( self ):
      """Instance of Fil of the filesystem holding the archive."""

      return self.dev.fs

   #################################################################################
   # Paths
   #################################################################################

   @property
   def quotaUserFilePath( self ):
      """Path of quota file."""

      return self.fs.quotaUserFilePath

   def searchArchivedArchivesAt( self, searchDirPath ):
      if not os.path.isdir( searchDirPath ):
         return []

      archivedArchivesPaths = [
         os.path.join( searchDirPath, filename )
         for filename in os.listdir( searchDirPath )
         if filename.startswith( Archive.archivedArchivePrefix )
      ]

      return sorted( archivedArchivesPaths,
                     key=os.path.getctime )

   @property
   def archivedArchivesPaths( self ):
      """
      List of archived archives absolute paths, sorted by time of last modification
      with the oldest one first.

      Notes
      -----
      Call with access locked
      """

      return self.searchArchivedArchivesAt( self.historyDirPath )

   @property
   def orphanedCoreFileLinksPaths( self ):
      """
      List of orphaned tmpfs core files links paths, .i.e the one pointing to
      a deleted archived core files.

      Notes
      -----
      Call with access locked
      """

      def isCoreFilePattern( filename ):
         return filename.startswith( 'core.' )

      def path( filename ):
         return os.path.join( EOS_CORE_DIR_PATH, filename )

      def isOrphanedLink( path ):
         return not os.path.exists( os.readlink( path ) )

      return [
         path( filename )
         for filename in os.listdir( EOS_CORE_DIR_PATH )
         if ( isCoreFilePattern( filename )
              and os.path.islink( path( filename ) )
              and isOrphanedLink( path( filename ) ) )
      ]

   #################################################################################
   # General status properties
   #################################################################################

   @property
   def _present( self ):
      """True if the base archive directory exists, False otherwise."""

      return os.path.isdir( self.path )

   @property
   def dirsExist( self ):
      """True if the archive directory and common subdirectories exist."""

      return self._present and self._subdirsPresent

   @property
   def _subdirsPresent( self ):
      """True if the common subdirs exist; False otherwise."""

      return all( os.path.isdir( path ) for path in self.subdirsPaths )

   @property
   def _validAccess( self ):
      """True if we have write access over the archive directory; False otherwise."""

      if not self._present or not self._subdirsPresent:
         return False

      for path in [ self.path ] + self.subdirsPaths:
         stat = os.stat( path )
         if ( stat.st_uid != Archive.archiveUsrId()
              or stat.st_gid != Archive.archiveGrpId() ):
            return False
         if oct( stat.st_mode )[ -3 : ][ 0 ] != '7':
            return False

      return True

   @property
   def _quickAccessPresent( self ):
      """
      True if the symbolic link '/archive' exists and points to the archive directory
      """

      return ( os.path.islink( Archive.quickAccesLink )
               and os.path.realpath( Archive.quickAccesLink ) == self.path )

   @property
   @_Decorators.runWithLockedAccess()
   def exists( self ):
      """
      True if all the following are True:
         * The archive directory is present.
         * The target device is mounted.
         * Common archive subdirs exists.
         * archive user have write access to the archive.
         * The /archive symlink exists and point to the archive directory.
      """

      return ( self.dev.mounted
               and self._present
               and self._subdirsPresent
               and self._validAccess
               and self._quickAccessPresent )

   @property
   def valid( self ):
      """True if the archive exists and quota are enabled."""

      if CEosHelper.isCeos() or not self.dev.quotasSupported:
         return self.exists
      return self.exists and self.quotaMount and self.quotasOn

   @property
   def disabled( self ):
      """True if archiving is disabled, False otherwise."""
      if self.dev.forceDisabled or self.quotaPct == 0:
         return True

      if os.path.isfile( Archive.configFilePath ):
         configFile = Archive.configFileInfo()
         if configFile[ 'storageLifetime' ] < DRIVE_LIFETIME_CUTOFF:
            return True
         return not configFile[ 'enabled' ]

      return False

   @property
   def enabled( self ):
      """True if archiving is enabled, False otherwise."""

      return not self.disabled

   @property
   def statusSummary( self ):
      """Dictionary with a snapshot of the current archive state."""

      # Make sure filesystem information are up-to-date.

      info = {
         # General information:
         'name' : self.name,
         'path' : self.path,
         'exists' : self.exists,
         'dirsExist' : self.dirsExist,
         'valid' : self.valid,
         'disabled' : self.disabled,
         'enabled' : self.enabled,
         # Device information
         'devName' : self.dev.name,
         'devPresent' : self.dev.present,
         'devPhysical' : self.dev.physical,
         'devMounted' : self.dev.mounted,
         'devForceDisabled' : self.dev.forceDisabled,
         # Archive space usage information
         'usedKiB' : self.spaceUsageSize,
         'usedPct' : self.spaceUsagePct
      }

      # Filesystem information
      if not info[ 'devMounted' ]:
         keys = { 'fsMntPt', 'fsSize', 'fsUsedKiB', 'fsUsedPct', 'fsQuotaMount' }
         info.update( dict.fromkeys( keys, None ) )
      else:
         allFsInfo = self.fs.fsInfo
         fsInfo = {
            'fsMntPt' : allFsInfo[ 'mntPt' ],
            'fsSize' : allFsInfo[ 'size' ],
            'fsUsedKiB' : allFsInfo[ 'used' ],
            'fsUsedPct' : allFsInfo[ 'usedPct' ],
            'fsQuotaMount' : self.fs.quotaMount
         }
         info.update( fsInfo )

      fileArchivalEnabled = True
      if self.disabled or self.dev.forceDisabled:
         fileArchivalEnabled = False
      info[ 'fileArchivalEnabled' ] = fileArchivalEnabled

      # Quota information
      keys = { 'quotasOn', 'quotasOff', 'quotaKiB', 'quotaPct' }
      if not info[ 'fsQuotaMount' ]:
         info.update( dict.fromkeys( keys, None ) )

         if not self.dev.quotasSupported:
            # Despite quotas not being enforced by the file system, quotas are
            # enforced by the archive scripts. Therefore, populate this
            # information in the status summary.
            info[ 'quotasOn' ] = True
            info[ 'quotasOff' ] = False
            try:
               if CEosHelper.isCeos():
                  info[ 'quotaPct' ] = self.defaultQuotaPct
               else:
                  info[ 'quotaPct' ] = self.quotaPct
               info[ 'quotaKiB' ] = self.fs.sizeFromPct( info[ 'quotaPct' ] )
            except ( SpaceMgmtLib.Quota.QuotaCmdException, AssertionError ):
               pass
      else:
         for key in keys:
            try:
               info[ key ] = getattr( self, key )
            except SpaceMgmtLib.Quota.QuotaCmdException:
               info[ key ] = None

      return info

   #################################################################################
   # Space usage properties
   #################################################################################

   @property
   @_Decorators.runWithLockedAccess()
   def spaceUsageSize( self ):
      """Archive space usage in KiB including archived archives."""

      return SpaceMgmtLib.Utils.spaceUsage( self.baseDirPath )

   @property
   def spaceUsagePct( self ):
      """Archive space usage as a percentage of the filesystem size."""

      return SpaceMgmtLib.Utils.ratioToPct( self.spaceUsageSize, self.fs.size )

   @property
   def defaultRequiredFreeSize( self ):
      """
      Minimum requirement of free space we want available for the archive in KiB.

      Notes
      -----
      This is the minimum size that should be free in general to ensure a sane
      archiving iteration without the need of reclaiming space.
      """

      return min( self.fs.size, Archive.defaultRequiredFreeSizeKiB )

   @property
   def defaultRequiredFreePct( self ):
      """
      Minimum requirement of free space we want available for the archive
      as a percentage of the filesystem size.

      Notes
      -----
      This the percentage equivalent of `defaultRequiredFreeSize`. If it represent
      less that 1 percent of the filesystem size, cap it to 1%.
      """

      return self.fs.pctFromSize( self.defaultRequiredFreeSize ) or 1

   @property
   def requiredFreePct( self ):
      """
      Percentage of the total size of the filesystem that should be free.
      Same as the defaultRequiredFreePct unless the quota limit is smaller.
      """

      quotaPct = self.quotaPct
      if quotaPct is None:
         quotaPct = 100

      return min( self.defaultRequiredFreePct, quotaPct )

   @property
   def neededReclaimSpaceSize( self ):
      """
      Size in KiB that needs to be removed from the archive to meet the space usage
      requirements: free space + used space <= reserved space, i.e. quota limit
      """

      return self.computeReclaimSpaceSize( self.requiredFreePct )

   #################################################################################
   # Quota properties
   #################################################################################

   @property
   def quotasOn( self ):
      """True if user quota are on, False otherwise."""

      if not self.dev.quotasSupported:
         return False
      return self.fs.quotasOn

   @property
   def quotasOff( self ):
      """True if user quotas are off, False otherwise."""

      if not self.dev.quotasSupported:
         return True
      return self.fs.quotasOff

   @property
   def quotaMount( self ):
      """True if the filesystem is mounted with quota options, False otherwise."""

      if not self.dev.quotasSupported:
         return False
      return self.fs.quotaMount

   @property
   def graceTimes( self ):
      """
      Quotas grace times

      Returns
      -------
      tuple(blkGraceTime, fileGraceTime):
         blkGraceTime: int
            Grace time for block quotas in seconds.
         fileGraceTime: int
            Grace time for file quotas in seconds.
      """

      if not self.dev.quotasSupported:
         return ( 0, 0 )

      try:
         return self.fs.graceTimes
      except SpaceMgmtLib.Quota.QuotaCmdException:
         return None, None

   @property
   @_Decorators.runWithLockedAccess()
   def quotaStats( self ):
      """
      Quota limits and usage statistics for the archive user.

      Returns
      -------
      dict of str: int
         'blkUsed': Number of blocks used, each block is 1KiB.
         'blkSoftLimit': Block quota soft limit in KiB.
         'blkHardLimit': Bock hard limit in KiB.
         'blkGraceLimit': If the user  is in grace period, time in seconds
                          since epoch when his block grace time runs out (or
                          has run out). Is 0 when no grace time is in effect.
         'fileUsed': Number of inode used.
         'fileSoftLimit': Inode quota soft limit (Number of inodes.)
         'fileHardLimit': Inode quota hard limit (Number of inodes.)
         'fileGraceLimit': If the user  is in grace period, time in seconds
                           since epoch when his file grace time runs out (or
                           has run out). Is 0 when no grace time is in effect.
      """

      if self.dev.quotasSupported:
         try:
            return self.fs.quotaStatsPerUser[ Archive.archiveUserName ]
         except ( KeyError, SpaceMgmtLib.Quota.QuotaCmdException ):
            pass

      return {
         'blkUsed' : None,
         'blkSoftLimit' : None,
         'blkHardLimit' : None,
         'blkGraceLimit' : None,
         'fileUsed' : None,
         'fileSoftLimit' : None,
         'fileHardLimit' : None,
         'fileGraceLimit' : None
      }

   @property
   def quotaKiB( self ):
      """Quota block limit in KiB."""

      return self.quotaStats[ 'blkHardLimit' ]

   @property
   def quotaPct( self ):
      """Quota block limit as a percentage relative to the filesystem size."""

      if not self.dev.quotasSupported:
         return self.quotaPctFileValue

      quotaKiB = self.quotaKiB
      return None if quotaKiB is None else self.fs.pctFromSize( quotaKiB )

   @property
   def quotaPctFileValue( self ):
      """
      Quota percentage from the configuration file.

      Returns
      -------
      int
         Percentage written in the quota pct config file.
      None
         If the config file does not exists.
      """

      if not os.path.isfile( Archive.configFilePath ):
         return None
      return Archive.configFileInfo()[ 'quotaPct' ]

   @property
   def storageLifetime( self ):
      """
      Retries the drive wear from the configuration file.

      Returns
      -------
      int
         Percentage used of drive lifetime. Defaults to 100 if config is not present
      """

      if not os.path.isfile( Archive.configFilePath ):
         return 100
      return Archive.configFileInfo()[ 'storageLifetime' ]

   @property
   def defaultQuotaPct( self ):
      """Default percentage value quota block limit."""

      if self.dev.isSSD:
         return Archive.defaultQuotaPctSdd
      elif self.dev.isFlash:
         return Archive.defaultQuotaPctFlash
      else:
         return Archive.defaultQuotaPctOther

   #################################################################################
   # Setup methods
   #################################################################################

   def createDir( self, path ):
      # Need Tac.run for root privileges
      if not os.path.exists( path ):
         cmd = [ 'mkdir', path ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

      cmd = [ 'chmod', '755', path ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

      # If Aaa is running, using numeric value for uid does not work as expected
      # See BUG252788
      cmd = [
         'chown',
         f'{Archive.archiveUserName}:{Archive.archiveGroupName}',
         path
      ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _createArchiveTree( self ):
      """
      Create the archive directory tree.

      with `self.path` the path of the archive directory, e.g. /mnt/drive/archive,
      the following directories will be created if they don't exist:
         * <self.path>
         * <self.path>/var
         * <self.path>/var/core
         * <self.path>/var/log
         * <self.path>/var/log/agents
      """

      paths = [ self.baseDirPath,
                self.path,
                self.historyDirPath,
                os.path.join( self.path, 'var' ) ]
      paths += self.subdirsPaths

      for path in paths:
         self.createDir( path )

   def _createQuickAccessLink( self ):
      """Create the /archive symbolic link pointing to the archive directory."""

      self._deleteQuickAccessLink()
      # Need Tac.run for root privileges
      cmd = [ 'ln', '-s', self.path, Archive.quickAccesLink ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   @_Decorators.runWithLockedAccess()
   def setupDev( self ):
      """Make sure dev is mounted with quota options."""

      if self.dev.quotasSupported:
         if ( not self.dev.mounted
              or ( not CEosHelper.isCeos() and not self.fs.quotaMount ) ):
            # don't use quota in ceos
            quotaMount = not CEosHelper.isCeos()
            self.dev.remount( quotaMount )

   def setupDirs( self, rotate=False ):
      """
      Make sure archive directories exists.

      Parameters
      ----------
      rotate: bool, optional
         If True, rotate the current archive directory if it exists.
      """

      if rotate:
         self.rotateArchiveDir()

      with self.runWithLockedAccess():
         if not self._validAccess:
            self._createArchiveTree()

         if not self._quickAccessPresent:
            self._createQuickAccessLink()

   def setupMarkerFile( self ):
      """Create the archive marker file."""

      SpaceMgmtLib.Utils.createFile( self.markerFilePath )
      # Need Tac.run for root privileges
      cmd = [
         'chown',
         f'{Archive.archiveUserName}:{Archive.archiveGroupName}',
         self.markerFilePath
      ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def setupQuota( self, quotaPct=None, ignoreConfig=False ):
      """
      Make sure quota is turned on and the limits are set as wanted.

      Parameters
      ----------
      quotaPct: int, optional
         Percentage of filesystem size to use as quota block limit.
         If not given, use the percentage from the quota pct configuration file.
         Use a default quota percentage if the configuration file does not exists.
         The default value depend on the destination, 20% for ssd, 5% for flash or
         10% any other destination.
      ignoreConfig: bool, optional, False by default
         If True, ignore the configuration file when choosing the quota percentage.
      """

      if self.dev.quotasSupported:
         if not self.fs.quotaMount:
            return

         if not self.quotasOn:
            createFiles = not os.path.exists( self.quotaUserFilePath )
            self.quotaCheck( createFiles=createFiles )
            self.enableQuotas()

      if self.graceTimes != ( 0, 0 ):
         self.disableGraceTimes()

      if quotaPct is None:
         if not ignoreConfig:
            quotaPct = self.quotaPctFileValue
         if quotaPct is None:
            if not CEosHelper.isCeos():
               quotaPct = self.defaultQuotaPct

      if self.quotaPct != quotaPct:
         self.setQuotaPct( quotaPct )
      elif self.quotaPctFileValue != quotaPct:
         self.writeQuotaPctFile( quotaPct )

   def setup( self, quotaPct=None, rotate=False, ignoreConfig=False ):
      """
      Setup the archive by creating the archive tree, setting quota limits and
      turning quotas on.

      Parameters
      ----------
      quotaPct: int, optional
         Percentage of filesystem size to use as quota block limit.
         If not given, use the percentage from the quota pct configuration file.
         or a default quota depending on the destination if configuration file
         doest not exists (20% for ssd, 5% for flash or 10% any other destination )
      rotate: bool, optional
         If True, rotate any existing 'archive' directory in 'dirPath'.
      ignoreConfig: bool, optional, False by default
         If True, ignore the configuration file when choosing the quota percentage.
      """

      self.setupDev()
      self.setupDirs( rotate=rotate )
      self.setupMarkerFile()
      # don't use quota in ceos
      if not CEosHelper.isCeos():
         self.setupQuota( quotaPct=quotaPct, ignoreConfig=ignoreConfig )

   def setAsConfig( self ):
      """Set this archive as the configured one."""

      Archive.writeConfig( self.name, self.rootDirPath, self.quotaPct,
                           self.enabled, self.storageLifetime )

   #################################################################################
   # Cleanup methods
   #################################################################################

   def _deleteQuickAccessLink( self ):
      """Delete /archive symbolic link."""

      # Need Tac.run for root privileges
      cmd = [ 'rm', '-f', Archive.quickAccesLink ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _deleteArchiveDir( self ):
      """Delete the archive directory."""

      # Need Tac.run for root privileges
      cmd = [ 'rm', '-rf', self.path ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _deleteArchiveHistoryDir( self ):
      """Delete the archive history directory."""

      # Need Tac.run for root privileges
      cmd = [ 'rm', '-rf', self.historyDirPath ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _deleteArchiveBaseDir( self ):
      """Delete the archive base directory."""

      # Need Tac.run for root privileges
      cmd = [ 'rm', '-rf', self.baseDirPath ]
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _deleteArchivedArchives( self, maxCount=0 ):
      """
      Delete archived archives in excess to keep a maximum of `maxCount` archived
      archives. Older archived archives will be deleted first.
      """

      for path in self.archivedArchivesPaths[ : -maxCount ]:
         # Need Tac.run for root privileges
         cmd = [ 'rm', '-rf', path ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def _deleteOrphanedCoreFileLinks( self ):
      """Remove any orphaned tmpfs core files."""

      for path in self.orphanedCoreFileLinksPaths:
         # Need Tac.run for root privileges
         cmd = [ 'rm', '-rf', path ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def cleanupMarkerFile( self ):
      SpaceMgmtLib.Utils.removeFile( self.markerFilePath )

   @_Decorators.runWithLockedAccess()
   def cleanup( self ):
      """Perform full archive cleanup."""

      self._deleteQuickAccessLink()
      self._deleteArchiveBaseDir()
      self._deleteOrphanedCoreFileLinks()

   def replace( self, src, dst ):
      """
      The LogArchiver operates on the same filesystem. As such, we want move
      operations to just be a metadata change. Unfortunately, shutil.move uses
      os.rename() internally which fails for non empty directories. As a result,
      this triggers a hardcopy. This is not desirable, as it's slow, consumes
      way more space than necessary, and could trigger artificial out of space
      conditions (BUG433495 is an example we got bitten by disk thrashing and out
      of space which ended up being an issue for ASU).

      The "proper" python way to do that would be to use os.replace(), but this
      was added in python 3.3. For now, let's just do a Tac.run( mv ). This uses
      the renameat() linux syscall, which only triggers metadata changes
      """
      Tac.run( [ 'mv', src, dst ], stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def clear( self ):
      """Clear the file archive, one time operation."""

      self.cleanup()
      if self.enabled:
         self.setupDirs()

   #################################################################################
   # General status control methods
   #################################################################################

   def enable( self ):
      """Enable logs archiving."""

      Archive.updateConfig( enabled=True )

   def disable( self ):
      """Disable logs archiving."""

      Archive.updateConfig( enabled=False )

   def rotateArchiveDir( self, maxCount=None ):
      """
      Rotate the current archive directory.

      Notes
      -----
      After rotating the current archive directory, a maximum of `maxCount` archived
      archives will be kept. Older archived archives will be deleted if needed.
      The archive directory is not rebuilt after the rotation.

      Parameters
      ----------
      maxCount: int, optional
         Maximum number of archived archives to keep. If None, the default value
         `Archive.maxArchivedArchivesCount` will be used.
      """

      if not self._present:
         return None

      if maxCount is None:
         maxCount = Archive.maxArchivedArchivesCount

      # Rename current archive dir
      mtimeStr = time.strftime( '%Y-%m-%d-%H:%M:%S',
                                time.localtime( os.stat( self.path ).st_mtime ) )
      filename = f'{Archive.archivedArchivePrefix}{mtimeStr}.dir'

      with self.runWithLockedAccess():
         # Rotate current directory
         self._deleteQuickAccessLink()
         self.replace( self.path, os.path.join( self.historyDirPath, filename ) )
         self._deleteOrphanedCoreFileLinks()
         # Remove oldest archived archives, keeping maxCount archived archives
         self._deleteArchivedArchives( maxCount=maxCount )

      return filename

   #################################################################################
   # Quota control methods
   #################################################################################

   def enableQuotas( self ):
      """Turn quota on."""

      # If quota are already enabled, 'quotaon' will fail with 'Device or resource
      if self.dev.quotasSupported and not self.quotasOn:
         self.fs.enableQuotas()

   def disableQuotas( self ):
      """Turn quota off."""

      if self.dev.quotasSupported:
         self.fs.disableQuotas()

   @_Decorators.runWithQuotaOff
   def quotaCheck( self, createFiles=True, remountRO=False ):
      """
      Run quotacheck on the archive destination filesystem.
      Return False if something wrong happened checking quota, True otherwise.

      If createFiles is true, quotacheck will be run with the --create-files
      option, See 'man quotacheck' for more details.
      If they are on, it will turn off quota before running quotacheck and then
      turned them on again since running quotacheck with quota enabled is not
      advised. See 'man quotacheck' for more details.
      """

      if self.dev.quotasSupported:
         self.fs.quotaCheck( createFiles=createFiles, remountRO=remountRO )

   def disableGraceTimes( self ):
      """Disable quota grace times."""

      if self.dev.quotasSupported:
         self.fs.disableGraceTimes()

   def setQuotaPct( self, quotaPct ):
      """
      Set quota block limits for the archive user to a percentage of the filesystem
      size.

      Parameters
      ----------
      quotaPct: int
         constaints: value in the range [0-100]
         Percentage of the filesystem size the block quota limits should be set to.
      """

      msg = 'Invalid percentage value: %s' % quotaPct
      assert SpaceMgmtLib.Utils.validPct( quotaPct ), msg

      if self.dev.quotasSupported:
         self.fs.setQuotaPct( quotaPct, Archive.archiveUserName )
      self.writeQuotaPctFile( quotaPct )

   def writeQuotaPctFile( self, quotaPct ):
      """Write the current quota percentage to the configuration file."""

      Archive.updateConfig( quotaPct=quotaPct )

   #################################################################################
   # Space management methods
   #################################################################################

   def computeReclaimSpaceSize( self, freePct ):
      """
      Compute the size that needs to be freed to have the given percentage free.

      Parameters
      ----------
      freePct: int
         Percentage of the filesystem we want free. If greater than
         quotaPct, it will be caped to quotaPct.

      Returns
      -------
      int
         Size that need to be freed in KiB.
      """

      quotaPct = self.quotaPct
      if quotaPct is None:
         quotaPct = 100

      return SpaceMgmtLib.computeReclaimSpaceSize( self.fs.size,
                                                   quotaPct,
                                                   freePct,
                                                   self.spaceUsageSize )

   @_Decorators.runWithLockedAccess()
   def reclaimLogSpace( self, maxFilesLog, maxFilesAgent ):
      """
      Reclaim space from the archive directory.

      See SpaceMgmtLib.reclaimSpaceFromLogsAt for details.

      Parameters
      ----------
      maxLogFileCount: int
         constraint: >= 0
         Max number of iterations to keep for a general log file.
      maxAgentLogFileCount: int
         constraint: >= 0
         Max number of iterations to keep for an agent log file.

      Returns
      -------
      tuple( size, fileCount, stats )
         size: int
            Total reclaimed size in KiB.
         fileCount: int
            Total number of files removed.
         stats: dict of str: tuple
            Detailed report of how space was reclaimed.
            key: category name corresponding to one of the specified directories:
               'logs': files removed from `logDirPath`.
               'agents': files removed from `agentLogDirPath`.
               'cores': files removed from `coreDirPath`.
            value: tuple( size, filenames )
               size: int
                  Total reclaimed size for that directory in KiB.
               filenames: list of str
                  List of filenames removed from that directory.
      """

      return SpaceMgmtLib.reclaimSpaceFromLogsAt( self.logDirPath,
                                                  self.agentLogDirPath,
                                                  self.coreDirPath,
                                                  maxFilesLog,
                                                  maxFilesAgent )

   @_Decorators.runWithLockedAccess()
   def reclaimArchivedArchivesSpace( self, reclaimSizeKiB ):
      """
      Reclaim space from archived archives.

      Delete archived archives until reclaiming at least the given size or until
      there is no more archived archives to delete.

      Parameters
      ----------
      reclaimSizeKiB: int
         Minimum size to reclaim in KiB.

      Returns
      -------
      tuple( size, deletedArchives )
         size: int
            Total reclaimed size in KiB.
         deletedArchives: list of str
            Deleted archived archives filenames.
      """

      totalSize = 0
      deletedArchives = []
      archivedArchivesPathIter = iter( self.archivedArchivesPaths )
      oldestArchivePath = next( archivedArchivesPathIter, None )

      while totalSize < reclaimSizeKiB and oldestArchivePath is not None:
         totalSize += SpaceMgmtLib.Utils.removeFileOrDir( oldestArchivePath,
                                                          retSize=True )
         deletedArchives.append( os.path.basename( oldestArchivePath ) )
         oldestArchivePath = next( archivedArchivesPathIter, None )

      return totalSize, deletedArchives

   def reclaimSpace( self, reclaimSizeKiB ):
      """
      Try to reduce the archive space usage by at least the given size.

      Notes
      -----
      First try to reclaim space by deleting archived archives starting by
      the oldest one. Then try to delete older iterations of log files in the
      archive directory.

      Parameters
      ----------
      reclaimSizeKiB: int
         Size to reclaim in KiB.

      Return
      ------
      int
         Total reclaimed size in KiB.
      """

      ( rmArchivedArchivesSz,
        rmArchivedArchives ) = self.reclaimArchivedArchivesSpace( reclaimSizeKiB )

      totalSize = rmArchivedArchivesSz
      totalFileCount = len( rmArchivedArchives )
      totalStats = dict.fromkeys( [ 'logs', 'agents', 'cores' ], ( 0, [] ) )

      maxFilesCount = Archive.maxMaxFilesCount

      while totalSize < reclaimSizeKiB and maxFilesCount >= Archive.minMaxFilesCount:

         rmSz, fileCount, stats = self.reclaimLogSpace( maxFilesCount,
                                                        maxFilesCount )
         # update stats
         totalSize += rmSz
         totalFileCount += fileCount
         for logType, ( size, filenames ) in stats.items():
            # pylint: disable-next=unsubscriptable-object
            totalStats[ logType ] = ( totalStats[ logType ][ 0 ] + size,
                                      # pylint: disable-next=unsubscriptable-object
                                      totalStats[ logType ][ 1 ] + filenames )

         maxFilesCount //= 2

      with self.runWithLockedAccess():
         # Remove broken eos core file symlinks
         self._deleteOrphanedCoreFileLinks()

      # Add archived archives stats
      totalStats[ 'archivedArchives' ] = ( rmArchivedArchivesSz, rmArchivedArchives )

      return totalSize, totalFileCount, totalStats

   #################################################################################
   # Routine checks
   #################################################################################

   def sizeCheck( self ):
      """
      Perform a routine check of archive space usage and reclaim space if needed.

      Archived archive will be the first selected target for deletion and then
      oldest logrotated version of logs and agent logs.
      """

      if not self.dirsExist:
         return None

      return self.reclaimSpace( self.neededReclaimSpaceSize )

   def statusCheck( self ):
      """
      Perform a routine check of the archive status.
      Try to repair anything that looks wrong.
      """

      # If the device is force disabled, we do nothing
      if self.dev.forceDisabled:
         return

      if self.enabled:
         # don't use quota in ceos
         if not CEosHelper.isCeos():
            # Make sure dev is mounted with quota options
            self.setupDev()

            # Try get the current quota, will be None in case of error
            quotaPct = self.quotaPct

            # Make sure quota are working
            self.setupQuota( quotaPct=quotaPct )

            # Retry to get actual quotaPct if the first attempt failed
            if quotaPct is None:
               quotaPct = self.quotaPct

         # Archive directories are missing, rebuild them
         if not self.exists:
            self.setupDirs()

         # Make sure marker file exists
         if not os.path.isfile( self.markerFilePath ):
            self.setupMarkerFile()

      # clear all archive data since the reserved space is 0
      if self.quotaPct == 0:
         self.cleanup()
