# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
from Arnet.IpTestLib import newUdpPkt, UdpHdrSize, ipVersionOf
import six

IbBTHdrSize = 12
IbRethHdrSize = 16

RoceV2UdpPortNum = 4791

def newRoceV2Pkt( srcMacAddr,
                  dstMacAddr,
                  vlanId=None,
                  vlanPriority=None,
                  cfiBit=False,
                  ethType="ethTypeIp",
                  srcIpAddr="10.1.1.1",
                  dstIpAddr="10.2.2.2",
                  version=None,
                  tos=0,
                  ipTotalLen=None,
                  ipId=0,
                  fragmentOffset=0,
                  moreFrag=False,
                  dontFrag=True,
                  ttl=None,
                  ipChecksum=None,
                  routerAlert=False,
                  srcPort=1999,
                  dstPort=RoceV2UdpPortNum,
                  calcUdpCheckSum=False,
                  data=b'',
                  opcode=0x1D, # IB_OPCODE_RC | IB_OPCODE_ATOMIC_WRITE
                  ibFlags=0xcf, # Solicited Event(1b) Mig_Req(1b)
                                # Pad( 2b ) Hdr vers( 4b )
                  ibReserved=0xff,
                  ibAck=1,
                  partitionKey=1000,
                  destQueuePair=1000,
                  ibReserved7=0x7f,
                  ibPktSeqNum=123456,
                  addRethHdr=False,
                  rethVirtualAddr=0x0123456789abcdef,
                  rethRemoteKey=0x12345678,
                  rethWriteLength=0x13579ace,
                  **kwargs ):
   '''Create a packet with ethernet, ip, udp and ib BT header and 
   data.  Return a tuple of the packet, the ethernet header
   wrapper, the dot1Q header wrapper, the ip header wrapper, the
   router alert option wrapper, the udp header wrapper,
   ib BT header wrapper and the current offset into the packet
   where the next header should go.
   '''

   version = ipVersionOf( version, srcIpAddr, dstIpAddr )
   dataLen = len( data )

   udpLen = UdpHdrSize + IbBTHdrSize + dataLen
   if addRethHdr:
      udpLen += IbRethHdrSize

   ( p, ethHdr, qHdr, ipHdr, raOpt, udpHdr, currentOffset ) = \
       newUdpPkt( srcMacAddr=srcMacAddr,
                  dstMacAddr=dstMacAddr,
                  vlanId=vlanId,
                  vlanPriority=vlanPriority,
                  cfiBit=cfiBit,
                  ethType=ethType,
                  srcIpAddr=srcIpAddr,
                  dstIpAddr=dstIpAddr,
                  version=version,
                  ipTotalLen=ipTotalLen,
                  ipId=ipId,
                  fragmentOffset=fragmentOffset,
                  moreFrag=moreFrag,
                  dontFrag=dontFrag,
                  ttl=ttl,
                  tos=tos,
                  ipChecksum=ipChecksum,
                  routerAlert=routerAlert,
                  srcPort=srcPort,
                  dstPort=dstPort,
                  udpLen=udpLen,
                  **kwargs )

   ibBTHdr = Tac.newInstance( "Arnet::IbBTHdrWrapper",
                               p, currentOffset )
   ibBTHdr.opcode = opcode
   ibBTHdr.flags = ibFlags
   ibBTHdr.reserved = ibReserved
   ibBTHdr.ack = ibAck
   ibBTHdr.partitionKey = partitionKey
   ibBTHdr.queuePair = destQueuePair
   ibBTHdr.reserved7 = ibReserved7
   ibBTHdr.psn = ibPktSeqNum
   currentOffset += IbBTHdrSize

   ibRethHdr = None
   if addRethHdr:
      ibRethHdr = Tac.newInstance( 'Arnet::IbRethHdrWrapper', p, currentOffset )
      ibRethHdr.virtualAddress = rethVirtualAddr
      ibRethHdr.remoteKey = rethRemoteKey
      ibRethHdr.writeLength = rethWriteLength

      currentOffset += IbRethHdrSize

   # fill in the data
   for num, byte in enumerate( six.iterbytes( data ) ):
      p.rawByte[ currentOffset + num ] = byte

   # re-calculate udp checksum
   udpHdr.checksum = 0
   if calcUdpCheckSum:
      udpHdr.checksum = udpHdr.computedChecksum
   return ( p, ethHdr, qHdr, ipHdr, raOpt, udpHdr, ibBTHdr, ibRethHdr,
            currentOffset )
