# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import socket

import ManagedSubprocess
import Tac
import Tracing
import Netns
import os

t0 = Tracing.trace0

NamespaceType = Netns.NamespaceType

def getErrno():
   return Netns.getErrno()

def setns( fd, nsType=NamespaceType.ANY ):
   return Netns.setns( fd, nsType=nsType )

DEFAULT_NS = Tac.Value( "Arnet::NamespaceName" ).defaultNamespace()

def runMaybeInNetNs( netNs, execCmd, deviceNs=None, **kwargs ):
   """This is a helper function which wraps over Tac.run() to
      be able to run a cmd inside of a network namespace using the
      "ip netns exec ..." cmd. If the netNs argument is "" or
      DEFAULT_NS, it simply uses Tac.run(). If not is adds
      asRoot=True and runs under "ip netns exec ...".
      deviceNs is the namespace of the helper devices that has namespace of its
      own under the test namepsace. ( e.g : ribd1, rtr2, ip2 ), the deviceNS is
      not needed if we want to execute a command directly under the test namespace
      ( e.g : Artest-1234 )
      Rest of the kwargs are directly passed into Tac.run"""
   # For non-default netNs, if asRoot is not passed, add it in
   asRoot = False

   # Since we're running a short-lived command, we don't need to use closeFds.
   # The problem with Popen( close_fds=True ) is that it will call close() on
   # all FD numbers from 1 to MAXFD (1M).  Do not override it if it has been
   # explicitly specified, though.
   if 'closeFds' not in kwargs:
      kwargs[ 'closeFds' ] = False

   shell = kwargs.get( 'shell', False )

   cmd = list() # pylint: disable=use-list-literal
   if deviceNs:
      cmd = [ "netns", deviceNs ]
   if not netNs or netNs == DEFAULT_NS:
      cmd.extend( execCmd )
   else:
      assert( isinstance( execCmd, list ) ) # pylint: disable=superfluous-parens
      if 'asRoot' not in kwargs:
         asRoot = True
      cmd.extend( [ "ip", "netns", "exec", netNs ] )
      if shell:
         # pylint: disable-next=consider-using-f-string
         cmd.append( "sh -c '%s'" % " ".join( execCmd ) )
      else:
         cmd.extend( execCmd )

   if shell:
      cmd = [ " ".join( cmd ) ]

   t0( "Running", cmd )
   if asRoot:
      return Tac.run( cmd, asRoot=True, **kwargs )
   else:
      return Tac.run( cmd, **kwargs )

# pylintish: We have to "redefine" the variable "type" because we are
# matching the socket.socket API, though pylint shouldn't complain
# about this use, as we don't actually redefine the built-in "type" in
# the method.

# pylint: disable-msg=W0622
def socketAt( family=socket.AF_INET, type=socket.SOCK_STREAM, proto=0, ns=None ):
   """A socket.socket replacement that creates a socket in a network namespace.

   Args:
      ns: str. The name of the namespace to create the socket in.
      Network access is scoped to the VRF using the namespace.

   Returns:
      A socket object, the same as the result of socket.socket().

   Raises:
      socket.error: A client socket error occured (e.g., permission denied)
   """
   helper = Tac.newInstance( "Arnet::Netns", "dummy" )
   if not ns or ns == DEFAULT_NS:
      ns = 'default'
   sd = helper.netNsSocketAt( int( family ), int( type ), int( proto ), ns )
   sock = socket.socket( fileno=sd )
   return sock

def openAt( path, flags, ns ):
   helper = Tac.newInstance( "Arnet::Netns", "dummy" )
   if not ns or ns == DEFAULT_NS:
      ns = 'default'
   helper.ns = ns
   open( path ) # pylint: disable=consider-using-with
   fd = helper.netNsOpenAt( path, flags, ns )
   return os.fdopen( fd )

def popenMaybeInNetNs( netNs, execCmd, stdin=None, stdout=None, stderr=None,
                       **kwargs ):
   """This is a helper function which wraps over Popen() to be
      able to run a cmd inside of a network namespace. If netNs
      is a non-default namespace, the command is run using the
      "ip netns exec ..." cmd. If the netNs is "" or DEFAULT_NS,
      cmd is run unmodified. However, in both cases cmd is run
      as a daemon process using ManagedSubprocess.Popen()."""
   if not netNs or netNs == DEFAULT_NS:
      cmd = execCmd
   else:
      cmd = [ "ip", "netns", "exec", netNs ]
      cmd.extend( execCmd )

   t0( "Running", cmd )
   return ManagedSubprocess.Popen( cmd, stdin=stdin, stdout=stdout, stderr=stderr,
                                   **kwargs )

