# -*- mode: python -*-
# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

"""Arastra networking utilities.

This module contains utilities relating to networking, in particular, for
manipulating IP addresses.
"""

# pkgdeps: specline %{python3_sitelib}/Arnet/__init__.py

import collections
import functools
import random
import re
import socket
from functools import reduce

import Tac
from ArPyUtils import naturalOrderKey

# Moved some utilities that used to be here to Ark.IpUtils so that ArosTest
# doesn't depend on Arnet.  Not all of these utilities are used within this file,
# but that's fine: we're importing them for compatibility with old code that used
# to import them from here.
# pylint: disable-msg=W0611
# pylint: disable-msg=W0212
# pylint: disable-msg=R1710
from IpUtils import Mask

# This is required by nextAddress4() and nextAddress6(). These should
# ideally use the Arnet types and not the IpUtils types.
import IpUtils

Ip6AddrRe = r'[A-Fa-f0-9:.]{0,46}'
IpAddrRe = r'(\d+)\.(\d+)\.(\d+)\.(\d+)$'
IpAddrReFull = r'^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}'\
               '(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$'
IpAddrCompiledRe = re.compile( IpAddrRe )
IpAddrFullCompiledRe = re.compile( IpAddrReFull )
Ip6AddrCompiledRe = re.compile( Ip6AddrRe )

IpAddrWithFullMaskCompiledRe = \
   re.compile( r"(\d+\.\d+\.\d+\.\d+)(?:/(\d+\.\d+\.\d+\.\d+))?$" )
IpAddrWithMaskCompiledRe = re.compile( r"(\d+\.\d+\.\d+\.\d+)(/\d+)?$" )
Ip6AddrWithMaskCompiledRe = re.compile( r"^([A-Fa-f0-9:.]{0,46})(/\d+)?$" )
Ip6AddrWithFullMaskCompiledRe = \
      re.compile( r"^([A-Fa-f0-9:.]{0,46})(/([A-Fa-f0-9:.]{0,46}))?$" )

mcastGroupCompiledRe = re.compile( r'([0-9]+).([0-9]+).([0-9]+).([0-9]+)' )

def ip6MaskFromPrefix( prefixLength ):
   binaryString = "1" * prefixLength + "0" * ( 128 - prefixLength )
   hexString = format( int( binaryString, 2 ), '032x' )
   mask = ':'.join( re.findall( '.{4}', hexString ) )
   return mask

def intFromMask( sMask ):
   """
   Attempts to create a 32 bit integer from a string representing an ipv4 mask.

   s_mask will be converted to an int if it is in dotted-decimal or prefix format.
   """
   try:
      # mask is a prefix length, or a number that looks like one.
      mLen = int( sMask )
      return Mask._ipPrefixLenToMask[ mLen ]
   except ValueError:
      # mask is full, or some unsupported format.
      m = IpAddrCompiledRe.match( sMask )
      if m:
         return ( 0xffffffff &
                 reduce( lambda a, b: a | b,
                        [ int( m.groups()[ i ] ) << ( 24 - 8 * i )
                          for i in range( 4 ) ] ) )
      return None

def intFromPrefix( prefix ):
   """
   Variation of intFromMask. Attempts to create a 64 bits integer from
   an ipv4 prefix.
   """
   address = intFromMask( prefix.address )
   return ( ( address << 8 ) + int( prefix.len ) ) if address is not None else None

def numFromAddr( addr ):
   """
   Convert an instance of Arnet::IpAddr or Arnet::Ip6Addr into a
   numeric type (int or long as necessary).

   As noted in BUG33844, using this conversion may be a sign that we
   need more functionality in the C++ layer for low-level address /
   mask hacking so that we can avoid implementing the same
   functionality in C++ and python.
   """
   if isinstance( addr, Tac.Type( 'Arnet::IpAddr' ) ):
      return addr.value
   elif isinstance( addr, Tac.Type( 'Arnet::Ip6Addr' ) ):
      return ( addr.word0 << 96 ) | ( addr.word1 << 64 ) \
          | ( addr.word2 << 32 ) | addr.word3
   else:
      raise ValueError()

def isContiguousMask( iMask ):
   """
   Determines whether or not the given 32 bit integer may represent 
   a contiguous ipv4 netmask.
   """
   return iMask == 0 or ~iMask == 0 or \
       not ( ( ( ~iMask >> 1 ) ^ ~iMask ) >> 1 ) + iMask == 0

def AddrWithFullMask( sAddr, iMask=0 ):
   """
   Constructs a Tac.Value representing an Arnet::IpAddrWithFullMask.

   s_addr should be a string represetnign an ipv4 address in dotted decimal format.

   i_mask should be the integer form of the mask as it will be applied; 
   i.e., should not be a wildcard mask.

   If neither a mask nor a prefix length is specified, the address is taken
   to be a host address and the mask defaults to 0.0.0.0.
   """
   if isinstance( sAddr, Tac.Type( 'Arnet::IpAddrWithFullMask' ) ) and iMask == 0:
      return sAddr
   sAddr = str( sAddr )

   m = IpAddrWithMaskCompiledRe.match( sAddr )
   if m:
      if iMask == 0:
         sAddr = m.group( 1 )
         if m.group( 2 ):
            iMask = Mask._ipPrefixLenToMask[ int( m.group( 2 )[ 1 : ] ) ]
      return Tac.ValueConst( "Arnet::IpAddrWithFullMask",
                             address=sAddr,
                             mask=( 0xffffffff & iMask ) )
   raise ValueError(
      "Bad Address or Mask, addr=%s, mask=%s" % ( sAddr, iMask ) )

def AddrWithMask( s ):
   """Constructs a Tac.Value representing an Arnet::AddrWithMask from
   a string (in the format '1.2.3.4' or '1.2.3.4/16')."""

   if isinstance( s, Tac.Type( 'Arnet::IpAddrWithMask' ) ):
      return s
   s = str( s )

   m = IpAddrWithMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         maskLen = int( m.group( 2 )[ 1 : ] )
      else:
         maskLen = 32
      return Tac.ValueConst(
         "Arnet::IpAddrWithMask", address=m.group( 1 ), len=maskLen )

   raise ValueError( "Expected prefix in addr/len format; got '" + s + "'" )

def IpAddr( s ):
   if isinstance( s, Tac.Type( 'Arnet::IpAddr' ) ):
      return s
   return Tac.ValueConst( "Arnet::IpAddr", stringValue=str( s ) )

def Ip6Addr( s ):
   if isinstance( s, Tac.Type( 'Arnet::Ip6Addr' ) ):
      return s
   return Tac.ValueConst( "Arnet::Ip6Addr", stringValue=str( s ) )

def IntfId( s ):
   if isinstance( s, Tac.Type( 'Arnet::IntfId' ) ):
      return s
   return Tac.ValueConst( "Arnet::IntfId", stringValue=str( s ) )

def Ip6AddrWithMask( s, mask=128 ):
   """Constructs a Tac.Value representing an Arnet::Ip6AddrWithMask from
   a string (in the format 'A:B:C:D:E:F:G:H' or ''A:B:C:D:E:F:G:H/64')."""

   if isinstance( s, Tac.Type( 'Arnet::Ip6AddrWithMask' ) ):
      return s
   s = str( s )

   m = Ip6AddrWithMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         maskLen = int( m.group( 2 )[ 1 : ] )
      else:
         maskLen = mask
      return Tac.ValueConst( "Arnet::Ip6AddrWithMask",
                             address=Ip6Addr( m.group( 1 ) ),
                             len=maskLen )

   raise ValueError( "Expected prefix in addr/len format; got '" + s + "'" )

def Ip6AddrWithFullMask( s, mask="FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF" ):
   """Constructs a Tac.Value representing an Arnet::Ip6AddrWithFullMask from
   a string (with format 'A:B:C:D:E:F:G:H' or 'A:B:C:D:E:F:G:H/A:B:C:D:E:F:G:H')."""

   if isinstance( s, Tac.Type( 'Arnet::Ip6AddrWithFullMask' ) ):
      return s
   s = str( s )
   m = Ip6AddrWithFullMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         mask = m.group( 2 )[ 1 : ]
      else:
         mask = mask # pylint: disable=self-assigning-variable
      return Tac.ValueConst( "Arnet::Ip6AddrWithFullMask",
                              address=Ip6Addr( m.group( 1 ) ),
                              mask=Ip6Addr( mask ) )

   raise ValueError( "Expected mask in addr/mask format; got '" + s + "'" )

def Prefix( s ):
   if isinstance( s, Tac.Type( 'Arnet::Prefix' ) ):
      return s

   addrWithMask = AddrWithMask( s )

   if not addrWithMask.validAsPrefix:
      raise ValueError( "'%s' is not a valid prefix" % s )

   return Tac.ValueConst( "Arnet::Prefix",
                          address=addrWithMask.address,
                          len=addrWithMask.len )

def Ip6Prefix( s ):
   if isinstance( s, Tac.Type( 'Arnet::Ip6Prefix' ) ):
      return s

   addrWithMask = Ip6AddrWithMask( s )

   if not addrWithMask.validAsPrefix:
      raise ValueError( "'%s' is not a valid prefix" % s )

   return Tac.ValueConst( "Arnet::Ip6Prefix",
                          address=addrWithMask.address,
                          len=addrWithMask.len )

def EthAddr( ethAddrStr ):
   ethAddrObj = Tac.Value( "Arnet::EthAddr" )
   ethAddrObj.stringValue = ethAddrStr
   return ethAddrObj

def uint64FromPrefix6( prefix ):
   """
   Attempts to create a 64 bits integer from a string representing an ipv6 prefix. 
   It returns None if the string is
   not in the correct format. 
   """
   try:
      ip = Ip6AddrWithMask( prefix )
   except ValueError:
      return None
   address = ( ip.address.word0 << 32 ) | ip.address.word1
   return ( address ) + int( ip.len )

def IpAddress( rep, addrFamily=socket.AF_INET ):
   '''Construct an IP address (v4 or v6) TAC object from one of
   several input formats.

   This can convert from string, int, or long. Passing in an object
   that is already an instance of Arnet::IpAddr or Arnet::Ip6Addr is
   also supported.
   '''

   if addrFamily == socket.AF_INET:
      typeName = "Arnet::IpAddr"
   elif addrFamily == socket.AF_INET6:
      typeName = "Arnet::Ip6Addr"
   else:
      raise ValueError()

   try:
      if isinstance( rep, str ):
         return Tac.ValueConst( typeName,
                                stringValue=rep )
      elif isinstance( rep, int ):
         if addrFamily == socket.AF_INET:
            return Tac.ValueConst( typeName,
                                   value=rep )
         else:
            if ( rep ).bit_length() <= 32:
               return Tac.ValueConst( typeName,
                                      word3=rep )
            else:
               word3 = rep & 0xffffffff
               word2 = ( rep >> 32 ) & 0xffffffff
               word1 = ( rep >> 64 ) & 0xffffffff
               word0 = ( rep >> 96 ) & 0xffffffff
               return Tac.ValueConst( typeName,
                                      word0=word0, word1=word1,
                                      word2=word2, word3=word3 )
      elif isinstance( rep, Tac.Type( typeName ) ):
         # BUG31255 Allowing construction from an instance that is
         # already a TAC object is deprecated
         #
         # Round-tripping via stringValue is necessary to work around
         # a shortcoming in pylint; if we return rep itself, it
         # believes that we could be returning an integer and throws
         # bogus erros.
         return Tac.ValueConst( typeName, stringValue=rep.stringValue )
      else:
         raise TypeError()
   except IndexError:
      # pylint: disable-next=raise-missing-from
      raise ValueError( "'%s' is not a valid IP address" % rep )

_intfNameKeyCache = collections.defaultdict( list )

def intfNameKey( item ):
   # We have a defaultdict.
   # If the key does not exist, it will create a list,
   # insert it in the cache, and return it.
   result = _intfNameKeyCache[ item ]
   if not result:
      # The list is empty, so we need to cache the result of `naturalOrderKey`.
      # NB: id( result ) == id( cache[ item ] ),
      # which means we're changing that same list.
      result += naturalOrderKey( item )
   return result

# Convenience API for sorting interfaces.
sortIntf = functools.partial( sorted, key=intfNameKey )

class Interface:
   '''Some code would use the `intfNameKey` not for sorting,
   but to use the data in some other way.
   This no longer works, so we need another object with the appropriate attrs.
   '''
   __slots__ = ( 'basename', 'stack', 'mod', 'port', 'sub' )
   intfNameRe = re.compile( r'(?P<basename>[^0-9/]+)'
                            r'(?:(?P<stack>\d+)/?)?'
                            r'(?:(?P<mod>\d+)/?)?'
                            r'(?:(?P<port>\d+))?'
                            r'(?:\.(?P<sub>\d+))?$' )

   def __init__( self, intfString ):
      def value( group ):
         return int( m.group( group ) or '0' )
      m = self.intfNameRe.match( intfString )
      self.basename = m.group( 'basename' )
      self.stack = value( 'stack' )
      self.mod = value( 'mod' )
      self.port = value( 'port' )
      self.sub = value( 'sub' )

def getMcastGroupAddresses( startAddr, numGroups ):
   groups = [ ]
   m = mcastGroupCompiledRe.match( startAddr )
   addr = ( int( m.group( 1 ) ) << 24 ) + \
          ( int( m.group( 2 ) ) << 16 ) + \
          ( int( m.group( 3 ) ) << 8 ) + \
          int( m.group( 4 ) )
   for _ in range( numGroups ):
      groups.append( "%s.%s.%s.%s" % ( ( addr >> 24 ) & 0xFF,
                                       ( addr >> 16 ) & 0xFF,
                                       ( addr >> 8 ) & 0xFF,
                                       ( addr & 0xFF ) ) )
      addr = addr + 1
   return groups

def prefixStrToObject( prefixes, addrFamily ):
   '''Concerts a given prefix or a set of prefixes to list of AddrWithMask or
   Ip6AddrWithMask objects'''
   if isinstance( prefixes, str ):
      prefixes = [ prefixes ]
   if addrFamily == socket.AF_INET:
      return [ Tac.const( AddrWithMask( p ) ) for p in prefixes ]
   else:
      return [ Tac.const( Ip6AddrWithMask( p ) ) for p in prefixes ]

class Subnet:
   """Represents an IP subnet (v4/v6), defined by an IP address and a subnet mask.  
   Subnet objects are immutable."""

   def __init__( self, a, b=None, addrFamily=socket.AF_INET ):
      """Constructs a Subnet object from a string 
      (in the format '1.2.3.4/16' for v4 and 1:2:3::4/64 for v6), a
      dict containing 'address' and 'len' items, an object with 'address' and 'len'
      attributes, or a pair of objects from which an IpAddress and Mask object,
      repsectively, can be constructed."""
      if addrFamily not in [ socket.AF_INET, socket.AF_INET6 ]:
         errMsg = "addrFamily must be either AF_INET or AF_INET6"
         raise ValueError( "%s: %s" % ( errMsg, addrFamily ) )

      self.addrFamily_ = addrFamily
      if b is None:
         # We got one parameter, which contains both an IP address and a mask
         if isinstance( a, str ):
            [ addr, mask ] = re.split( '/', a )
            self.address_ = IpAddress( addr, addrFamily=addrFamily )
            self.mask_ = Mask( int( mask ), addrFamily=addrFamily )
         elif isinstance( a, dict ):
            if isinstance( a[ 'address' ], str ):
               self.address_ = IpAddress( a[ 'address' ], addrFamily=addrFamily )
            else:
               self.address_ = a[ 'address' ]
            self.mask_ = Mask( a[ 'len' ], addrFamily=addrFamily )
         else:
            self.address_ = IpAddress( a.address, addrFamily=addrFamily )
            self.mask_ = Mask( a.len, addrFamily=addrFamily )
      else:
         # We got two parameters, a separate IP address and mask.
         self.address_ = IpAddress( a, addrFamily=addrFamily )
         self.mask_ = Mask( b, addrFamily=addrFamily )

   def __str__( self ):
      """Returns the subnet as a string in the format '1.2.3.4/16' for v4 
      and 1:2:3::4/64 for v6) """
      return "%s/%d" % ( self.address_.stringValue, self.mask_.masklen_ )

   def _addrWithMask( self ):
      """This is a transitional helper method while refactoring the
      code to use TAC objects rather than Python. It generates an
      Arnet::AddrWithMask or Arnet::Ip6AddrWithMask. In the longer
      term, this Python class should become a thin wrapper around an
      instance of one of these types
      """
      if self.addrFamily_ == socket.AF_INET:
         return AddrWithMask( '%s/%s'
                              % ( self.address_, self.mask_.masklen_ ) )
      elif self.addrFamily_ == socket.AF_INET6:
         return Ip6AddrWithMask( '%s/%s'
                                 % ( self.address_, self.mask_.masklen_ ) )

   def toNum( self ):
      """Returns the subnet address as a non-negative int or long in host byte order.
      The subnet address of the subnet 1.2.3.4/16, for example, is 1.2.0.0."""
      if self.addrFamily_ == socket.AF_INET:
         return self.address_.value & self.mask_.toNum()
      elif self.addrFamily_ == socket.AF_INET6:
         return ( ( self.address_.word0 << 96 )
                  + ( self.address_.word1 << 64 )
                  + ( self.address_.word2 << 32 )
                  + self.address_.word3 ) & self.mask_.toNum()

   def toValue( self ):
      """Returns the subnet as a Tac.Value representing an Arnet::Prefix.
      The host bits in the network address are zeroed out by applying the mask. """
      if self.addrFamily_ == socket.AF_INET:
         return Prefix( self._addrWithMask().subnet.stringValue )
      elif self.addrFamily_ == socket.AF_INET6:
         return Ip6Prefix( self._addrWithMask().subnet.stringValue )

   def containsAddr( self, addr ):
      """Returns whether or not an IP address is contained within this subnet."""
      if ( isinstance( addr, Tac.Type( 'Arnet::IpAddr' ) ) and
             self.addrFamily_ != socket.AF_INET ):
         return False
      elif ( isinstance( addr, Tac.Type( 'Arnet::Ip6Addr' ) ) and
            self.addrFamily_ != socket.AF_INET6 ):
         return False

      if isinstance( addr, ( str, int ) ):
         addr = IpAddress( addr, addrFamily=self.addrFamily_ )

      return self._addrWithMask().contains( addr )

   def numHostInSubnet( self ):
      """Returns the number of IP addresses that are contained within this subnet."""
      return self.mask_.numHostInSubnet()

   def overlapsWith( self, otherSubnet ):
      """Returns True if one subnet is a subset of the other."""
      if self.addrFamily_ == socket.AF_INET:
         smallerMask = Mask._ipPrefixLenToMask[ min( self.mask_.masklen_,
                                                     otherSubnet.mask_.masklen_ ) ]
         return self.address_.value & smallerMask == \
             otherSubnet.address_.value & smallerMask
      else:
         return self._addrWithMask().overlaps( otherSubnet._addrWithMask() )

   def isAllZerosAddress( self ):
      if self.addrFamily_ == socket.AF_INET:
         if self.mask_.masklen_ >= 31:
            # ignore special cases.
            return False
         mask = Mask._ipPrefixLenToMask[ self.mask_.masklen_ ]
         return ( ~mask & self.address_.value ) == 0
      else:
         mask = Mask._ip6PrefixLenToMask[ self.mask_.masklen_ ]
         return ( ~mask & self.address_.value ) == 0

   def isBroadcastAddress( self ):
      if self.addrFamily_ == socket.AF_INET:
         if self.mask_.masklen_ >= 31:
            # There is no broadcast address on a /31 or /32.
            return False
         addr = self.address_.value
         mask = Mask._ipPrefixLenToMask[ self.mask_.masklen_ ]
         return ( mask | addr ) == 0xffffffff
      else:
         return False

def CompactIpPrefix( p ):
   if not isinstance( p, str ):
      p = p.stringValue
   return Tac.ValueConst( 'Arnet::CompactIpPrefix', IpGenPrefix( p ) )

def IpGenPrefix( s ):
   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenPrefix", s )
   except AttributeError:
      pass
   except ValueError:
      pass
   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenPrefix", s )
   except AttributeError:
      # pylint: disable-next=raise-missing-from
      raise ValueError( "'%s' is not a valid prefix" % s )

def IpGenAddrWithMask( s ):
   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithMask", s )
   except AttributeError:
      pass
   except ValueError:
      pass
   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithMask", s )
   except AttributeError:
      # pylint: disable-next=raise-missing-from
      raise ValueError( "'%s' is not a valid address" % s )

def IpGenAddrWithFullMask( s ):
   if s == '':
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", "" )
   try:
      m = IpAddrWithFullMaskCompiledRe.match( s )
      _addr = IpAddr( m.group( 1 ) )
      if m.group( 2 ):
         _mask = IpAddr( m.group( 2 ) )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError ):
      pass

   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, ValueError, IndexError ):
      pass

   try:
      _addrWithMask = Ip6AddrWithFullMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError ):
      pass

   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError, ValueError ):
      # pylint: disable-next=raise-missing-from
      raise ValueError( "'%s' is not a valid address" % s )

def IpGenAddr( s=None ):
   # Put no value in arg if we want a default Ip address value assigned by Tac
   if s is None:
      return Tac.ValueConst( "Arnet::IpGenAddr" )
   # No check for IpAddr since that is treated as a string anyway
   if isinstance( s, Tac.Type( 'Arnet::Ip6Addr' ) ):
      genAddr = Tac.Value( "Arnet::IpGenAddr" )
      genAddr.handleV6Addr( s )
      return Tac.const( genAddr )
   ipv6 = re.search( ":", s )
   if ipv6:
      _addr = Ip6Addr( s )
      return Tac.ValueConst( "Arnet::IpGenAddr", stringValue=s )
   else:
      try:
         _addr = IpAddress( s )
         return Tac.ValueConst( "Arnet::IpGenAddr", stringValue=s )
      except AttributeError:
         # pylint: disable-next=raise-missing-from
         raise ValueError( "'%s' is not a valid address" % s )

def nextAddress4( addr, maskLen ):
   '''Get the next incremental IP addr given a prefix addr and mask len.
   For example, the next addr of 10.0.0.1/32 is 10.0.0.2/32, the next
   addr of 2.2.2.2/24 is 2.2.3.2/24. Exclude common broadcast addresses,
   but not other well-known special addresses (yet).'''
   step = 2 ** ( 32 - maskLen )
   if isinstance( addr, str ):
      addr = IpUtils.IpAddress( addr )
   addr = IpUtils.IpAddress( addr.toNum() + step )
   if maskLen == 32:
      if addr.toNum() % 256 == 0:
         addr = IpUtils.IpAddress( addr.toNum() + step )
      elif ( addr.toNum() + 1 ) % 256 == 0:
         addr = IpUtils.IpAddress( addr.toNum() + step + 1 )
   return addr

def nextAddress6( addr, maskLen=128 ):
   '''Get the next incremental IP addr given a prefix addr and mask len.
   For example, the next addr of 100::1/128 is 100::2/128, the next
   addr of 100::2/64  is 100:0:0:1::2/64'''
   if isinstance( addr, str ):
      addr = IpUtils.IpAddress( addr, addrFamily=socket.AF_INET6 )
   step = 2 ** ( 128 - maskLen )
   addr = IpUtils.IpAddress( addr.toNum() + step, addrFamily=socket.AF_INET6 )
   return addr

def nextAddrMask4( addrMask ):
   '''Get next addr/mask. For example 10.0.2.1/24 => 10.0.3.1/24'''
   if isinstance( addrMask, str ):
      addrMask = AddrWithMask( addrMask )
   addr = nextAddress4( addrMask.address, addrMask.len )
   return AddrWithMask( "%s/%s" % ( addr, addrMask.len ) )

def nextAddrMask6( addrMask ):
   if isinstance( addrMask, str ):
      addrMask = Ip6AddrWithMask( addrMask )
   assert addrMask.len == 64 # only do /64 for now
   addr = IpAddress( numFromAddr( addrMask.address ) + 2 ** 64,
                     addrFamily=socket.AF_INET6 )
   return Ip6AddrWithMask( addr, 64 )

def nextAddrInAddrMask4( addrMask ):
   '''
   Get next addr/mask with a subsequent /32 IP address. For example,
   10.0.2.1/24 => 10.0.2.2/24. Raise exception if the next address is out of
   current subnet.
   '''
   if isinstance( addrMask, str ):
      addrMask = AddrWithMask( addrMask )
   addr = nextAddress4( addrMask.address, 32 )
   if not addrMask.contains( str( addr ) ):
      raise Exception( "Next address out of subnet range %s" % addrMask )
   return AddrWithMask( "%s/%s" % ( addr, addrMask.len ) )

def nextAddrInAddrMask6( addrMask ):
   '''
   Return next address as a Ip6Addr value in /64 network of addrMask. No
   checking of boundary. Hopefully enough addresses.
   '''
   if isinstance( addrMask, str ):
      addrMask = Ip6AddrWithMask( addrMask )
   assert addrMask.len == 64
   addr = IpUtils.IpAddress( numFromAddr( addrMask.address ) + 1,
                             addrFamily=socket.AF_INET6 )
   return Ip6Addr( str( addr ) )

def randomEthAddr():
   '''Return a random unicast ethernet address.'''
   return max( b'\0' + random.randbytes( 5 ), b'\0\0\0\0\0\1' ).hex( ':' )

def arnetPkt( bytesValue ):
   pkt = Tac.newInstance( 'Arnet::Pkt' )
   pkt.bytesValue = bytesValue
   return pkt
