# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliParser import safeInt

U32_MAX_VALUE =  0xFFFFFFFF
U16_MAX_VALUE =  0xFFFF

def asnStrToNum( asdotStr, minValue=1, maxValue=U32_MAX_VALUE ):
   """This function converts an AS number in asdot string format into
   U32. if the AS number string does not have '.' it converts 
   the string into a U32 number using safeInt.
   Example: 
   AS number in asdot String is '65535.65535' returned as 4294967295.
   AS number in asplain String is '65535' returned as 65535.
   """
   if asdotStr.count( '.' ) == 1:
      spl = asdotStr.split( '.' )
      highU16 =  safeInt( spl[ 0 ] )
      lowU16 = safeInt( spl[ 1 ] )
      if highU16 is None or lowU16 is None:
         return None
      if ( highU16 < 1 or highU16 > U16_MAX_VALUE or
           lowU16 < 0 or lowU16 > U16_MAX_VALUE ):
         return None
      asNumber = highU16 << 16 | lowU16 
   else:
      asNumber = safeInt( asdotStr )
      if asNumber is None:
         return None
   if minValue <= asNumber <= maxValue:
      return asNumber 
   return None

def bgpFormatAsn( asn, asdotConfigured=False ):
   if asn > U16_MAX_VALUE and asdotConfigured:
      # pylint: disable-next=consider-using-f-string
      return "%d.%d" % ( asn >> 16, asn & 0x0000FFFF )
   else:
      return str(asn)

def formatRd( rd, asdotConfigured ):
   """RD is represented as <admin>:<local number>, where
   <admin> can either be an ASN or IP address
   In case <admin> field is a 4 byte ASN, it can either be in
   Asplain or Asdot format.
   If @asdotConfigured is True then convert ASN in Asdot
   notation (if not already) else convert it to Asplain 
   notation (if not already) and return the modified RD.
   """
   if not rd:
      return None
   rdSplit = rd.split( ':' )
   admin = rdSplit[ 0 ]
   if asdotConfigured:
      if '.' not in admin:
         # admin has ASN in Asplain format
         admin = bgpFormatAsn( int( admin ), asdotConfigured )
      else:
         # Already in Asdot format or admin is an ip-address
         return rd
   else:
      if '.' not in admin or len( admin.split( '.' ) ) == 4:
         # admin has ASN in Asplain format or has ip-address
         return rd
      else:
         admin = str( asnStrToNum( admin ) )
   return admin + ':' + rdSplit[ 1 ]

