# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module defines a set of functions that can be used by clients of
# AsicUtils to explore generated *MetaInfoFieldDefinitions_out.py files.

def haveFieldDef( fieldDefModule, chipType, fieldName ):
   """
   True if a given field is defined in the given AsicUtils-generated
   field definitions Python module.
   """
   # Get a list of all the register and format definitions in the module.
   # This is equivalent to dir( fieldDefModule ), but without any entries
   # that start with a double-underscore (__), such as __doc__.
   fieldDefList = [ entry for entry in dir( fieldDefModule )
                    if not entry.startswith( "__" ) ]
   return fieldName + "_" + chipType in fieldDefList

def getFieldNameAtOffset( module, chipType, regOrFmtName, offset ):
   """
   Get the name of the field at a given bit offset within a given register or
   format defined by a given FieldDefinitions_out module. chipType is the AsicUtils
   chip type (e.g., "BCM56870_A0"). defName is the name of the register or format
   definition (e.g., "ING_EVENT_DEBUG" or "INGRESS_COUNTER_UPDATE_VECTOR").
   """
   regDef = getattr( module, regOrFmtName + "_" + chipType )
   for fieldName in regDef:
      if fieldName == 'ALL':
         continue
      fieldOffset = regDef[ fieldName ][ "offset" ]
      fieldWidth = regDef[ fieldName ][ "bits" ]
      if fieldOffset <= offset < ( fieldOffset + fieldWidth ):
         if "VECTOR" in fieldName:
            # if vector was not found, maybe field is defined as a single field
            try:
               return getFieldNameAtOffset( module,
                                            chipType,
                                            fieldName,
                                            offset - fieldOffset )
            except AttributeError:
               continue
         else:
            return fieldName
   return 'BIT%d' % offset # pylint: disable=consider-using-f-string

def getNonZeroFields( module, chipType, regOrFmtName, value ):
   """
   Get a list of the names of the fields that have at least one bit set in a
   given register or format defined by a given FieldDefinitions_out module.
   chipType is the AsicUtils chip type (e.g., "BCM56870_A0").  defName is the
   name of the register or format definition (e.g., "ING_EVENT_DEBUG" or
   "INGRESS_COUNTER_UPDATE_VECTOR").  value is the integer value of the register.
   """
   offset = 0
   events = []
   while value != 0:
      if value & 1:
         events.append( getFieldNameAtOffset( module,
                                              chipType,
                                              regOrFmtName,
                                              offset ) )
      value >>= 1
      offset += 1
   return events
