#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Assert: A module to export the helpful unittest.TestCase assertions.

As the unittest.TestCase assertions are contained within the TestCase
class, we need to create a singleton instance of a TestCase (using the
TestCase.id method as the method to initially run, to avoid the
regular side effect of starting the test case.

For information about the available assertion methods, see the Python
unittest documentation at:

   https://docs.python.org/2/library/unittest.html

In particular, the majority of the assertions are listed here:

   https://docs.python.org/2/library/unittest.html#assert-methods

These assertions can then be used in the following way:

---snip---

import Assert

# Start up test framework and populate the 'longString' variable

expected = '''We expect that there will be some long
text appearing in the longString output.'''

Assert.assertMultiLineEqual( expected, longString )
Assert.assertIn( 3, [ 1, 2, 3 ] )

---snip---

These methods are far more useful than Python's built in assert(),
because they print meaningful output (based on the assertion type)
without the programmer otherwise having to provide an assertion
message. This means that objects not immediately on the stack can
easily be tested and useful results appear in test failures.

"""

import math
import unittest

# pylint: disable=unused-import
from AssertSubTest import BaseExceptionGroup, subTest, subTestGroup
# pylint: enable=unused-import

# Binds a test case to the module for access to the assertion methods
# (in Python, this is effectively a singleton, since module scope code
# is only executed once per program no matter how many times the
# module is imported)
__tc = unittest.TestCase( methodName='id' )

def increaseMaxDiffTo( lowerLimit ):
   '''Increases maxDiff value to at least `lowerLimit`

   This value controls how much output (in characters) is produced
   describing tested values in the AssertionError messages.
   `None` or `math.inf` means output size is unlimited.
   Default value is 10000 (ten thousand) characters.

   This function can only increase the maxDiff value.
   If the value is already greater than the specified `lowerLimit`,
   nothing changes. The idea is, each module or test can call this function
   with the smallest value it needs, and the greatest of these will be
   the effective value, satisfying everyone's requirements.
   '''
   # Convert `None` values to `math.inf` before passing to `max`.
   lowerLimit = math.inf if lowerLimit is None else lowerLimit
   maxDiff = math.inf if __tc.maxDiff is None else __tc.maxDiff
   maxDiff = max( maxDiff, lowerLimit )
   # Convert `math.inf` back to `None`s before writing to __tc.maxDiff.
   __tc.maxDiff = None if maxDiff == math.inf else maxDiff

# Set initial maxDiff value to at least 10000 (ten thousand) characters.
increaseMaxDiffTo( lowerLimit=10**4 )

def assertAlmostEqual( *a, **kw ):
   __tc.assertAlmostEqual( *a, **kw )

assertAlmostEquals = assertAlmostEqual

def assertDictContainsSubset( *a, **kw ):
   __tc.assertDictContainsSubset( *a, **kw )

def assertDictEqual( *a, **kw ):
   __tc.assertDictEqual( *a, **kw )

def assertEqual( *a, **kw ):
   __tc.assertEqual( *a, **kw )

assertEquals = assertEqual

def assertFalse( *a, **kw ):
   __tc.assertFalse( *a, **kw )

def assertGreater( *a, **kw ):
   __tc.assertGreater( *a, **kw )

def assertGreaterEqual( *a, **kw ):
   __tc.assertGreaterEqual( *a, **kw )

def assertIn( *a, **kw ):
   __tc.assertIn( *a, **kw )

def assertIs( *a, **kw ):
   __tc.assertIs( *a, **kw )

def assertIsInstance( *a, **kw ):
   __tc.assertIsInstance( *a, **kw )

def assertIsNone( *a, **kw ):
   __tc.assertIsNone( *a, **kw )

def assertIsNot( *a, **kw ):
   __tc.assertIsNot( *a, **kw )

def assertIsNotNone( *a, **kw ):
   __tc.assertIsNotNone( *a, **kw )

def assertCountEqual( *a, **kw ):
   __tc.assertCountEqual( *a, **kw )

# assertItemsEqual renamed to assertCountEqual in Python 3
assertItemsEqual = assertCountEqual

def assertLess( *a, **kw ):
   __tc.assertLess( *a, **kw )

def assertLessEqual( *a, **kw ):
   __tc.assertLessEqual( *a, **kw )

def assertListEqual( *a, **kw ):
   __tc.assertListEqual( *a, **kw )

def assertMultiLineEqual( *a, **kw ):
   __tc.assertMultiLineEqual( *a, **kw )

def assertNotAlmostEqual( *a, **kw ):
   __tc.assertNotAlmostEqual( *a, **kw )

assertNotAlmostEquals = assertNotAlmostEqual

def assertNotEqual( *a, **kw ):
   __tc.assertNotEqual( *a, **kw )

assertNotEquals = assertNotEqual

def assertNotIn( *a, **kw ):
   __tc.assertNotIn( *a, **kw )

def assertNotIsInstance( *a, **kw ):
   __tc.assertNotIsInstance( *a, **kw )

def assertRaises( *a, **kw ):
   return __tc.assertRaises( *a, **kw )

def assertRaisesRegex( *a, **kw ):
   return __tc.assertRaisesRegex( *a, **kw )

def assertRegex( *a, **kw ):
   __tc.assertRegex( *a, **kw )

def assertNotRegex( *a, **kw ):
   __tc.assertNotRegex( *a, **kw )

# assert(Not)RegexpMatches renamed to assert(Not)Regex in Python 3
# See https://docs.python.org/3/library/unittest.html
assertRegexpMatches = assertRegex
assertNotRegexpMatches = assertNotRegex

def assertSequenceEqual( *a, **kw ):
   __tc.assertSequenceEqual( *a, **kw )

def assertSetEqual( *a, **kw ):
   __tc.assertSetEqual( *a, **kw )

def assertTrue( *a, **kw ):
   __tc.assertTrue( *a, **kw )

def assertTupleEqual( *a, **kw ):
   __tc.assertTupleEqual( *a, **kw )

def maxDiffIs( maxDiff ):
   __tc.maxDiff = maxDiff

def fail( *a, **kw ):
   __tc.fail( *a, **kw )
