#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import json
from copy import deepcopy
from configparser import RawConfigParser
from collections import defaultdict

import Tac
import Plugins, Tracing

# pickle uses _pickle in python 3 and is just as fast. Continuing
# to use cPickle/_pickle for speed in python 2 during transition.
from six.moves import cPickle as pickle

t0 = Tracing.t0

asuFS = Tac.Type( "Asu::AsuFS" )
pStoreFilePathOld = asuFS.asuPStoreDirOld
pStoreFilePathDefault = asuFS.asuDataDirDefault
pStoreFileNameJson = asuFS.asuPStoreDefaultFileName
pStoreFileName = asuFS.asuPStoreDefaultFile
asuPStoreEnvironmentVar = asuFS.asuPStoreDirEnv

class AsuPStoreContext:
   def __init__( self, opcode, entityManager, featureEventHandlerRegistry ):
      assert opcode in [ 'Store', 'GetKeys', 'GetSupportedKeys',
                         'CheckHitlessReloadSupported' ]
      self.opcode_ = opcode
      self.entityManager_ = entityManager
      self.featureEventHandlerRegistry_ = featureEventHandlerRegistry
      
   def opcode( self ):
      return self.opcode_

   def entityManager( self ):
      return self.entityManager_

   def registerAsuPStoreEventHandler( self, name, featureEventHandler ):
      assert name not in self.featureEventHandlerRegistry_
      self.featureEventHandlerRegistry_[ name ] = featureEventHandler
   
   def mountsComplete( self, mg, plugin, pluginCallbackFnc ):
      mg.close( blocking=True )
      pluginCallbackFnc()

class PStoreEventHandler:
   '''Abstract base class that handles store/restore state to/from PStore'''
   
   def save( self, pStoreIO ):
      '''Implemented by derived class. Convert Sysdb state to
      key-value pairs and save to pStoreIO.'''
      pass

   def getSupportedKeys( self ):
      '''Implemented by derived class. Generate and return supported
      key list for this feature'''
      return None
      
   def getKeys( self ):
      '''Implemented by derived class. Generate and return key list to
      be saved to PStore'''
      return None

   def hitlessReloadSupported( self ):
      '''Implemented by derived class. Checks for hitless reload support,
      and returns (warningList, blockingList).
      If no warnings or blocking, ( None, None ) or ( [], [] ) is acceptable.
      '''
      warningList = None
      blockingList = None
      return ( warningList, blockingList )

   def getFileName( self ):
      return pStoreFileNameJson

class FeaturePStoreIO:

   def __init__( self, pStoredata ):
      self.pStoredata_ = pStoredata

   def set( self, key, value ):
      self.pStoredata_[ key ] = deepcopy( value )

   def setItems( self, keyValueList ):
      for k, v in keyValueList.items():
         self.set( k, v )

   def getItems( self ):
      return self.pStoredata_

   def delete( self, key ):
      if key in self.pStoredata_:
         del self.pStoredata_[ key ]

def maybeConvertPstoreFileFormat( ):
   pStoreFilePath = os.environ.get( asuPStoreEnvironmentVar, pStoreFilePathOld )
   pStoreFilePathName = pStoreFilePath + '/' + pStoreFileName

   t0( 'Entering maybeConvertPstoreFileFormat' )
   if os.path.exists( pStoreFilePathName ):
      t0( 'Converting pstore file to Json ' )
      doCreateJsonPstoreFile()
   else:
      t0( '%s does not exist.' % pStoreFilePathName )
      return

# This function will be called to convert Pstore from ConfigParser format to Json
# Note: For new release we store Pstore in Json format directly.
def doCreateJsonPstoreFile( ):
   pStoreFilePath = os.environ.get( asuPStoreEnvironmentVar, pStoreFilePathOld )
   pStoreFilePathName = pStoreFilePath + '/' + pStoreFileName
   pStoreFilePathNameJson = pStoreFilePath + '/' + pStoreFileNameJson

   configParser = RawConfigParser()
   configParser.optionxform = str
   configParser.read( pStoreFilePathName )

   featuresInPStore = configParser.sections()

   data = {}
   for _, featureName in enumerate( featuresInPStore ):
      data[ featureName ] = {}
      for _, featureKey in enumerate( configParser.options( featureName ) ):
         valueStr = configParser.get( featureName, featureKey )
         valueStr = valueStr.encode( 'ascii', 'replace' )
         data[ featureName ] [ featureKey ] = pickle.loads( valueStr )
   try:
      with open( pStoreFilePathNameJson, "w" ) as outFile:
         t0( 'Writing json file' )
         json.dump( data, outFile )
   except OSError:
      # No space in flash?
      t0( 'Unable to write' )
      raise
   finally:
      # backup the pstore file
      pStoreFilePathNameBak = pStoreFilePathName + '.bak'
      os.rename( pStoreFilePathName, pStoreFilePathNameBak )

def doSaveAsuState( entityManager, plugins=None, pluginPath=None ):
   '''Client api to trigger features save ASU state to persistent storage.'''
   assert entityManager
   featureEventHandlerRegistry = {}
   # the expected structure of pStoredata, the fileName get from getFileName() method
   # {
   # fileName1 :
   #    { featureName1 : featureData1,
   #      featureName2 : featureData2,
   #      ... },
   # fileName2 :
   #    { featureName3 : featureData3,
   #      featureName4 : featureData4,
   #      ... },
   # ...
   # }
   pStoredata = defaultdict( dict )
   ctx = AsuPStoreContext( 'Store', entityManager, featureEventHandlerRegistry )
   Plugins.loadPlugins( "AsuPStorePlugin", ctx,
                        plugins=plugins, pluginPath=pluginPath )

   for fname, featureEventHandler in featureEventHandlerRegistry.items():
      pStoredata[ featureEventHandler.getFileName() ][ fname ] = {}
      pStoreIO = FeaturePStoreIO(
                    pStoredata[ featureEventHandler.getFileName() ][ fname ] )
      featureEventHandler.save( pStoreIO )

   pStoreFilePath = os.environ.get( asuPStoreEnvironmentVar, pStoreFilePathDefault )

   # Alway remove the existing store files before reload
   for fileName, data in pStoredata.items():
      pStoreFilePathNameJson = pStoreFilePath + '/' + fileName
      if os.path.exists( pStoreFilePath + fileName ):
         os.unlink( pStoreFilePath + fileName )

      try:
         os.makedirs( pStoreFilePath, exist_ok=True )
         with open( pStoreFilePathNameJson, "w" ) as outFile:
            t0( 'Writing json file' )
            json.dump( data, outFile )
      except OSError:
         # No space in flash?
         t0( 'Unable to write' )
         raise


def doCheckHitlessReloadSupported( entityManager, plugins=None, pluginPath=None ):
   assert entityManager
   featureEventHandlerRegistry = {}
   ctx = AsuPStoreContext( 'CheckHitlessReloadSupported', entityManager,
                            featureEventHandlerRegistry )
   Plugins.loadPlugins( "AsuPStorePlugin", ctx, plugins=plugins, 
                        pluginPath=pluginPath )
   blockingList = []
   warningList = []
   # iterate over plugin feature and check for hitless support 
   for featureEventHandler in featureEventHandlerRegistry.values():
      ( queryResultWarn, queryResultBlock ) = \
                                featureEventHandler.hitlessReloadSupported()
      if queryResultWarn:
         warningList += queryResultWarn 
      if queryResultBlock: 
         blockingList += queryResultBlock  
   return( warningList, blockingList )

def _getFeatureKeysByOpcode( opcode, entityManager, plugins, pluginPath=None ):
   assert (opcode == 'GetKeys' and entityManager) or \
       (opcode == 'GetSupportedKeys' and not entityManager)
   featureEventHandlerRegistry = {}
   ctx = AsuPStoreContext( opcode, entityManager, featureEventHandlerRegistry )
   Plugins.loadPlugins( "AsuPStorePlugin", ctx,
                        plugins=plugins, pluginPath=pluginPath )
   
   featureKeys = {}
   for fname, feHandler in featureEventHandlerRegistry.items():
      assert fname not in featureKeys
      if opcode == 'GetKeys':
         keys = feHandler.getKeys()
         if keys is not None:
            featureKeys[ fname ] = keys
      else:
         supportedKeys = feHandler.getSupportedKeys()
         if supportedKeys is not None:
            featureKeys[ fname ] = supportedKeys
   return featureKeys

def getFeatureKeys( entityManager, plugins=None, pluginPath=None ):
   '''Generated and return dictionary of per feaure key list that need
   to be stored'''
   assert entityManager
   return _getFeatureKeysByOpcode( 'GetKeys', entityManager,
                                   plugins=plugins, pluginPath=pluginPath )
   
def getFeatureSupportedKeys( plugins=None, pluginPath=None ):
   '''Generate and return dictionary of feature supported key list:
   { 'featureX' : [ keys ], ... }'''
   return _getFeatureKeysByOpcode( 'GetSupportedKeys', None, 
                                   plugins=plugins, pluginPath=pluginPath )

def getAsuPStorePath():
   # returns the full file path: /mnt/flash/<ssu>/asu-persistent-store.json
   return asuFS.getAsuPStoreDefaultFilePath()
