#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import AsuPStore
import SharedMem
import Smash
import Tracing
from collections import defaultdict
from Ethernet import convertMacAddrCanonicalToDisplay

__defaultTraceHandle__ = Tracing.Handle( "AleMacAddrAsu" )
t0 = Tracing.trace0

class AleMacAddrPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''
   Save all the local dynamic macs in PStore during ASU reload.
   '''
   def __init__( self, bridgingStatus, smashBridgingStatus ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.bridgingStatus_ = bridgingStatus
      self.smashBridgingStatus_ = smashBridgingStatus

   def hitlessReloadSupported( self ):
      return ( [], [] )

   def save( self, pStoreIO ):
      t0( 'Saving bridgingStatus asu state' )
      pStoreIO.set( 'fdbStatus', self.getFdbStatusInfo() )

   def getSupportedKeys( self ):
      return [ 'fdbStatus' ]

   def getKeys( self ):
      return [ 'fdbStatus' ]

   def getFdbStatusInfo( self ):
      fdbStatus = defaultdict( list )
      if self.bridgingStatus_:
         for fid in self.bridgingStatus_.fdbStatus:
            learnedHost = self.bridgingStatus_.fdbStatus[ fid ].learnedHost
            for lh in learnedHost.values():
               if lh.entryType == 'learnedDynamicMac':
                  t0( 'Saving dynamic learned host', lh )
                  fdbStatus[ fid ].append(
                        ( convertMacAddrCanonicalToDisplay( lh.address ), lh.intf ) )
      if self.smashBridgingStatus_:
         for key in self.smashBridgingStatus_.smashFdbStatus:
            entry = self.smashBridgingStatus_.smashFdbStatus[ key ]
            if entry.entryType == 'learnedDynamicMac':
               t0( 'Saving dynamic learned host', entry )
               fdbStatus[ entry.fid ].append(
                  ( convertMacAddrCanonicalToDisplay(
                    entry.address ), entry.intf ) )
      return fdbStatus

def Plugin( ctx ):
   featureName = 'AleMacAddrSm'
   bridgingStatus = None
   smashBridgingStatus = None
   if ctx.opcode() == 'Store':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      bridgingHwCapabilities = mg.mount( 'bridging/hwcapabilities',
            'Bridging::HwCapabilities', 'r' )
      mg.close( blocking=True )

      if bridgingHwCapabilities.macTblAsuPstoreNeeded:
         if bridgingHwCapabilities.smashFdbStatusSupported:
            sharedMemEm = SharedMem.entityManager( sysdbEm=entityManager )
            smashMountInfo = Smash.mountInfo( 'reader' )
            smashBridgingStatus = sharedMemEm.doMount(
                        'bridging/status',
                        'Smash::Bridging::Status',
                         smashMountInfo )
         else:
            mg = entityManager.mountGroup()
            bridgingStatus = mg.mount( 'bridging/status',
                     'Bridging::Status', 'r' )
            mg.close( blocking=True )
      else:
         t0( 'Pstoring fdb status not required on current platform' )

   ctx.registerAsuPStoreEventHandler(
         featureName,
         AleMacAddrPStoreEventHandler( bridgingStatus, smashBridgingStatus ) )
