#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import AsuPStore
from Ethernet import convertMacAddrCanonicalToDisplay
import SharedMem
import Smash
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "AleVxlanMacAddrAsu" )
t0 = Tracing.trace0

HostKey = Tac.Type( 'Bridging::HostKey' )
VxlanEntryType = Tac.Type( 'Vxlan::EntryType' )

class AleVxlanMacAddrPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   ''' VXLAN remote macs are saved to pstore.
   '''
   def __init__( self, vxlanFdbStatus, bridgingStatus, smashBridgingStatus ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.vxlanFdbStatus_ = vxlanFdbStatus
      self.bridgingStatus_ = bridgingStatus
      self.smashBridgingStatus_ = smashBridgingStatus

   def hitlessReloadSupported( self ):
      return ( [], [] )

   def save( self, pStoreIO ):
      t0( 'Saving vxlanHwStatusDir asu state' )
      pStoreIO.set( 'vxlanFdbStatus', self.getVxlanFdbStatusInfo() )

   def getSupportedKeys( self ):
      return [ 'vxlanFdbStatus' ]

   def getKeys( self ):
      return [ 'vxlanFdbStatus' ]

   def getVxlanFdbStatusInfo( self ):
      vxlanFdbStatus = defaultdict( list )
      learnedHost = self.vxlanFdbStatus_.learnedHost
      for lh in learnedHost.values():
         if lh.entryType == VxlanEntryType.learnedMac:
            vlan = lh.vlanId
            macaddr = lh.macAddr
            intf = ''
            if self.smashBridgingStatus_:
               try:
                  key = HostKey( vlan, macaddr )
                  entry = self.smashBridgingStatus_.smashFdbStatus[ key ]
                  intf = entry.intf
               except KeyError:
                  continue
            elif self.bridgingStatus_:
               try:
                  fdbStatus = self.bridgingStatus_.fdbStatus[ vlan ]
                  bridgingLh = fdbStatus.learnedHost[ macaddr ]
                  intf = bridgingLh.intf
               except KeyError:
                  continue
            vxlanFdbStatus[ vlan ].append(
               ( convertMacAddrCanonicalToDisplay( macaddr ),
                 str( lh.remoteVtepAddr ), intf ) )
      return vxlanFdbStatus
   
def Plugin( ctx ):
   featureName = 'AleVxlanMacAddrSm'
   vxlanFdbStatus = None
   bridgingStatus = None
   smashBridgingStatus = None
   if ctx.opcode() == 'Store':
      entityManager = ctx.entityManager()
      sharedMemEm = SharedMem.entityManager( sysdbEm=entityManager )

      mg = entityManager.mountGroup() 
      bridgingHwCapabilities = mg.mount( 'bridging/hwcapabilities',
                  'Bridging::HwCapabilities', 'r' )
      mg.close( blocking=True )
      # The smash mount needs to be consistent with how is mounted in VxlanCli.py
      # as this plugin run in the ConfigAgent context
      smashMountInfo = Smash.mountInfo( 'reader' )
      vxlanFdbStatus = sharedMemEm.doMount(
         'vxlan/hardware/fdbStatus',
         'Smash::Vxlan::FdbStatus',
         smashMountInfo )
      if bridgingHwCapabilities.macTblAsuPstoreNeeded:
         if bridgingHwCapabilities.smashFdbStatusSupported:
            smashBridgingStatus = sharedMemEm.doMount(
               'bridging/status',
               'Smash::Bridging::Status',
               smashMountInfo )
         else:
            mg = entityManager.mountGroup()
            bridgingStatus = mg.mount( 'bridging/status', 'Bridging::Status', 'r' )
            mg.close( blocking=True )
      else:
         t0( 'Pstoring fdb status not required on current platform' )

   ctx.registerAsuPStoreEventHandler(
         featureName,
         AleVxlanMacAddrPStoreEventHandler( vxlanFdbStatus, bridgingStatus,
                                            smashBridgingStatus ) )
