#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import AsuPStore
import Tracing

t0 = Tracing.trace0

# Refer FeatureStore in Asu/AsuTestType.tac for store fields.
class EbraPStoreEventHandler( AsuPStore.PStoreEventHandler ):

   def __init__( self, bridgingConfig, subIntfConfigDir, internalVlanStatusDir,
                 intfHwCapability, vlanIntfStatusDir ):
      t0( 'EbraPStoreEventHandler.__init__' )
      AsuPStore.PStoreEventHandler.__init__( self )
      self.bridgingConfig_ = bridgingConfig
      self.subIntfConfigDir_ = subIntfConfigDir
      self.internalVlanStatusDir_ = internalVlanStatusDir
      self.intfHwCapability_ = intfHwCapability
      self.vlanIntfStatusDir_ = vlanIntfStatusDir

   def storeIntfIdToVlanIdCol( self ):
      t0( 'storeIntfIdToVlanIdCol' )
      pyColDir = {}
      config = self.bridgingConfig_
      for intf in config.switchIntfConfig:
         sic = config.switchIntfConfig[ intf ]
         if sic.switchportMode == 'routed' and sic.nativeVlan:
            pyColDir[ intf ] = sic.nativeVlan 
      assert len( pyColDir ) == len( set( pyColDir.values() ) )
      return pyColDir

   def storeInternalVlanStatus( self ):
      pyColDir = {}
      for ksp, evid in self.internalVlanStatusDir_.extendedStatus.items():
         key = ksp.pstoreStr()
         assert key not in pyColDir
         pyColDir[ key ] = evid
      return pyColDir

   def storeSviNameToMacAddrCol( self ):
      t0( 'storeSviNameToMacAddrCol' )
      pyColDir = {}
      for intf in self.vlanIntfStatusDir_.intfStatus:
         intfStatus = self.vlanIntfStatusDir_.intfStatus[ intf ]
         pyColDir[ intf ] = intfStatus.addr
      return pyColDir

   def bridgeMacAddr( self ):
      return self.bridgingConfig_.bridgeMacAddr

   def subIntfToEncapVlan( self ):
      rv = {}
      for name, subIntfConfig in self.subIntfConfigDir_.intfConfig.items():
         rv[ name ] = subIntfConfig.dot1qEncap.outerTag
      return rv

   def subIntfMaxNetDevStackSize( self ):
      return self.intfHwCapability_.subIntfMaxNetDevStackSize

   def save( self, pStoreIO ):
      t0( 'EbraPStoreEventHandler.save' )
      # Save TAC collection as dictionary in string format
      pyCol = self.storeIntfIdToVlanIdCol()
      pStoreIO.set( 'intfNameToVlanId', pyCol )
      pStoreIO.set( 'bridgeMacAddr', self.bridgeMacAddr() )
      pStoreIO.set( 'subIntfNameToEncapVlanId', self.subIntfToEncapVlan() )
      pyCol2 = self.storeInternalVlanStatus()
      pStoreIO.set( 'internalVlanByKsp', pyCol2 )
      pStoreIO.set( 'subIntfMaxNetDevStackSize', self.subIntfMaxNetDevStackSize() )
      pyCol3 = self.storeSviNameToMacAddrCol()
      pStoreIO.set( 'sviNameToMacAddr', pyCol3 )

   def getSupportedKeys( self ):
      return [ 'intfNameToVlanId', 'bridgeMacAddr', 'internalVlanByKsp',
               'sviNameToMacAddr' ]

   def getKeys( self ):
      return [ 'intfNameToVlanId', 'bridgeMacAddr', 'internalVlanByKsp',
               'sviNameToMacAddr' ]

def Plugin( ctx ):

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( 'Ebra',
                                         EbraPStoreEventHandler(
                                            None, None, None, None, None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   bridgingConfig = mg.mount( 'bridging/config', 'Bridging::Config', 'r' )
   subintfConfigDir = mg.mount( 'interface/config/subintf',
                                'Interface::SubIntfConfigDir', 'r' )
   internalVlanStatusDir = mg.mount( 'bridging/internalvlan/status',
                                     'Bridging::InternalVlanStatusDir', 'r' )
   intfHwCapability = mg.mount( 'interface/hardware/capability',
                                'Interface::Hardware::Capability', 'r' )
   vlanIntfStatusDir = mg.mount( 'interface/status/eth/vlan',
                                 'Interface::VlanIntfStatusDir', 'r' )
   t0( "asuStore: ", bridgingConfig )
   def registerEventHandlers():
      ctx.registerAsuPStoreEventHandler(
         'Ebra', EbraPStoreEventHandler( bridgingConfig, subintfConfigDir,
                                         internalVlanStatusDir, intfHwCapability,
                                         vlanIntfStatusDir ) )
   ctx.mountsComplete( mg, 'EbraAsuPStore', registerEventHandlers )
