#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The EthIntf ASU PStore plugin is used for preserving the names of runtime-errdisabled
interfaces accross the reboot. Runtime-errdisable interfaces are those interfaces
that are errdisabled due to the runtime conditions (such as link flaps) as opposed
to user config (such as portgroup-disabled).

Also used for storing all MTUs so interfaces that are brought up by AFPR have the
correct MTU.
"""

import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason
from ErrdisableCliLib import causeNameWithoutSliceId
import Tac
import Tracing

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

# Errdisable cause groups for interfaces that are errdisabled due to the runtime
# conditions, as opposed to due to user config, such as port-group-shutdown.
# Interfaces errdisabled with bpduguard cause group are taken care of with the
# stpErrDisabled warning.
# When adding new cause groups here, also update test/EthIntfAsuPStoreTest.py.
warnErrdisableCauseGroups = [
   'acl',
   'arp-inspection',
   'dot1x',
   'fabric-link-failure',
   'fabric-link-flap',
   'license-enforce',
   'link-flap',
   'loopprotectguard',
   'mlagissu',
   'no-internal-vlan',
   'portchannelguard',
   'portsec',
   'storm-control',
   'stuck-queue',
   'tapagg',
   'tpid',
   'uplink-failure-detection',
   'xcvr-overheat',
   'xcvr-unsupported',
   ]

class EthIntfPStoreEventHandler( AsuPStore.PStoreEventHandler ):

   def __init__( self, errdisableCauses, intfStatuses ):
      self.errdisableCauses = errdisableCauses
      self.intfStatuses = intfStatuses
      super().__init__()

   def save( self, pStoreIO ):
      reloadDownErrdisableIntfs = set()
      for cause in self.errdisableCauses:
         for intf in self.errdisableCauses[ cause ].intfStatus:
            # pylint: disable-next=consider-using-f-string
            t0( 'Saving errdisable intf %s (%s).' % ( intf, cause ) )
            reloadDownErrdisableIntfs.add( intf )
      pStoreIO.set( 'reloadDownErrdisableIntfs', list( reloadDownErrdisableIntfs ) )

      intfMtus = {}
      for name, intfStatus in self.intfStatuses.intfStatus.items():
         if name.startswith( 'Unconnected' ) or \
            intfStatus.operStatus != 'intfOperUp':
            continue
         intfMtus[ name ] = { 'l2Mtu' : intfStatus.l2Mtu, 'mtu' : intfStatus.mtu }
      pStoreIO.set( 'intfMtus', intfMtus )

   def getKeys( self ):
      return [ 'reloadDownErrdisableIntfs' ]

   def getSupportedKeys( self ):
      return [ 'reloadDownErrdisableIntfs' ]

   def hitlessReloadSupported( self ):
      warningList, blockingList = [], []
      for cause in self.errdisableCauses:
         strippedCause = causeNameWithoutSliceId(
            self.errdisableCauses[ cause ].name )
         if strippedCause in warnErrdisableCauseGroups:
            if self.errdisableCauses[ cause ].intfStatus:
               # pylint: disable-next=consider-using-f-string
               t0( "Runtime errdisabled interfaces present for cause %s. "
                   "Adding warning." % cause )
               warningList.append(
                  ReloadHitlessWarningReason( reason='runtimeErrdisabledIntfs' ) )
               break
      return warningList, blockingList

def Plugin( ctx ):
   featureName = 'EthIntf'

   if ctx.opcode() == 'Restore':
      return
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
         featureName, EthIntfPStoreEventHandler( None, None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()

   errdisableCauses = mg.mount( 'interface/errdisable/cause', 'Tac::Dir', 'ri' )

   intfStatuses = mg.mount( 'interface/status/eth/intf',
                            'Interface::EthIntfStatusDir', 'ri' )

   def registerEventHandlers():
      ctx.registerAsuPStoreEventHandler(
         featureName, EthIntfPStoreEventHandler( errdisableCauses, intfStatuses ) )
   ctx.mountsComplete( mg, 'EthIntfAsuPStore', registerEventHandlers )
