# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import AsuPStore

class FruPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''
   This PStore plugin is storing the list of all card names in the system
   which are FruReady. A card is FruReady once it has been detected and its
   Fru plugins have run to completion. This list of FruReady card names will
   then be used during ASU startup to ensure that all of the cards once again
   become FruReady before marking the Fru-Ready stage as complete.
   '''
   def __init__( self, hwSliceDir ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.hwSliceDir_ = hwSliceDir
      self.cardColName = "cardCol"

   def save( self, pStoreIO ):
      cardCol = self._getCardCol()
      if cardCol:
         pStoreIO.set( self.cardColName, cardCol )

   def getSupportedKeys( self ):
      return [ self.cardColName ]

   def getKeys( self ):
      keys = []

      cardCol = self._getCardCol()
      if cardCol:
         keys.append( self.cardColName )

      return keys

   def _getCardCol( self ):
      fruReadyCards = []
      for cardName, cardDir in self.hwSliceDir_.items():
         if "FruReady" in cardDir.entryState:
            fruReadyCards.append( cardName )
      return fruReadyCards

def Plugin( ctx ):
   featureName = "Fru"

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
            featureName, FruPStoreEventHandler( None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()

   hwSliceDir = mg.mount( "hardware/slice", "Tac::Dir", "ri" )

   mg.close( blocking=True )
   ctx.registerAsuPStoreEventHandler(
         featureName, FruPStoreEventHandler( hwSliceDir ) )
