# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This PStore Plugin ensures persistence of L1TopologyModule requests while the agents
that usually request the modules boot up and reconcile with Sysdb state. Since some
of these agents can depend on the state of L1Topology while they initialize
hitlessly, we get stuck in a catch-22 of needing the requests before the agents can
boot and create them. Thus we simply store the total set of requests in PStore to
cover this brief initialization period. These "overriden" requests will get cleared
once the ASU process is complete, allowing the system to syncronize itself with the
current state of the hardware once the boot is complete.

There is only a single key used in this plugin, structured in PStore as below:
 - 'l1TopologyModuleRequests' : {
      modulePath : moduleName
   }
"""

from AsuPStore import PStoreEventHandler
from TypeFuture import TacLazyType

featureName = 'L1TopologyModule'
L1TopoSysdbPaths = TacLazyType( 'Hardware::L1Topology::SysdbPathConstants' )
L1TopoSysdbTokens = TacLazyType( 'Hardware::L1Topology::SysdbPathTokenConstants' )

class L1TopologyModuleAsuPStoreEventHandler( PStoreEventHandler ):

   def __init__( self, topoDir, requestDir ):
      PStoreEventHandler.__init__( self )
      self.topoDir = topoDir
      self.requestDir = requestDir

   def getSupportedKeys( self ):
      return [ 'l1TopologyModuleRequests' ]

   def getKeys( self ):
      return [ 'l1TopologyModuleRequests' ]

   def getRequests( self ):
      resolvedRequests = {}
      # We loop through the slices that we've actually resolved the requests for to
      # ensure we don't try and store requests with conflicts
      for path in self.topoDir[ L1TopoSysdbTokens.sliceDirName ]:
         # We only want to handle submodules, not any root cards
         if '-' not in path:
            continue

         # We have to look through each agent's requests to find the right one. We
         # are guaranteed to only find a single match due to the FruPlugin only
         # resolving those with a single request type.
         for requests in self.requestDir.values():
            if path in requests.module:
               resolvedRequests[ path ] = requests.module[ path ]
               break

      return resolvedRequests

   def save( self, pStoreIO ):
      pStoreIO.setItems( { 'l1TopologyModuleRequests': self.getRequests() } )

def Plugin( ctx ):
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
            featureName, L1TopologyModuleAsuPStoreEventHandler( None, None ) )
      return

   mg = ctx.entityManager().mountGroup()
   topoDir = mg.mount( L1TopoSysdbPaths.topologyDirPath, 'Tac::Dir', 'ri' )
   requestDir = mg.mount( L1TopoSysdbPaths.moduleRequestDirPath, 'Tac::Dir', 'ri' )
   mg.close( blocking=True )

   ctx.registerAsuPStoreEventHandler(
         featureName, L1TopologyModuleAsuPStoreEventHandler( topoDir, requestDir ) )
