# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# N.B. If you edit this file you must regenerate and replace
# the associated AsuPatch patch RPM. For more information see
# /src/AsuPatch/packages/README.

import Tracing
import AsuPStore
import Tac
from Toggles.StageMgrToggleLib import toggleMacsecSsuEnabled

__defaultTraceHandle__ = Tracing.Handle( "MacsecAsu" )

t0 = Tracing.trace0
t1 = Tracing.trace1

class MacsecPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self, status, mkaStatus ):
      self.status = status
      self.mkaStatus = mkaStatus
      self.funcs = { 'cpStatus': self.storeCpStatus }
      super().__init__()

   def save( self, pStoreIO ):
      # Save TAC collection as dictionary in string format
      for k in self.getKeys():
         t0( 'saving', k )
         pyCol = self.funcs[ k ]()
         pStoreIO.set( k, pyCol )

   def getKeys( self ):
      return [ 'cpStatus' ]

   def getSupportedKeys( self ):
      return self.getKeys()

   def hitlessReloadSupported( self ):
      t0( self.__class__.__name__ )
      return ( [], [] )

   def storeMacsecParams( self, macsecParams ):
      jsonMacsecParam = {}
      jsonMacsecParam[ 'cipher' ] = macsecParams.cipher
      jsonMacsecParam[ 'includeSci' ] = macsecParams.includeSci
      jsonMacsecParam[ 'bypassLldp' ] = macsecParams.bypassLldp
      jsonMacsecParam[ 'bypassLldpUnauth' ] = macsecParams.bypassLldpUnauth
      jsonMacsecParam[ 'bypassPtp' ] = macsecParams.bypassPtp
      jsonMacsecParam[ 'ethFlowControl' ] = macsecParams.ethFlowControl
      jsonMacsecParam[ 'replayProtection' ] = macsecParams.replayProtection
      jsonMacsecParam[ 'replayProtectionWindow' ] = \
            macsecParams.replayProtectionWindow
      jsonMacsecParam[ 'only1BitAnMode' ] = macsecParams.only1BitAnMode
      jsonMacsecParam[ 'fipsPostFailureForced' ] = \
            macsecParams.fipsPostFailureForced
      jsonMacsecParam[ 'keyServerPriority' ] = macsecParams.keyServerPriority
      jsonMacsecParam[ 'keyDerivationPadding' ] = \
            macsecParams.keyDerivationPadding
      return jsonMacsecParam

   def storeSakData( self, sak ):
      jsonSak = {}
      # TODO: Store SAK data after BUG985403 is resolved
      return jsonSak

   def storeSakContext( self, context ):
      jsonSakContext = {}
      jsonSakContext[ 'sak' ] = self.storeSakData( context.sak )
      jsonSakContext[ 'lpn' ] = context.lpn
      jsonSakContext[ 'txEnabled' ] = context.txEnabled
      jsonSakContext[ 'rxEnabled' ] = context.rxEnabled
      jsonSakContext[ 'txInstalled' ] = context.txInstalled
      jsonSakContext[ 'rxInstalled' ] = context.rxInstalled
      return jsonSakContext

   def createSciJson( self, sci ):
      jsonSci = {}
      jsonSci[ 'addr' ] = sci.addr
      jsonSci[ 'portNum' ] = sci.portNum
      return jsonSci

   def storeSsci( self, ssci ):
      jsonSsci = {}
      jsonSsci[ 'sci' ] = self.createSciJson( ssci.sci )
      jsonSsci[ 'ssci' ] = ssci.ssci
      return jsonSsci

   def storeRxLPN( self, rxLPN ):
      jsonRxLPN = {}
      jsonRxLPN[ 'lpn' ] = rxLPN.lpn
      jsonRxLPN[ 'an' ] = rxLPN.an
      return jsonRxLPN

   def storeCpStatus( self ):
      t0( self.__class__.__name__, "store Macsec cpStatus only from status" )
      pyCol = {}
      for intf in self.mkaStatus.mkaSuspensionStatus:
         # Expect cpStatus for a suspended interface must be present
         cpStatus = self.status.cpStatus[ intf ]
         info = pyCol.setdefault( intf, {} )
         info[ 'confidentialityOffset' ] = cpStatus.confidentialityOffset
         info[ 'macsecParams' ] = self.storeMacsecParams( cpStatus.macsecParams )
         info[ 'controlledPortEnabled' ] = cpStatus.controlledPortEnabled
         info[ 'unprotectedTraffic' ] = cpStatus.unprotectedTraffic
         info[ 'oldSakContext' ] = \
               self.storeSakContext( cpStatus.oldSakContext )
         info[ 'latestSakContext' ] = \
               self.storeSakContext( cpStatus.latestSakContext )
         info[ 'peerSsci' ] = self.storeSsci( cpStatus.peerSsci )
         rxSakCol = info.setdefault( 'rxSak', {} )
         for an, rxSak in cpStatus.rxSak.items():
            rxSakCol[ str( an ) ] = self.storeSakData( rxSak )
         info[ 'mySsci' ] = self.storeSsci( cpStatus.mySsci )
         info[ 'txSak' ] = self.storeSakData( cpStatus.txSak )
         rxLPNCol = info.setdefault( 'rxLPN', {} )
         for an, rxLPN in cpStatus.rxLPN.items():
            rxLPNCol[ str( an ) ] = self.storeRxLPN( rxLPN )
      return pyCol

def Plugin( ctx ):
   if not toggleMacsecSsuEnabled():
      return

   featureName = 'Macsec'
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( featureName,
                                         MacsecPStoreEventHandler( None, None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   status = mg.mount( 'macsec/status', 'Macsec::Status', 'r' )
   mkaStatus = mg.mount( 'macsec/mkaStatus', 'Macsec::MkaStatus', 'r' )
   mg.close( blocking=True )

   ctx.registerAsuPStoreEventHandler( featureName,
                                      MacsecPStoreEventHandler( status, mkaStatus ) )
