# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This PStore Plugin ensures persistence of Serdes Lane Mappings and
Serdes Polarity Swaps. These mappings are a result of card resolution
algorithms as they are inserted. After an ASU reboot, any and all present
cards would be resolved in an arbitrary order which does not guarantee
the same resulting mappings, so we would like to use this PStore Plugin
to restore that information.

There are three keys used in this plugin, and how it is structured in PStore :
 - 'l1TopologyVx' : where x is a number indicating the source version of
                    the format of the mappings that are to be stored
 - 'l1TopologyGlobalCoreIds' : {
      ## a descriptorKey uniquely identifies a phyScopeTopology using chip+core
      ## values, globalCoreId is the value we need to restore in the models
      descriptorKey : globalCoreId
   }
 - 'l1TopologySerdesLaneMappings' : {
      descriptorKey : { ## a descriptorKey uniquely identifies a phyScopeTopology
         'tx' : txLaneMappings, ## corresponds to phyScopeTopology.txP2L
         'rx' : rxLaneMappings, ## corresponds to phyScopeTopology.rxP2L
      }
   }
 - 'l1TopologySerdesPolarityMappings' : {
      descriptorKey : { ## a descriptorKey uniquely identifies a phyScopeTopology
         'tx' : txPolaritySwaps, ## corresponds to phyScopeTopology.txPolarity
         'rx' : rxPolaritySwaps, ## corresponds to phyScopeTopology.rxPolarity
      }
   }
"""

from AsuPStore import PStoreEventHandler
from TypeFuture import TacLazyType

import Cell
import Tac

featureName = 'L1TopologySerdesMappings'
PhyTypeTopology = TacLazyType( 'Hardware::L1Topology::PhyTypeTopology' )

class SerdesMappingsPStoreEventHandler( PStoreEventHandler ):

   def __init__( self, topoDir ):
      PStoreEventHandler.__init__( self )
      self.topoDir = topoDir

   def getSupportedKeys( self ):
      return [ 'l1TopologyV1',
               'l1TopologyGlobalCoreIds',
               'l1TopologySerdesLaneMappings',
               'l1TopologySerdesPolarityMappings', ]

   def getKeys( self ):
      return [ 'l1TopologyV1',
               'l1TopologyGlobalCoreIds',
               'l1TopologySerdesLaneMappings',
               'l1TopologySerdesPolarityMappings', ]

   # TODO: See BUG486097; This should use the descriptor once chip is in the
   #       hierarchy. For now we generate the key from the scope topology itself.
   def scopeToKey( self, scopeTopo ):
      coreTopo = scopeTopo.phyCoreTopology()
      return ','.join( map( str, ( coreTopo.sliceId(),
                                   coreTopo.chipType(),
                                   coreTopo.chipId(),
                                   coreTopo.phyType(),
                                   coreTopo.componentLocalCoreId,
                                   scopeTopo.scope ) ) )

   # TODO: See BUG486097; This core information should be removed once we have chip
   #       in the hierarchy correctly.
   def coreToKey( self, coreTopo ):
      return ','.join( map( str, ( coreTopo.sliceId(),
                                   coreTopo.chipType(),
                                   coreTopo.chipId(),
                                   coreTopo.phyType(),
                                   coreTopo.componentLocalCoreId ) ) )

   def allCoresInTopology( self ):
      for sliceDir in self.topoDir[ 'slice' ].values():
         # Skip this slice if it does not populate L1 Topology
         if len( sliceDir ) == 0:
            continue

         chipTypes = sliceDir[ 'chip' ].phyChipTypeTopology
         for chipType in chipTypes.values():
            chips = chipType.phyChipTopology
            for chip in chips.values():
               phyTypes = chip.phyTypeTopology
               for phyType in phyTypes.values():
                  phyCores = phyType.phyCoreTopology
                  for phyCore in phyCores.values():
                     yield phyCore

   def allScopesInTopology( self ):
      for phyCore in self.allCoresInTopology():
         for phyScope in phyCore.scopeTopology.values():
            yield phyScope

   def getGlobalCoreIds( self ):
      coreIds = {}
      for core in self.allCoresInTopology():
         coreIds[ self.coreToKey( core ) ] = core.coreId
      return coreIds

   def getLaneMappings( self ):
      laneMappings = {}
      for scope in self.allScopesInTopology():
         laneMappings[ self.scopeToKey( scope ) ] = {
               'tx': dict( scope.txP2L ),
               'rx': dict( scope.rxP2L ),
         }
      return laneMappings

   def getPolarityMappings( self ):
      polarityMappings = {}
      for scope in self.allScopesInTopology():
         polarityMappings[ self.scopeToKey( scope ) ] = {
               'tx': dict( scope.txPolarity ),
               'rx': dict( scope.rxPolarity ),
         }
      return polarityMappings

   def save( self, pStoreIO ):
      # We don't need to pStore L1Topo mappings on fixedsystems for now.
      if Cell.cellType() in [ 'fixed' ]:
         return

      pStoreIO.setItems( {
         'l1TopologyGlobalCoreIds': self.getGlobalCoreIds(),
         'l1TopologySerdesLaneMappings': self.getLaneMappings(),
         'l1TopologySerdesPolarityMappings': self.getPolarityMappings(),
         'l1TopologyV1': True, } )

def Plugin( ctx ):
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler(
            featureName, SerdesMappingsPStoreEventHandler( None ) )
      return
   mg = ctx.entityManager().mountGroup()
   topoDir = mg.mount(
         'hardware/l1/topology', 'Tac::Dir', 'ri' )
   mg.close( blocking=True )
   ctx.registerAsuPStoreEventHandler(
         featureName, SerdesMappingsPStoreEventHandler( topoDir ) )
