#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import datetime
import six
import threading

# Stages involved in the AsuPatch process (any changes here should be propogated
# to AsuPatchDb.py)
class Stage( object ):
   PREINSTALL = 1
   POSTINSTALL = 2
   CLEANUP = 3
   CHECKSWI = 4

# Updating AsuPatchData would require incrementing version num (any change here
# should be propogated to AsuPatchDb.py)
class AsuPatchData( object ):
   def __init__( self ):
      self.version = 2
      self.logFunc = lambda *args, **kwargs: True
      self.checkSwi = lambda arg: True

# SSU patch might start its own thread when getting applied, but the thread
# won't exit gracefully when SSU is boarted (e.g. due to unsaved config)
# This base thread class adds a new flag stopThread_, which will be set to
# True when SSU is being aborted in the shutdown path, and the inherited
# thread class needs to use this flag to terminate its thread gracefully
class AsuPatchBaseThread( threading.Thread ):
   def __init__( self ):
      threading.Thread.__init__( self )
      self.stopThread_ = False

   def stopSSUPatchThread( self ):
      self.stopThread_ = True

   def isSSUPatchThreadStopped( self ):
      return self.stopThread_

#
# Users of AsuPatch who wish to specify scripts for preInstall or postInstall
# sections will derive from AsuPatchBase class here and implement check(),
# checkSwi(), reboot() and cleanup() functions.
# checkSwi() function should validate the swi and check reload compatibility.
# check() function should have all the required prechecks that would let
# reboot() to run without any issues. In certain cases this cannot be guaranteed
# and reboot() can return a non-zero value after cleaning up the state that got
# modified.
# cleanup() is called when other scripts had hit an unforeseeable issue, thus it
# is recommended to do a best effort to unwind to the state prior to reboot().
#
class AsuPatchBase( object ):
   def __init__( self, initName ):
      self.initName = initName
      self.em = None
      self.asuPatchData = AsuPatchData()

   def name( self ):
      return self.initName

   # Override to init any args used by checkSwi/check/reboot/cleanup.
   # You can return a non-zero value if this fails
   def initArgs( self, *args, **kwargs ):
      pass

   # Override to save any args that might be modified after initArgs
   def saveArgs( self, *args, **kwargs ):
      pass

   # To abort AsuPatch, return non-zero value here.
   # Validate the swi and check reload compatibility (e.g: free space avail).
   def checkSwi( self ):
      self.log( 'checkSwi function not implemented' )
      return 0

   # To fail AsuPatch, return non-zero value here
   def check( self ):
      raise NotImplementedError( 'check function is required' )

   # If you see unexpected failure (which check() was not able to identify),
   # return non-zero value here. Returning non-zero value would call cleanup()
   # for all previously rebooted scripts.
   def reboot( self ):
      raise NotImplementedError( 'reboot function is required' )

   # This should not fail. Try a best effort to unwind reboot().
   def cleanup( self ):
      self.log( 'cleanup function not implemented' )

   def executeStage( self, stage ):
      if stage == Stage.CLEANUP:
         self.log( 'AsuPatch calling cleanup()' )
         self.cleanup()
      elif stage == Stage.CHECKSWI:
         self.log( 'AsuPatch calling checkSwi()' )
         return self.checkSwi()
      elif stage == Stage.PREINSTALL:
         self.log( 'AsuPatch calling check()' )
         return self.check()
      elif stage == Stage.POSTINSTALL:
         self.log( 'AsuPatch calling reboot()' )
         return self.reboot()
      else:
         raise NotImplementedError( 'unknown stage %s' % stage )
      return 0

   def execute( self, stageVal, *args, **kwargs ):
      if kwargs is not None:
         for argName, argVal in six.iteritems( kwargs ):
            if argName == 'stage':
               assert stageVal is None
               stageVal = argVal
            elif argName == 'asuReloadData':
               self.em = argVal.mode.entityManager
            elif argName == 'asuPatchData':
               self.asuPatchData = argVal
      assert stageVal is not None
      if self.em is None:
         return 1

      ret = self.initArgs( *args, **kwargs )
      if not ret:
         ret = self.executeStage( stageVal )
      self.saveArgs( *args, **kwargs )
      return ret

   # Users should use this function to log the Asu event related data
   # The AsuEvent.log will hold the Asu event history. Format:
   # YYYY-MM-DD HH:MM:SS <agentName>: <message>
   # We also call the logFunc to dump the log.
   def log( self, message ):
      if self.asuPatchData.version >= 1:
         self.asuPatchData.logFunc( self.name() + ': ' + message )

      dirPath = '/mnt/flash/persist'
      if not os.path.exists( dirPath ):
         print( self.name() + ': ' + message )
         return
      filePath = os.path.join( dirPath, 'AsuEvent.log' )
      with open( filePath, 'a' ) as f:
         f.write( datetime.datetime.now().strftime( '%Y-%m-%d %H:%M:%S ' ) +
                  self.name() + ': ' + message + '\n' )
