# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import functools
import re
import six

class Version( object ): # pylint: disable=useless-object-inheritance
   @functools.total_ordering
   class VersionField( object ): # pylint: disable=useless-object-inheritance
      field = None

      def __init__( self, field=None ):
         if field is not None:
            self.field = int( field )

      def __eq__( self, other ):
         return False if self.field is None else self.field == other

      def __hash__( self ):
         return hash( self.field )

      def __lt__( self, other ):
         return True if self.field is None else self.field < other

      def __str__( self ):
         return '?' if self.field is None else str( self.field )

      def __repr__( self ):
         return str( self )

   productRe = r'^(\w+)-'
   versionRe = r'-(\d+)\.(\d+)\.(\d+)'

   def __init__( self, versionString ):
      self.product = "?"
      self.line = Version.VersionField()
      self.major = Version.VersionField()
      self.minor = Version.VersionField()

      versionString = six.ensure_str( versionString )

      m = re.search( self.productRe, versionString )
      if m:
         self.product = m.group( 1 )

      m = re.search( self.versionRe, versionString )
      if m:
         self.line = self.VersionField( m.group( 1 ) )
         self.major = self.VersionField( m.group( 2 ) )
         self.minor = self.VersionField( m.group( 3 ) )

   @property
   def version( self ):
      # pylint: disable-next=consider-using-f-string
      return '%s.%s.%s' % ( str( self.line ), str( self.major ), str( self.minor ) )

   def __str__( self ):
      # pylint: disable-next=consider-using-f-string
      return "%s-%s" % ( self.product, self.version )

class ParseError( Exception ):
   pass

class ValidVersion( Version ):
   def __init__( self, versionString ):
      # pylint: disable-next=super-with-arguments
      super( ValidVersion, self ).__init__( versionString )
      if ( self.product == "?" or self.line.field is None or
           self.major.field is None or self.minor.field is None or
           str( self ) != versionString ):
         raise ParseError

   def __eq__( self, other ):
      return ( isinstance( other, ValidVersion ) and
               self.product == other.product and self.line == other.line and
               self.major == other.major and self.minor == other.minor )

   def __hash__( self ):
      return hash( ( self.product, self.line, self.major, self.minor ) )
