#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import os
import Tracing
import Tac
import sys

# pylint: disable=attribute-defined-outside-init

ad = Tracing.Handle( 'AutoDel' )

class AutoDelSwitchoverHelper:
   def __init__( self, sysname, rootDir, cellDir ):
      self.sysname = sysname
      self.rootDir = rootDir
      self.cellDir = cellDir
      self.handledAutoDelPaths = False

   def _moveAutoDelPaths( self, oldAutoDelDir, newAutoDelDir, isMounted ):
      ad.t5( '_moveAutoDelPaths' )
      ad.t9( f'Moving {"mounted" if isMounted else "non-mounted"} '
             f'autodeletions from {oldAutoDelDir} to {newAutoDelDir}' )
      for agentName in oldAutoDelDir.keys():
         if agentName not in newAutoDelDir.keys():
            ad.t1( f'Agent {agentName} has no data on our supervisor. '
                   f'Creating a new set for them.' )
            newAutoDelDir.createEntity( 'Tac::AutoDeletingEntitiesSet', agentName )
         oldAgent = oldAutoDelDir.entity[ agentName ]
         newAgent = newAutoDelDir.entity[ agentName ]
         if isMounted:
            fromEntity = oldAgent.autoDeletingMountedEnt
            toEntity = newAgent.autoDeletingMountedEnt
         else:
            fromEntity = oldAgent.autoDeletingNonMountedEnt
            toEntity = newAgent.autoDeletingNonMountedEnt

         # don't copy any elements in old active's cell Id dir, as those will be
         # deleted
         peerPrefix = f'/{self.sysname}/Sysdb/cell/{Cell.peerCellId()}'
         for path in fromEntity:
            if not path.startswith( peerPrefix ):
               ad.t9( f'Moving {path}' )
               toEntity.add( path )
            else:
               ad.t9( f'Skipping {path} because it is an old-active cellified path' )
         fromEntity.clear()

   def deleteNonLaunchedAgents( self ):
      ad.t5( 'deleteNonLaunchedAgents' )
      errorSuffix = 'not found. Not looking for non-launched agents to perform '\
                    'autodeletions for.'
      launcherDir = self.myCellDir.get( 'launcher' )
      if not launcherDir:
         ad.t0( 'launcher directory ' + errorSuffix )
         return
      newLaunchedAgentsDir = launcherDir.get( 'LaunchedAgents' )
      if not newLaunchedAgentsDir:
         ad.t0( 'LaunchedAgents directory ' + errorSuffix )
         return

      # Check if we see any agents that were launched on old active but are not
      # launched here, and perform any autodeletions they had. This can happen if
      # runnability became false on the old active and synced to standby, but
      # LaunchedAgents did not get synced.
      nonLaunchedAgents = self.oldLaunchedAgentsSet - \
            set( newLaunchedAgentsDir.keys() )
      self.performAutoDelsOnSet( nonLaunchedAgents, self.newAutoDelDir )

   def performAutoDelsOnSet( self, agentNames, autoDelDir ):
      ad.t5( 'performAutoDelsOnSet' )
      autoDelsPerformed = 0
      for autoDelAgentName in agentNames:
         ad.t1(
            f'Checking there are any autodeletions to perform for {autoDelAgentName}'
         )
         autoDelAgent = autoDelDir.get( autoDelAgentName )
         if not autoDelAgent or \
               ( len( autoDelAgent.autoDeletingMountedEnt ) == 0 and
               len( autoDelAgent.autoDeletingNonMountedEnt ) == 0 ):
            ad.t1( f'{autoDelAgentName} does not have any autodeletions '
                  'to perform.' )
            continue
         ad.t0( f'Finishing autodeletions for {autoDelAgentName}' )
         autoDelsPerformed += 1
         for col in [ autoDelAgent.autoDeletingNonMountedEnt,
                     autoDelAgent.autoDeletingMountedEnt ]:
            for path in col:
               try:
                  ent = Tac.root.entity[ path ]
               except KeyError:
                  ad.t0( f'Entity {path} not found, it was probably deleted '
                           'already' )
                  continue
               parent = ent.parent
               if not parent:
                  ad.t0( f'Parent of {path} not found, skipping entity deletion' )
                  continue
               ad.t0( f'Deleting {path}' )
               parent.deleteEntity( ent.name )
            col.clear()
      return autoDelsPerformed

   def finishPendingAutoDeletions( self ):
      ad.t5( 'finishPendingAutoDeletions' )
      errorSuffix = 'not found in old active Sysdb. Not checking launchedAgents '\
                  'for in-progress deletions.'
      oldLauncherDir = self.peerCellDir.get( 'launcher' )
      if not oldLauncherDir:
         ad.t0( 'launcher directory ' + errorSuffix )
         return
      oldLaunchedAgentsDir = oldLauncherDir.get( 'LaunchedAgents' )
      if not oldLaunchedAgentsDir:
         ad.t0( 'LaunchedAgents directory ' + errorSuffix )
         return

      self.oldLaunchedAgentsSet = set( oldLaunchedAgentsDir.keys() )
      notLaunchedAgents = set( self.oldAutoDelDir.keys() ) - \
            self.oldLaunchedAgentsSet
      numPendingAutoDel = self.performAutoDelsOnSet( notLaunchedAgents,
                                                     self.oldAutoDelDir )
      if numPendingAutoDel > 1:
         print( 'More than 1 pending autodeletion found. This should not happen',
               file=sys.stderr )

      ad.t1( "Kicking off ClockNotifiee to perform autodeletions for any agents "
             "that don't end up getting launched" )
      if os.getenv( "TESTING_AUTODEL" ) == '1':
         timeToWait = 10
         ad.t1( 'Detected we are in a test. Waiting for 10 seconds' )
      else:
         timeToWait = 600
         ad.t1( 'Waiting for 600 seconds' )
      self.pendingAutoDelClock = Tac.ClockNotifiee(
            self.deleteNonLaunchedAgents, Tac.now() + timeToWait )

   def moveAllAutoDelPaths( self ):
      ad.t5( 'moveAllAutoDelPaths' )
      # copy nonmounted entities
      ad.t0( 'Moving and upgrading autodeleting paths on new active.' )
      self._moveAutoDelPaths( self.oldAutoDelDir, self.newAutoDelDir, False )
      # copy mounted entities
      self._moveAutoDelPaths( self.oldAutoDelDir, self.newAutoDelDir, True )

      # upgrade paths that were deferred on standby
      upgradeableDir = self.rootDir.get( 'deferredAutoDel' )
      if not upgradeableDir:
         ad.t1( 'No deferred paths to upgrade.' )
         return
      ad.t1( 'Upgrading deferred paths.' )
      self._moveAutoDelPaths( upgradeableDir, self.newAutoDelDir, False )
      self._moveAutoDelPaths( upgradeableDir, self.newAutoDelDir, True )


   def handleAutoDelPaths( self ):
      ad.t5( 'handleAutoDelPaths' )
      if self.handledAutoDelPaths:
         ad.t1( 'We already moved autodeletions, returning.' )
         return
      ad.t0( 'Getting required entities to process autodeletions.' )
      self.handledAutoDelPaths = True
      # Check every step of each autodel path and log errors if they don't exist
      errorSuffix = 'not found in Sysdb. Not copying registrations.'
      self.peerCellDir = self.cellDir.get( str( Cell.peerCellId() ) )
      if not self.peerCellDir:
         ad.t0( 'Peer cell dir ' + errorSuffix )
         return
      self.myCellDir = self.cellDir.get( str( Cell.cellId() ) )
      if not self.myCellDir:
         ad.t0( 'My cell dir ' + errorSuffix )
         return

      self.oldAutoDelDir = self.peerCellDir.get( 'autoDel' )
      if not self.oldAutoDelDir:
         ad.t0( 'Old autodeletion dir ' + errorSuffix )
         return

      self.newAutoDelDir = self.myCellDir.get( 'autoDel' )
      if not self.newAutoDelDir:
         ad.t1( 'No autodeletion data on our supervisor. Creating new autoDel dir.' )
         self.myCellDir.createEntity( 'Tac::Dir', 'autoDel' )

      ad.t1( 'Found all required autodeletion entities.' )

      # Check launched agents and see if there are is an autodeletion that we were
      # in the middle of. Note that we do this before merging the active/standby
      # autodeletion sets to keep the active/standby autodeletions separate. Also
      # sets a clock for 600 seconds to see if any agents failed to come up and
      # delete those entities then.
      self.finishPendingAutoDeletions()

      # Move the autodeletion paths from old active's cell dir to standby's
      self.moveAllAutoDelPaths()
