# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import sys
import tty

import CliCommon
import CliParser
import Logging
import Syscall
import Tac
import UtmpDump

class EapiIncompatible:
   def __init__( self, additionalMsg='', overrideFn=None, getModeFn=None ):
      """ This decorator is used to mark a valueFunction of a
      CliCommand as inaccessible when used via an API. If the current
      CLI session was initiated with eapiClient=True,
      this function will emit an error instead of executing the
      decorated valueFunction.

      - overrideFn: If provided, this function will be called with the
        valueFunction's parameters, and if overrideFn returns True,
        the command will be allowed to run. Otherwise the incompatible
        error will be generated.
      - additionalMsg: If supplied, it will be appended to the error
        message.
      - getModeFn: If specified, it is used to return the mode from
        valueFunction's parameters. Otherwise, we assume the mode is the
        either explicitly named or is the first parameter passed to the
        valueFunction.

      Example: Only allow a reload if 'reload now' was specified.
      @BasicCli.EapiIncompatible(
          overrideFn=lambda *args, **kwargs: "now" in kwargs and kwargs[ "now" ],
          additionalMsg="To reload the machine over the API, "
          "please use 'reload now' instead." )
      def doReload( mode, now=False, ... ):
         ...
      """

      self.additionalMsg = additionalMsg
      self.overrideFn = overrideFn
      self.getMode = getModeFn if getModeFn else EapiIncompatible._getMode

   def __call__( self, f ):
      def wrapperFn( *args, **kwargs ):
         mode = self.getMode( *args, **kwargs )
         assert isinstance( mode, CliParser.Mode ), (
            f"mode has unexpected type {type( mode ).__name__}: {mode!r}" )
         if mode.session_.isEapiClient() and (
            not self.overrideFn or not self.overrideFn( *args, **kwargs ) ):
            # pylint: disable-next=consider-using-f-string
            additionalMsg = ' %s' % self.additionalMsg if self.additionalMsg else ''
            # pylint: disable-next=consider-using-f-string
            msg = "Command not permitted via API access.%s" % additionalMsg
            raise CliCommon.CommandIncompatibleError( msg )
         return f( *args, **kwargs )
      return wrapperFn

   @staticmethod
   def _getMode( *args, **kwargs ):
      assert args or "mode" in kwargs, "Unable to find the CLI Mode"
      mode = kwargs[ "mode" ] if "mode" in kwargs else args[ 0 ]
      return mode

SYS_CONFIG_I = Logging.LogHandle( "SYS_CONFIG_I",
   severity=Logging.logNotice,
   fmt="Configured from %s by %s on %s (%s)",
   explanation=( "A network administrator has exited global config mode or "
                 "updated the system configuration from a configuration file." ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_E = Logging.LogHandle( "SYS_CONFIG_E",
   severity=Logging.logNotice,
   fmt="Enter configuration mode from %s by %s on %s (%s)",
   explanation=( "A network administrator has entered global config mode or "
                 "updated the system configuration from a configuration file." ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

# 4771 - Add syslog event on entering config
def logConfigSource( configSource, slog=SYS_CONFIG_I ):
   info = UtmpDump.getUserInfo()
   if not configSource:
      configSource = "Unknown"
   slog( configSource, info[ 'user' ], info[ 'tty' ], info[ 'ipAddr' ] )

##########################################################
#     Some input related APIs
##########################################################
def isTtyInput():
   tinFd = sys.stdin.fileno()
   return os.isatty( tinFd )

def getMultiLineInput( mode, cmd, lstrip=False, prompt="Enter TEXT message" ):
   """ Captures multi-line input from the user, returning it to the caller.
   'cmd' is the cli command that is requesting multi-line input, and is
   used for error reporting purposes. """

   output = ""
   configFile = mode.session_.configFile
   if configFile is not None: # pylint: disable=no-else-raise
      lineNo = configFile.lineNo
      # We're loading the config from a file.
      for line in configFile:
         if line.lstrip().startswith( "EOF" ):
            return output
         if lstrip:
            line = line.lstrip()
         if not line.endswith( '\n' ):
            # we should always have a \n at the end of each line
            line += '\n'
         output += line
      # If we got here, then there was no EOF in the config file.
      # This is probably a config file bug, and we may have swallowed
      # up a bunch of lines that were not intended for us.
      mode.session_.addError( "Incomplete command: EOF not found", cmd, lineNo )
      raise CliParser.AlreadyHandledError()
   else:
      # Check if running off of stdin
      isTty = isTtyInput()

      if isTty:
         # pylint: disable-next=consider-using-f-string
         mode.addMessage( "%s. Type 'EOF' on its own line to end." % prompt )

      while True:
         try:
            if isTty:
               prevKeyBindingsEnabled = mode.session_.cliInput.keyBindingsEnabled()
               try:
                  mode.session_.cliInput.keyBindingsEnabledIs( False )
                  line = mode.session_.cliInput.simpleReadline( "" )
               finally:
                  mode.session_.cliInput.keyBindingsEnabledIs(
                                                        prevKeyBindingsEnabled )
               # readline returns the line without the trailing \n.
               line += "\n"
            else:
               # Stdin is some file-like object.
               line = sys.stdin.readline()
               if not line:
                  break
         except EOFError:
            # Hit Ctrl-D. Treat equivalently to typing EOF
            break
         else:
            if line.lstrip().startswith( "EOF" ):
               break
            if lstrip:
               line = line.lstrip()
            output += line
   # In case, the trailing \n is missing (e.g: Capi), add one
   if output and output[ -1 ] != '\n':
      output += '\n'
   return output

def removeConfigIndentation( multiLineCmd, nestingLevel ):
   """ Used to postprocess multiline input from user when the input is a script,
   returns the script with its original indentation.

   There are two cases when parsing the config:
   1. Interactive input from user.
   2. Reading from config.

   The latter comes with extra indentation depending on the nesting level
   of the sub-mode, each sub-mode indenting it further by 3 spaces.
   The original indentation of the script should be preserved since it is
   important like in a python script.

   The helper can be called with the parsed script and nesting level of the mode,
   for both cases and it returns the original script for case 1,
   and for case 2 it returns the script with its original indentation by
   stripping the extra spaces.
   """
   spacesPerNestingLevel = 3
   indentLen = spacesPerNestingLevel * nestingLevel
   indent = ' ' * indentLen

   lines = []
   for line in multiLineCmd.split( '\n' ):
      if not line.strip():
         lines.append( '' )
      elif line[ 0 : indentLen ] == indent:
         lines.append( line[ indentLen : ] )
      else:
         # If the input has at least one line which is indented with
         # less than indentLen spaces, it means we're in case 1, so return
         # the input with no modifications.
         return multiLineCmd
   # If this line is reached, it means we're reading config,
   # return the postprocessed output without the trailing newline.
   return '\n'.join( lines )

def getSingleLineInput( mode, prompt ):
   output = ""
   configFile = mode.session_.configFile
   # We're loading the config from a file.
   if configFile is not None:
      # Single line input only cares about one line.
      for line in configFile:
         output = line
         break
   else:
      if isTtyInput():
         try:
            prevKeyBindingsEnabled = mode.session_.cliInput.keyBindingsEnabled()
            try:
               mode.session_.cliInput.keyBindingsEnabledIs( False )
               output = mode.session_.cliInput.simpleReadline( prompt )
            finally:
               mode.session_.cliInput.keyBindingsEnabledIs( prevKeyBindingsEnabled )

         except EOFError:
            output = ""
      else:
         output = sys.stdin.readline()
   return output.lstrip( ' ' ).rstrip( '\n' )

def getSingleChar( prompt ):
   isTty = isTtyInput()
   tinFd = sys.stdin.fileno()

   try:
      if isTty:
         # put the terminal into raw mode to avoid waiting for a newline in the input
         # do it before prompting the user so our tests won't send input too early
         tinAttr = tty.tcgetattr( tinFd )
         tty.setraw( tinFd )

      sys.stdout.write( prompt )
      sys.stdout.flush()

      answer = os.read( tinFd, 1 ).decode( errors='ignore' )

   finally:
      if isTty:
         # pylint: disable-next=used-before-assignment
         tty.tcsetattr( tinFd, tty.TCSAFLUSH, tinAttr )

   return answer

getSingleByte = getSingleChar

def confirm( mode, prompt, answerForReturn=True, nonInteractiveAnswer=True ):
   '''Print a prompt and get a single key input from the user.
   If the input is Y, return True; if the input is a return,
   return answerForReturn; else return False. If commandConfirmation
   is disabled or the shell is not interactive, return nonInteractiveAnswer.'''
   if mode.session.commandConfirmation():
      answer = getSingleChar( prompt )
      print( answer )
      if answer in ( 'y', 'Y' ):
         return True
      elif answer in ( '\r', '\n' ):
         return answerForReturn
      else:
         return False
   else:
      return nonInteractiveAnswer

def getChoice( mode, prompt, choices, defaultChoice ):
   if mode.session.commandConfirmation():
      answer = getSingleLineInput( mode, prompt +
            '[' + '/'.join( choices ) + ']:' )
      return answer
   else:
      return defaultChoice

# Some utility functions

# Build a regular expression that matches if the input is not a (case-insensitive)
# prefix of any of the keywords 'include', 'exclude', 'begin', or 'nz' (or
# 'redirect', 'append' or 'tee', which are defined in FileCli.py).

def eitherCase( c ):
   """Returns a regular expression pattern that matches either case of
   character c."""
   assert len( c ) == 1
   return f'[{c.lower()}{c.upper()}]'

def anyCaseRegex( inRegex ):
   """This method returns a modified version of the regex inRegex
   where each alphabet "x" in [a-zA-Z] is replaced with "[xX]".
   For example:
      anyCaseRegex( "(aB|c)" ) = "([aA][bB]|[cC])"
   """
   def anyCaseChar( c ):
      return f"[{c.lower()}{c.upper()}]"
   reList = ( anyCaseChar( c ) if c.isalpha() else c for c in inRegex )
   return r"".join( reList )

def notAPrefixOf( keyword ):
   """Returns a regular expression pattern that matches anything that is not a
   (case-insensitive) prefix of the specified keyword."""
   pattern = '(?!'
   pattern += eitherCase( keyword[ 0 ] )
   for c in keyword[ 1 : ]:
      pattern += '('
      pattern += eitherCase( c )
   for c in keyword[ 1 : ]:
      pattern += ')?'
   pattern += '$)'
   return pattern

def showRunningConfigWithFilter( content, modeFilterExp, parentFilterExp,
                                 printText=True ):
   ''' Helper function for showActive, for filtering out config
   commands based on the filter regular expression. '''

   # If we have more than 1 parent, we need to fix the indentations,
   # where the indentation of each mode should be it's own indentation
   # concatenated with its parent's
   for i in range( 1, len( parentFilterExp ) ):
      indentation = '   ' * i
      # pylint: disable-next=consider-using-f-string
      parentFilterExp[ i ] = '^{}{}'.format( indentation,
                                             parentFilterExp[ i ][ 1 : ] )

   parentMatched = [ False ] * len( parentFilterExp )
   savedParentLines = []

   # Now get this active mode's filterExp and indentation
   modeIndentation = '   ' * len( parentFilterExp )
   modeFilterExp = f'^{modeIndentation}{modeFilterExp[ 1 : ]}'
   matched = False

   output = ''
   for line in content:
      if matched:
         # Found our match, keep printing until we reach config line with
         # unindent to an outer block from a change in indentation level
         # and then we break
         # pylint: disable-next=consider-using-f-string,no-else-continue
         if line.startswith( '%s   ' % modeIndentation ):
            output += line
            continue
         else:
            break

      # Match config line against current active mode expression
      matched = re.match( modeFilterExp, line ) is not None
      if matched:
         # Check if parent modes have matched as well, if not we found
         # same-named child mode of a different parent mode so we reset
         # matched and continue search
         if parentMatched and not all( parentMatched ):
            matched = False
         else:
            # Found our mode! Print our saved parent lines, then this line.
            if savedParentLines:
               output += "".join( savedParentLines )
            output += line
         # continue search with the next line in config
         continue

      if not parentMatched:
         # No parent modes to match so continue search for active mode
         continue
      if not all( parentMatched ):
         # Match first next parent's parentFilterExp (the next parent mode
         # in list that hasn't matched yet) against the config line. Save
         # all config lines with matched parent modes
         idx = parentMatched.count( True )
         if re.match( parentFilterExp[ idx ], line ) is not None:
            parentMatched[ idx ] = True
            savedParentLines.append( line )
      else:
         # If all parent modes matched but indent level in config has
         # changed i.e. there was an unindent to an outer block then our
         # previously matched parent mode is no longer active. We have
         # moved past that mode in config w/o finding any matching config
         # in the currently active mode. We reset both parentMatched
         # and savedParentLines before continuing search.
         if not line.startswith( modeIndentation ):
            assert not matched
            parentMatched = [ False ] * len( parentFilterExp )
            savedParentLines = []
   if printText and output:
      print( output, end='' )
   return output

class NonOverridablePrompt( type ):
   # pylint: disable-next=bad-mcs-classmethod-argument
   def __new__( mcs, name, bases, classdict ):
      if 'shortPrompt' in classdict or 'longPrompt' in classdict:
         for baseCls in bases:
            if isinstance( baseCls, NonOverridablePrompt ):
               raise SyntaxError( 'Cannot override shortPrompt or longPrompt. Use '
                                  'modeKey or longModeKey instead.' )
      return type.__new__( mcs, name, bases, classdict )

class RootPrivilege:
   # Context manager to allow the current Cli thread to gain root privilege
   # Passing in euid=-1 to make it a no-op.
   def __init__( self, euid=0 ):
      self.euid = euid
      self.savedEuid = -1

   def __enter__( self ):
      self.savedEuid = Syscall.geteuid()
      Syscall.setresuid( -1, self.euid, -1 )
      return self

   def __exit__( self, excType, value, excTraceback ):
      Syscall.setresuid( -1, self.savedEuid, -1 )
      self.savedEuid = -1
