# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct
# pylint: disable=c-extension-no-member
import lzma as xz
import lz4.block as lz4

import BiosToolsLib

PLD_SIGNATURE = b"EPLD"
PLD_HEADER_SIZE = 0x10
PLD_ENTRY_SIZE = 0xC
PLD_ENTRY_OFFSET = 0x40

def getInrdPayload( data ):
   payloadOffset = 0
   offset = 0

   while True:
      payloadOffset = data.find( PLD_SIGNATURE, payloadOffset )
      if payloadOffset < 0:
         break

      payloadData = data[ payloadOffset : ]
      try:
         header = payloadData[ : PLD_HEADER_SIZE ]
         # pylint: disable=unused-variable
         name, ver, size, auth = struct.unpack( "<4sHH8s", header )
         offset = size
      except Exception: # pylint: disable=broad-except
         pass
      else:
         compData = payloadData[ PLD_HEADER_SIZE : ]
         hOffset = 0
         while hOffset < len( compData ) - PLD_ENTRY_OFFSET:
            try:
               containerHeader = compData[ hOffset : hOffset + PLD_ENTRY_SIZE ]
               # pylint: disable=unused-variable
               name, offs, size = struct.unpack( "<4sII", containerHeader )
               if name == b"INRD":
                  return payloadData[ offset : offset + size ]
               offset += size
               hOffset += PLD_ENTRY_OFFSET
            except Exception: # pylint: disable=broad-except
               break
      payloadOffset += len( PLD_SIGNATURE )

   return None

def getUnzippedInrd( data ):
   lz4_len = 0
   lz4_data = data[ PLD_HEADER_SIZE : ]
   try:
      lz_hdr = data[ : PLD_HEADER_SIZE ]
      # pylint: disable=unused-variable
      name, lz4_len, length, version = struct.unpack( "<4s3I", lz_hdr )
      if name == b"LZDM" or lz4_len == 0:
         return xz.decompress( lz4_data[ : length ] )
      if name != b"LZ4 ":
         return None
      xz_data = lz4.decompress( lz4_data[ : lz4_len ] )
      cpio_data = xz.decompress( xz_data )
      return cpio_data
   except Exception: # pylint: disable=broad-except
      pass

   return None

def getAbootVersion( data ):
   initrd = None
   version = None

   payload = getInrdPayload( data )
   if payload:
      initrd = getUnzippedInrd( payload )

   if initrd:
      version = BiosToolsLib.getVersionFromInitrd( initrd )

   return version
