# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import os
import re

def path( sysctl ):
   return os.path.join( "/proc/sys", *sysctl.split( '.' ) )

def get( sysctl ):
   with open( path( sysctl ) ) as f:
      return int( f.read().strip() )

def sysctlSet( sysctl, value ):
   with open( path( sysctl ), "w" ) as f:
      print( value, file=f )

def process( sysctl, value ):
   progress = f"{sysctl}:"
   try:
      curValue = get( sysctl )
      progress += f" {curValue}"
      if curValue > value:
         print( f"{progress} > {value}" )
         return
      if curValue == value:
         print( f"{progress} = {value}" )
         return
      sysctlSet( sysctl, value )
      print( f"{progress} => {value}" )
      return
   except Exception as e: # pylint: disable-msg=broad-except
      print( f"{progress} - {e}" )

def parseConfLines( confLines ):
   desiredIncrease = {}
   for line in confLines:
      m = re.match( r'#>\s*(\S+)\s*=\s*(\d+)\s*$', line )
      if m:
         desiredIncrease[ m.group( 1 ) ] = int( m.group( 2 ) )
         continue
      m = re.match( r'-\s*(\S+)$', line )
      if m:
         if m.group( 1 ) in desiredIncrease:
            del desiredIncrease[ m.group( 1 ) ]
         continue
   return desiredIncrease

def main():
   conf = Tac.run( [ '/usr/lib/systemd/systemd-sysctl', '--cat-config' ],
                   stdout=Tac.CAPTURE )
   desiredIncrease = parseConfLines( conf.splitlines() )
   for sysctl, value in desiredIncrease.items():
      process( sysctl, value )
