# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Module to provide a registry to store callback functions. The user defines
their own convention for the key (it can be any hashable type). Example:

class Derived( CallbackRegistry ):

   def foobar( self ):
      key = "myKey"
      for callback in self.getCallbacks( key ):
         callback( "foo", "bar", "baz" )

@Derived.registerCallback( "myKey" )
def myCallback( self, arg1, arg2, arg3 ):
   # Do something

We need a metaclass to do this. If we were to define the dictionary storing the
callback functions as a class variable all derived instances would share the same
set of callbacks. We can't use an instance variable because it wouldn't be in-scope
in @register, which is a class method. If that were an instance method, one would
have to treat the derived class as a singleton and store the instance in the module
scope to annotate functions:
   class Derived( CallbackRegistry ):
      pass
   instance = Derived()
   @instance.registerCallback( "foo" )
   def callback( self, bar ):
      pass
If python had a syntax like:
   class Derived( CallbackRegistry ):
      @self.registerCallback( "foo" )
      def myCallback( self, ... ):
         pass
This could have been avoided.
"""

from collections import defaultdict

class CallbackRegistryMeta( type ):
   """Metaclass to add the callbacks member. Do not use this directly, inheret from
   from CallbackRegistry instead. These are seperate because the metaclass is not in
   the MRO. Derived class instances don't see methods defined in this class.
   """
   def __new__( cls, name, bases, dct ):
      callbacks = defaultdict( lambda: [] )
      dct[ 'callbacks' ] = callbacks
      return type.__new__( cls, name, bases, dct )

class CallbackRegistry( metaclass=CallbackRegistryMeta ):
   """Callback registry base class.
   """
   @classmethod
   def getCallbacks( cls, key ):
      """Get the callbacks registered with "key".

      Args:
         key (str): The key to look up associated callbacks.
      """
      return getattr( cls, 'callbacks' )[ key ]

   @classmethod
   def registerCallback( cls, key ):
      """Decorator to register a callback.

      Args:
         key (str): The callback key to register the decorated function with.
      """
      def decorator( callback ):
         getattr( cls, 'callbacks' )[ key ].append( callback )
         return callback
      return decorator
