#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import io
import os
import sys
import traceback

import Tracing
from CliModel import InvalidRevisionRequestError
import CliModel

th = Tracing.Handle( 'CapiCliCommon' )
t1 = th.trace1

# This prints the exception backtrace to the ConfigAgent log file, which is were
# Capi exceptions go (no "show cli error 123" for capi), and adds a "CLI Exception:"
# prefix, which causes this backtrace to be skipped by "show agent logs crash", since
# they are not crashes.
def logCapiInternalError( excType, exception, excTraceback, msg=None, fd=None ):
   f = io.StringIO()
   if not msg:
      msg = f"Internal error {excType}: {exception}"
   print( msg, file=f )
   traceback.print_exception( excType, exception, excTraceback, file=f )
   msg = "\n".join( "CLI Exception: " + l for l in f.getvalue().split( "\n" ) )
   if fd:
      os.write( fd, msg.encode() + b"\n" )
   else:
      print( msg )

class CliCommandResponse:
   def __init__( self, status, model=None,
                 errors=None, warnings=None, messages=None,
                 execStartTime=None, execDuration=None, isDict=None,
                 globalVersion=None, revision=None ):
      """ A container for response corresponding to running a single
      CLI command. The `status` member is an integer defined in the
      CapiStatus class. `model` must be a CliModel.Model. If `status`
      is not a SUCCESS, the `errors` field should be set.

      To get the JSON-serializable version of this object, you can use
      the `result` property, which gets a dicitionary from the given
      Model and folds in the `errors`, `messages` and `warnings`
      arrays.

      Note: additional warnings and errors may be generated when
      creating the "result" parameter.
      """
      self.status = status
      self.execStartTime = execStartTime
      self.execDuration = execDuration
      self.model = model
      # Note we share the original errors/warnings/messages lists if possible.
      # During computeResultDict generators could call mode.addError/Warning/Message
      # and it'll appear here as well, so we will be able to pick it up.
      self.errors = errors if errors is not None else []
      self.warnings = warnings if warnings is not None else []
      self.messages = messages if messages is not None else []
      # if the passed in model is already a dict, just take it as is
      # and thus bypass the toDict that happens during result() below
      # this happens when we bypass the python model for speed and
      # implement the rendering in c.
      self.result_ = None
      self.isDict = isDict
      if isinstance( self.model, CliModel.Model ):
         self.requestedRevision = self.getAndValidateRevision( globalVersion,
                                                               revision )
      else:
         # TODO pass req version to c code, in such case no downgrade
         # code should produce right stuff in first attempt.
         self.requestedRevision = 1

   @property
   def result( self ):
      """ Returns the dictionary representation of this object. This
      value is cached, so if any of the members are changed between
      calls to result (which is not an expected use case of this
      class), the user should call computeResultDict again """
      if self.result_ is None:
         self.computeResultDict()
      return self.result_

   def getAndValidateRevision( self, globalVersion, revision ):
      """ Given a global version and revision, return the correspoding
      revision of the model the user has requested. If the
      globalVersion and revision do not map to a valid revision, we
      add an error message and we return None. """
      if not self.model:
         return 0

      requestedRevision = 0
      try:
         requestedRevision = self.model.getRevision( globalVersion, revision )
      except InvalidRevisionRequestError as e:
         self.errors.append( str( e ) )
      return requestedRevision

   def computeResultDict( self, streaming=False, raiseOnException=False ):
      """ Populates the dict representation of the model (or uses an
      empty dict if no model exists) with the errors/warnings/messages
      arrays. """
      if isinstance( self.model, str ): # happens when cliprinted
         self.result_ = self.model # it is already json, keep it this way
         return
      result = {}
      if self.model:
         try:
            if isinstance( self.model, CliModel.Model ):
               # For all cases that need special handling, use toDict, like when
               # model degradation is needed (msgspec can't handle that), or we don't
               # have a model cause the cmd failed, or extra meta data needs to be
               # additionally inserted (this is not the fast path!), or this is a
               # CliExtension CliModel (its toDict func produces the data).
               if ( self.errors or self.warnings or self.messages or
                    self.execStartTime is not None or
                    self.execDuration is not None or
                    self.requestedRevision < self.model.__revision__ or
                    self.model.__dict__.get( '__data' ) ):
                  result = self.model.toDict( self.requestedRevision,
                                              streaming=streaming )
               else:
                  result = self.model
            else:
               result = self.model
         except Exception:  # pylint: disable-msg=W0703
            # There was a problem serializing the model into a
            # dict. This is likely caused by an internal exception
            # while populating a GeneratorList/GeneratorDict.
            # In the context of Cli (|json) we re-raise it (-> internal error &
            # logged to /var/log/cli), in eapi context handle it here.
            if raiseOnException:
               raise
            excType, exception, excTraceback = sys.exc_info()
            # pylint: disable-next=consider-using-f-string
            msg = "Exception while serializing {}: {}".format(
               type( self.model ).__name__, exception )
            logCapiInternalError( excType, exception, excTraceback, msg=msg,
                                  fd=sys.stderr.fileno() )
            # self.status = 500 is same as CapiStatus.INTERNAL_ERROR but it can't
            # be used here as it requires import of httplib which imports ssl. This
            # causes Eos/test/CliTests.py to fail in CliWeightWatcher test because
            # ssl is a memory hog forbidden to be imported by Cli.
            self.status = 500
            self.errors.append( msg )
      if self.errors:
         result[ "errors" ] = self.errors
      if self.warnings:
         result[ "warnings" ] = self.warnings
      if self.messages:
         result[ "messages" ] = self.messages

      metadata = {}
      hasMeta = False
      if self.execStartTime is not None:
         hasMeta = True
         metadata[ "execStartTime" ] = self.execStartTime
      if self.execDuration is not None:
         hasMeta = True
         metadata[ "execDuration" ] = self.execDuration

      if hasMeta is True:
         result[ "_meta" ] = metadata

      self.result_ = result

   def __repr__( self ):
      return f"{self.__class__.__name__}({self.status!r}, {self.result!r})"
