# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import io
import os
import signal
import json
import sys
from Syscall import gettid
import tempfile
import threading
import time
import traceback

import BasicCli
import BasicCliSession
import CapiCliCommon
import CliCommon
import CliModel
import CliParser
import CliPatchSigint
import ConfigMount
import Tac
import TacSigint
import Tracing

ERR_408_TEXT = "Command exceeded its allowed time limit"

__defaultTraceHandle__ = Tracing.Handle( 'CliApi' )
th = Tracing.defaultTraceHandle()
t1 = th.trace1
t2 = th.trace2
t3 = th.trace3

class CapiStatus:
   """ Status codes for a CliCommandResponse. """
   SUCCESS = 200 # httplib.OK
   ERROR = 400 # httplib.BAD_REQUEST
   UNAUTHORIZED = 401 # httplib.UNAUTHORIZED
   FORBIDDEN = 403 # httplib.FORBIDDEN
   NOT_FOUND = 404 # httplib.NOT_FOUND
   REQUEST_TIMEOUT = 408 # httplib.REQUEST_TIMEOUT
   NOT_EXECUTED = 412 # httplib.PRECONDITION_FAILED
   CONFIG_LOCKED = 423 # httplib.LOCKED
   CLOSED_NO_RESP = 444 # "Connection Closed Without Response"; not in httplib
   INTERNAL_ERROR = 500 # httplib.INTERNAL_SERVER_ERROR
   NOT_CAPI_READY = 501 # httplib.NOT_IMPLEMENTED


# All the following types of exceptions are translated into NOT_FOUND:
NOT_FOUND_EXCEPTIONS = ( CliCommon.InvalidInputError,
                         CliCommon.AmbiguousCommandError,
                         CliCommon.IncompleteCommandError,
                         CliCommon.IncompleteTokenError,
                         CliCommon.GuardError )

# Encodes a CliModel into a json binary string and returns it, or writes it directly
# to the provided fd.
# climodel=True below means we will remove attributes set to None or whose names
# start with an underscore (a deviation from standard json encoding).
def encodeCliModel( cliModel, fd=None ):
   # top level import of msgspec causes Snmp test errors because of supposed leaks
   import msgspec # pylint: disable-msg=import-outside-toplevel
   return msgspec.json.encode( cliModel, stream=fd, climodel=True )

def streamCliModel( model, streamFile ):
   # As long as a CliModel can contain a generator that can fail, we can still get
   # errors at this point. We hope those happen before 100kB (the streaming buffer
   # size) is output in which case we won't end up with bad json (anyway,
   # coding errors should not escape into the world).
   try:
      encodeCliModel( model, fd=streamFile.fileno() )
   except Exception as e: # pylint: disable-msg=broad-except
      # This could happen for faulty generator-dict/list (internal error)
      traceback.print_exc() # leave detailed debugs in log file
      err = json.dumps( f"internal error: json serialization: {str(e)}" )
      streamFile.write( f'{{"errors":[{err}]}}'.encode() )

# Encodes a CapiCliCommon.CliCommandResponse into json (needed in tests).
# A CliCommandResponse has the cliModel, a status and a result, and a req revision.
# The cliModel could have generators that call mode.addError, so we need to expect
# new errors and convert those, even internal errors are possible, like generators
# yielding bad data.
# The first time result is accessed, cliModel will be toDict-ed into it if the
# requested revision is not the latest, otherwise the cliModel is moved into result
# (with msgspec, CliModel types can be directly converted to json, no toDict req)
def serializeCliCommandResponse( response, session ):
   try:
      if not session.errors_: # we have a model to serialize to json
         j = encodeCliModel( response.result )
      if session.errors_: # an error could have been added in the line above!
         j = f'{{"errors": {json.dumps(session.errors_)}}}'.encode()
         response.status = CapiStatus.ERROR
   except Exception as e: # pylint: disable-msg=broad-except
      j = f'{{"errors": {json.dumps(str(e))}}}'.encode()
      response.status = CapiStatus.INTERNAL_ERROR
   return j

class TimeMonitorThread( threading.Thread ):
   """ Thread class to execute a function and return the results or HTTP 408 """
   def __init__( self, timeout ):
      super().__init__()
      self.timeout_ = timeout
      self.maxTimeCondition_ = threading.Condition()
      self.monitoredThread_ = threading.currentThread()
      self.monitoredThreadTid_ = gettid()
      self.wakenedFromThread_ = False

   def wakeup( self ):
      t1( 'Wakeup TimeMonitorThread for pid:', self.monitoredThreadTid_ )
      self.wakenedFromThread_ = True
      self.maxTimeCondition_.acquire()
      self.maxTimeCondition_.notify()
      self.maxTimeCondition_.release()

   def run( self ):
      t1( 'Running TimeMonitorThread for pid:', self.monitoredThreadTid_, 'for',
            self.timeout_ )
      # wait for the maxTimeCondition. We will either be woken up by our monitored
      # thread or we will timeout.
      self.maxTimeCondition_.acquire()
      self.maxTimeCondition_.wait( self.timeout_ )
      t1( 'Done waiting for pid:', self.monitoredThreadTid_, 'for',
            self.timeout_ )
      self.maxTimeCondition_.release()

      # if awakened from monitored thread then don't send any kill signals
      if self.wakenedFromThread_:
         return

      # Send a sigint to the monitored thread
      t1( 'Sending sigint for pid:', self.monitoredThreadTid_ )
      if hasattr( self.monitoredThread_, 'killChildThreads' ):
         # this can also be run outside of the CliServer context, namely within
         # within testing. However on the CliServer this should work
         self.monitoredThread_.killChildThreads( signal.SIGINT )
      CliPatchSigint.kill( self.monitoredThreadTid_ )

class OutputBuffer:
   """Something akin to a StringIO that correctly handles fileno().

   Use this if you want to capture stdout or stderr while running
   subprocesses or doing other similar things that may directly use
   fd 1 or 2.

   This uses a cStringIO under the hood and falls back to a temporary file
   when fileno() is called.  Only write operations are supported.  Seeking
   is not supported.  Unsupported methods are left unimplemented to trigger
   errors as early as possible (and give a chance to pylint to find out too).
   """

   def __init__( self ):
      self.buf_ = io.StringIO()
      self.tmpfile_ = None
      self.pipeFile_ = None

   def usePipe( self, pipeFile ):
      self.pipeFile_ = pipeFile

   def fileno( self ):
      if self.pipeFile_:
         return self.pipeFile_
      elif self.tmpfile_ is None:
         self.tmpfile_ = tempfile.TemporaryFile( mode="w+t", prefix="CapiBuffer" )
      # Before handing out the file descriptor, let's make sure we flush
      # anything that may have been written from Python to the tempfile,
      # that could still be buffered.
      self.tmpfile_.flush()
      return self.tmpfile_.fileno()

   def flush( self ):
      if self.tmpfile_:
         self.tmpfile_.flush()
      self.buf_.flush()

   def close( self ):
      if self.tmpfile_:
         self.tmpfile_.close()
         self.tmpfile_ = None
      self.buf_.close()

   def isatty( self ):
      return False

   def tell( self ):
      pos = self.buf_.tell()
      if self.tmpfile_:
         pos += self.tmpfile_.tell()
      return pos

   def truncate( self, size=None ):
      bufsize = self.buf_.tell()
      filesize = self.tmpfile_.tell() if self.tmpfile_ else 0
      if size is None:
         size = bufsize + filesize
      if self.tmpfile_:
         if size <= bufsize:
            self.tmpfile_.close()
            self.tmpfile_ = None
         else:
            self.tmpfile_.truncate( size - bufsize )
            return
      self.buf_.truncate( size )

   def write( self, s ):
      if self.pipeFile_:
         os.write( self.pipeFile_, s.encode() )
      elif self.tmpfile_:
         self.tmpfile_.write( s )
      else:
         self.buf_.write( s )

   def writelines( self, iterable ):
      if self.pipeFile_:
         for i in iterable:
            os.write( self.pipeFile_, i.encode() )
            os.write( "\n" )
      elif self.tmpfile_:
         self.tmpfile_.writelines( iterable )
      else:
         self.buf_.writelines( iterable )

   def getvalue( self ):
      assert not self.pipeFile_, "Can't get value with pipe installed"
      buf = self.buf_.getvalue()
      if self.tmpfile_:
         self.tmpfile_.seek( 0 )
         buf += self.tmpfile_.read()
         self.tmpfile_.close()
         self.tmpfile_ = None
      return buf

class IOManager:
   """Context manager that redirects stdout/stderr to a StringIO, and
   overrides stdin when provided with an input string."""

   def __init__( self ):
      self.output_ = OutputBuffer()
      self.input_ = None
      self.inputStr_ = None
      self.savedStdinFd_ = None
      self.savedStdoutFd_ = None
      self.savedStderrFd_ = None

   def usePipe( self, pipeFile ):
      self.output_.usePipe( pipeFile )

   def replaceFd( self, new, original ):
      saved = None
      if new != original:
         saved = os.dup( original )
         os.dup2( new, original )
      return saved

   def restoreFd( self, saved, original ):
      if saved:
         os.dup2( saved, original )
         os.close( saved )

   def __enter__( self ):
      self.output_.truncate( 0 )
      sys.stdout.flush()
      self.savedStdoutFd_ = self.replaceFd( self.output_.fileno(),
                                            sys.stdout.fileno() )
      devnull = os.open( os.devnull, os.O_WRONLY )
      self.savedStderrFd_ = self.replaceFd( devnull,
                                            sys.stderr.fileno() )
      os.close( devnull )

      if self.inputStr_:
         # Use a temporary file containing the contents of the input as stdin
         inIO = tempfile.TemporaryFile( mode='w+t' )
         inIO.write( self.inputStr_ )
         inIO.flush()
         inIO.seek( 0 )
         self.input_ = inIO
         self.savedStdinFd_ = self.replaceFd( inIO.fileno(), sys.stdin.fileno() )

      return self

   def __exit__( self, excType, value, excTraceback ):
      sys.stdout.flush()
      self.restoreFd( self.savedStdoutFd_, sys.stdout.fileno() )
      self.restoreFd( self.savedStderrFd_, sys.stderr.fileno() )
      self.restoreFd( self.savedStdinFd_, sys.stdin.fileno() )

      self.savedStdinFd_ = None
      self.savedStdoutFd_ = self.savedStderrFd_ = None
      # Clear the input so it isn't reused multiple times
      self.input_ = self.inputStr_ = None

   def inputIs( self, inputStr ):
      """Sets the input string which will be fed as stdin next time
      this class is used. The input will be cleared after one use."""
      self.inputStr_ = inputStr

   def get( self ):
      """Returns a string with the captured output."""
      return self.output_.getvalue()

   def close( self ):
      self.output_.close()

class CliCommand:

   # Let us use 'input' as a valid parameter name:
   # pylint: disable-msg=W0622
   def __init__( self, cmd, revision=None, input=None ):
      """ A structure representing a CLI command along with any relevant options.
      - cmdStr: the CLI command
      - revision: the desired Model revision
      - inputStr: a string which will be fed as stdin during the command's execution
      """
      self.cmdStr = cmd
      self.revision = revision
      self.input = input

   def __str__( self ):
      """ Provide a nice view of the command, i.e.
      "'show aliases' (revision='1' input='Hello World!')"
      """
      ret = "'%s'" % self.cmdStr
      if self.revision or self.input:
         annotation = ''
         if self.revision:
            annotation += 'revision=%r' % self.revision
         if self.input:
            annotation += 'input=%r' % self.input
         ret += '(%s)' % annotation
      return ret

class TextResponse( CliModel.Model ):
   """ A simple Model which represents the output of a command in
   'text' mode. """
   # No need to implement a render() method, as it should never be
   # called for this model.
   output = CliModel.Str( "ASCII output from the CLI command" )

class CapiExecutor:
   """ Main class responsible for executing CLI commands and returning
   CliCommandResponses. External programs that wish to interact with a
   programatic CLI should instantiate an instance of this class, and
   use the executeCommand[s] methods to run commands. """

   def __init__( self, cli, session, stateless=True ):
      self.cli_ = cli
      self.session_ = session
      self.stateless_ = stateless
      t2( "CapiExecutor initialized" )

   def warmupCache( self ):
      cliSessionName = "capi-%s-warmup" % os.getpid()
      commands = [ CliCommand( "configure session %s" % cliSessionName ),
                   CliCommand( "rollback clean-config" ),
                   CliCommand( "copy startup-config session-config" ),
                   CliCommand( "exit" ),
                   CliCommand( "no configure session %s" % cliSessionName ) ]
      self.executeCommands( CliCommon.MAX_PRIV_LVL, commands )

   def _maybeRunPreCmdsFn( self, preCommandsFn, textOutput, timestamps ):
      if preCommandsFn:
         assert callable( preCommandsFn ), "preCommandsFn must be callable"
         t2( "We have a preCommandsFn" )
         for command in preCommandsFn():
            t2( "Running preCommand", command )
            response = self._execute( command, textOutput, timestamps )
            assert response.status == CapiStatus.SUCCESS, (
               f"Command '{command}' did not succeed, received {response}" )

   def _maybeRunPostCmdFn( self, postCommandsFn, errorSeen,
                           textOutput, timestamps ):
      if postCommandsFn:
         assert callable( postCommandsFn ), "postCommandsFn must be callable"
         t2( "We have a postCommand fnc and errorSeen was:", errorSeen )
         for command in postCommandsFn( errorSeen ):
            t2( "Running postCommand", command )
            response = self._execute( command, textOutput, timestamps )
            assert response.status == CapiStatus.SUCCESS, (
               f"Command '{command}' did not succeed, received {response}" )

   def _makeTimeoutResponse( self ):
      errors = [ ERR_408_TEXT ]
      rt = CapiStatus.REQUEST_TIMEOUT
      return CapiCliCommon.CliCommandResponse( rt, errors=errors )

   def _streamCliModel( self, model, streamFile ):
      # As long as a CliModel can contain a generator that can fail, we can still get
      # errors at this point. We hope those happen before 100kB (the streaming buffer
      # size) is output in which case we won't end up with bad json (anyway,
      # coding errors should not escape into the world).
      try:
         encodeCliModel( model, fd=streamFile.fileno() )
      except Exception as e: # pylint: disable-msg=broad-except
         # This could happen for faulty generator-dict/list (internal error)
         traceback.print_exc() # leave detailed debugs in log file
         err = json.dumps( f"internal error: json serialization: {str(e)}" )
         streamFile.write( f'{{"errors":[{err}]}}'.encode() )
      # Generators could also leave an error via mode.addError. In streaming mode,
      # which users request because they run a show command that can scale, we do
      # not support errors (show commands have no real reason to error actually!),
      # so do nothing but trace the fact.
      if self.session_.errors_:
         t1( "Generator with addError:", self.session_.errors_ )

   def _executeCommands( self, responses, commands, stopOnError, textOutput,
                         timestamps, preCommandsFn, postCommandsFn,
                         globalVersion, expandAliases, streamFd,
                         timeMonitorThread=None ):
      if responses:
         timedOutResponse = self._makeTimeoutResponse()
      streamFile = None
      try: # pylint: disable=too-many-nested-blocks
         self._maybeRunPreCmdsFn( preCommandsFn, textOutput, timestamps )
         try:
            errorSeen = False
            if streamFd is not None:
               # Use buffered IO, iterencode returns mini chunks
               streamFile = io.BufferedWriter( io.FileIO( streamFd, mode='wb',
                                                          closefd=False ) )
            for i, command in enumerate( commands ):
               responses[ i ] = timedOutResponse
               t2( "Running command", command )
               if streamFile and i > 0:
                  streamFile.write( b"," )
                  streamFile.flush()
               try:
                  resp = self._execute( command, textOutput, timestamps,
                                        globalVersion, expandAliases, streamFd )
                  TacSigint.check()
                  if streamFile and not textOutput:
                     # TODO implement this with polymorphism
                     if not isinstance( resp.model, CliModel.DeferredModel ):
                        if ( isinstance( resp.model, CliModel.Model ) and
                             not resp.model.__dict__.get( '__data' ) ):
                           streamCliModel( resp.model, streamFile )
                        else: # degraded or CliExtension models, or errors
                           encoder = json.JSONEncoder()
                           for chunk in encoder.iterencode( resp.result ):
                              streamFile.write( chunk.encode() )
                        streamFile.flush()
               except KeyboardInterrupt:
                  t1( 'Keyboard interrupt seen while running CAPI command' )
                  return
               finally:
                  BasicCliSession.doCommandHandlerCleanupCallback()

               responses[ i ] = resp
               errorSeen |= resp.status != CapiStatus.SUCCESS
               if stopOnError and errorSeen:
                  break
         finally:
            self._maybeRunPostCmdFn( postCommandsFn, errorSeen, textOutput,
                                     timestamps )
      finally:
         finalRes = responses[ -1 ] if responses else None
         self._terminateSession( finalRes, stream=streamFile )
         # notify our time monitor thread (if applicable) that we are done!
         if timeMonitorThread:
            timeMonitorThread.wakeup()

   def executeCommands( self, commands, stopOnError=True,
                        textOutput=False, timestamps=False,
                        preCommandsFn=None, postCommandsFn=None,
                        globalVersion=None, autoComplete=False, requestTimeout=None,
                        expandAliases=False, streamFd=None ):
      """Executes the given Cli commands at the given privilege level.

      Args:
        - commands: An iterable of `CliCommand`s to parse and execute.
        - stopOnError: Whether or not to stop executing commands on the first
          command that fails.  True is the safest, otherwise a sequence of
          commands such as [ "interface et1", "interface et99", "shutdown" ]
          will shut down et1 instead of another typo'ed interface name.
        - textOutput: If True, return the unstructured text output printed by
          the Cli, even if the command returned an Model.
        - timestamps: True if timestamps should be included in the response. This
          will include how long it took to execute and when the command started to
          execute
        - preCommandsFn: Function that return a list of commands that will run before
          the commands are run.
        - postCommandsFn: Function that return a list of commands that will run
          before the commands are run.
        - globalVersion: The global version of the API to use if the command
          does not override it with a specific revision.
      Returns:
        A list of CliCommandResponse objects

      Example:
        >> capi.executeCommands( 15, [ CliCommand( "configure" ),
                                       CliCommand( "interface Et1" ),
                                       CliCommand( "shutdown" ) ] )
      """
      self._updateSession( textOutput=textOutput, autoComplete=autoComplete )

      cmdNotExecuted = CapiCliCommon.CliCommandResponse( CapiStatus.NOT_EXECUTED,
                                           errors=[ "An earlier command failed" ] )
      responses = [ cmdNotExecuted ] * len( commands )
      timeMonitorThread = None
      if requestTimeout is not None:
         timeMonitorThread = TimeMonitorThread( requestTimeout )
         timeMonitorThread.start()
      self._executeCommands( responses, commands, stopOnError, textOutput,
            timestamps, preCommandsFn, postCommandsFn, globalVersion, expandAliases,
            streamFd, timeMonitorThread=timeMonitorThread )
      return responses

   def executeCommand( self, command, **kwargs ):
      """Executes the given CliCommand at the given privilege level.
      Returns a CliCommandResponse. """
      return self.executeCommands( [ command ], **kwargs )[ 0 ]

   def getHelpInfo( self, command, isText=False ):
      """ Get the help informations for each possible completions of
      the given CliCommand."""
      self._updateSession( textOutput=isText )

      response = None
      try:
         tokens, partialToken, completions = \
            self._getCompletionsHelper( command, startWithPartialToken=False )

         helpInfos = { c.name: c.help for c in completions }
         t2( 'Tokens: ', tokens )
         t2( 'Partial Token: ', partialToken )
         t2( 'Completions: ', completions )
         t2( 'HelpInfos: ', helpInfos )
         t2( 'Mode: ', self.session_.mode_ )

         return helpInfos

      finally:
         self._terminateSession( None )

      return response

   def _runCmd( self, command, expandAliases ):
      tokens = CliParser.textToTokens( command.cmdStr,
                                       mode=self.session_.mode_ )
      if not tokens:
         return None
      expandedTokens = None
      if expandAliases:
         tokenized = BasicCliSession.expandAlias( self.session_.mode_,
                                                  self.session_.cliConfig,
                                                  command, tokens, True )
         if len( tokenized ) > 1: # pylint: disable=no-else-raise
            raise CliCommon.CommandIncompatibleError( "Multi-line aliases "
                                               "are currently not supported" )
         else:
            expandedTokens = tokenized[ 0 ]
      else:
         expandedTokens = tokens

      if not expandedTokens:
         return None

      aaa = ( self.session_.authenticationEnabled() or
              self.session_.authorizationEnabled() )
      return self.session_.runTokenizedCmd( expandedTokens, aaa=aaa, fromCapi=True )

   def _execute( self, command, wantText, timestamps=False,
                 globalVersion=None, expandAliases=False, streamFd=None ):
      """Executes one CliCommand in the given mode."""
      ioManager = IOManager()
      self.session_.shouldPrintIs( wantText )

      # handlers that print json via CliPrint will need to know the revision before
      # even starting to respond, so put it where they can pick it up. Translation
      # of version to revision is done in BasicCli once we know the model.
      self.session_.requestedModelRevision_ = command.revision
      self.session_.requestedModelVersion_ = globalVersion
      ioManager.inputIs( command.input )
      self.session_.clearMessages()

      ts = time.time()
      execStartTime = ts if timestamps else None

      excType = excTraceback = None
      if streamFd is not None:
         ioManager.usePipe( streamFd )
      result = None
      isInternalError = False
      with ioManager:
         try:
            result = self._runCmd( command, expandAliases )
         except Exception:  # pylint: disable-msg=W0703
            # do this before getting the output as it might add errors
            excType, result, excTraceback = sys.exc_info()
            isInternalError = self._handleException( excType, result, excTraceback,
                                                     command )

      output = ioManager.get() if streamFd is None else ""
      t2( "Output:", output )

      ioManager.usePipe( None )
      ioManager.close()
      te = time.time()
      execDuration = ( te - ts ) if timestamps else None

      t = ( te - ts ) * 1000
      t2( "Executed", command, "in %.1fms" % t )

      deferredModel = False
      # some models like "show tech" can't be validated
      validateModel = CliModel.shouldValidateModel( result )
      if validateModel is False:
         result = type( result )
      try:
         deferredModel = issubclass( result, CliModel.DeferredModel )
      except TypeError as e: # pylint: disable=unused-variable
         pass
      if ( isinstance( result, CliModel.Model ) and result.__printed__ ):
         deferredModel = True
         result = type( result )

      if isinstance( result, Exception ):
         if isInternalError:
            # We have to do this after closing the ioManager or the output
            # will go into streaming file instead of agent logs.
            self._printInternalError( excType, result, excTraceback )
         ret = self._onException( result, wantText, output,
                                  execStartTime=execStartTime,
                                  execDuration=execDuration )
      elif streamFd is not None and deferredModel:
         # no need to return data, the result is already being written to the pipe
         return CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS,
                                                  model=CliModel.DeferredModel(),
                                                  execStartTime=execStartTime,
                                                  execDuration=execDuration )
      elif deferredModel:
         # value function may write to stream directly for speed, in this
         # case there is no python model that needs to be toDict-ed, the
         # captured stream already is json, store it as such in the response, a
         # response with a result of type str will be dumped as is later instead
         # of being toDict-ed if not a dict yet and then json.dump-ed.
         if not wantText:
            # fix empty outputs (if people dont even call start/end)
            if output == "":
               output = "{}"
            # In test env, check conformance to declared model; also return py dict
            # instead of json, since cohab btests expect that and we need the py dict
            # to check conformance anyway
            if ( self.cli_.cliConfig_.validateOutput or
                 os.environ.get( "A4_CHROOT" ) ):
               # convert json to py, an exception will appear like that:
               # "error": {"message": "No JSON object could be decoded"
               try:
                  j = json.loads( output )
               except Exception: # pylint: disable=broad-except
                  print( output )
                  print( ' '.join( hex( ord( c ) ) for c in output[ : 10 ] ) )
                  print( ' '.join( hex( ord( c ) ) for c in output[ -10 : ] ) )
                  raise
               if ( self.session_.requestedModelRevisionIsLatest_ and
                    validateModel ):
                  try:
                     CliModel.unmarshalModel( result, j, degraded=False )
                  except CliModel.ModelUnknownAttribute as e:
                     t2( "bad deferred model instance", str( e ) )
                     warnMsg = CliCommon.SHOW_OUTPUT_UNEXPECTED_ATTRIBUTE_WARNING
                     self.session_.addWarning( f"{warnMsg} {e}" )
               output = j
            # isDict below means 'no need to run toDict()' (migth still be a str)
            ret = CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS, output,
                                                    isDict=True )
         else:
            result = TextResponse( output=output )
            return CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS,
                                                     model=result )
      elif not wantText and isinstance( result, CliModel.Model ):
         ret = self._onModelReturn( result,
                                    globalVersion, command.revision,
                                    execStartTime=execStartTime,
                                    execDuration=execDuration )
      else:
         # Everything else. In this case we will ignore the return value of the
         # value funciton, and just return whatever was printed to the screen.
         # This means that we will ignore non-model return values for json output.
         # however this is probably ok behavior
         ret = self._onEmptyResult( wantText, output,
                                    execStartTime=execStartTime,
                                    execDuration=execDuration )

      origErrorCount = len( self.session_.errors_ )
      self._setSessionOutputs( ret )
      # call toDict and cache the result for later, cannot delay it
      # further because of the per command cleanup callbacks (RibCapi).
      ret.computeResultDict( streaming=streamFd is not None )
      # Errors can be added late, during toDict, when using generators: detect that
      # condition and if so rebuild the response. Note that if warning/messages
      # are added during generation, those will not show in the response, and if
      # the generation returned an extra error and data, the data is discarded.
      if origErrorCount != len( self.session_.errors_ ):
         ret = CapiCliCommon.CliCommandResponse( CapiStatus.ERROR,
                             execStartTime=execStartTime, execDuration=execDuration )
         self._setSessionOutputs( ret )
         ret.computeResultDict( streaming=streamFd is not None )
      return ret

   def _getCompletionsHelper( self, cmd, startWithPartialToken ):
      completions = []
      if CliParser.ignoredComment( self.session_.mode_, cmd.strip() ):
         # We don't need to print any help for a comment we ignore.
         return ( [], '', [] )

      allTokens = CliParser.textToTokens( cmd, mode=self.session_.mode_ )

      if not cmd or cmd != cmd.rstrip():
         tokens = allTokens
         partialToken = ''
      else:
         tokens = allTokens[ : -1 ]
         partialToken = allTokens[ -1 ]

      # partialToken is the empty string if the line was empty or there was
      # whitespace at the end of the line, or the last token from the line
      # otherwise.  tokens is a list of the rest of the tokens from the line, if
      # any.
      try:
         completions = self.session_.getCompletions(
            tokens, partialToken, startWithPartialToken=startWithPartialToken )
         unguardedCompletions = [ c for c in completions if c.guardCode is None ]
         if unguardedCompletions:
            completions = unguardedCompletions
         else:
            for c in completions:
               c.help = c.guardCode # BUG277628 help should be 'not available ...'
      except CliParser.ParserError as e:
         t1( 'There was an error getting completions:', e )

      return ( tokens, partialToken, completions )

   def getCompletions( self, cmd, preamble=None, startWithPartialToken=False ):
      self._updateSession()
      try:
         for preambleCmd in preamble or []:
            cliCmd = CliCommand( preambleCmd )
            resp = self._execute( cliCmd, False )
            if 'errors' in resp.result:
               return resp.result[ 'errors' ], {}
         _, _, completions = self._getCompletionsHelper( cmd, startWithPartialToken )
         return {}, completions
      finally:
         self._terminateSession( None )

   def _updateSession( self, textOutput=False, autoComplete=False ):
      assert self.session_ is not None
      self.session_.clearMessages()
      self.session_.shouldPrintIs( textOutput )
      self.session_.autoCompleteIs( autoComplete )

   def gotoUnprivMode( self, response=None ):
      while not isinstance( self.session_.mode, BasicCli.UnprivMode ):
         try:
            prev = self.session_.mode
            self.session_.gotoParentMode()
            t3( "Went back from mode", prev, "to", self.session_.mode )
         except Exception:  # pylint: disable-msg=W0703
            excType, exception, excTraceback = sys.exc_info()
            msg = f"Exception while leaving mode {prev.name!r}: {exception}"
            t1( msg )
            print( msg )  # Print to the agent log.
            traceback.print_exception( excType, exception, excTraceback )
            if response:
               response.status = CapiStatus.INTERNAL_ERROR
               if response.result.get( "errors" ):
                  response.result[ "errors" ].append( msg )
               else:
                  response.result[ "errors" ] = [ msg ]
            # brute force going to our parent
            self.session_.mode_ = self.session_.mode_.parent_

   def _terminateSession( self, response, stream=None ):
      """Returns to UnprivMode and kills the current Session.

      It's important that we leave all the modes and modelets we may have
      entered as some modes trigger side-effects upon being exited, such
      as changes made to an ACL not being truly recorded in Sysdb until the
      submode is left.
      If a `response` CliCommandResponse is provided, we'll update it
      with any errors, warnings or messages that we get while leaving
      modes.
      """
      self.session_.clearMessages()
      if self.stateless_:
         # this means that we expect that the session should be restored to
         # pristine conditions
         self.gotoUnprivMode( response )

      # Collect any new errors, warnings or messages that could have been
      # emitted while the session was torn down and add them to the response.
      if ( response and (    self.session_.errors_
                         or self.session_.warnings_
                         or self.session_.messages_ )
                    and not stream ):
         if isinstance( response.result, CliModel.Model ):
            # In case last command was a show: climodel don't have a get(), or
            # ability to add attributes, so convert to pure py dict now.
            # It is a bit of a circular waste, but this should be very rare.
            j = encodeCliModel( response.result )
            response.result_ = json.loads( j )
         if self.session_.errors_:
            if response.status == CapiStatus.SUCCESS:
               # We thought we succeeded, but exiting caused an error,
               # so overwrite the response status
               response.status = CapiStatus.ERROR
            if response.result.get( "errors" ):
               response.result[ "errors" ] += self.session_.errors_
            else:
               response.result[ "errors" ] = self.session_.errors_
         if self.session_.warnings_:
            if response.result.get( "warnings" ):
               response.result[ "warnings" ] += self.session_.warnings_
            else:
               response.result[ "warnings" ] = self.session_.warnings_
         if self.session_.messages_:
            if response.result.get( "messages" ):
               response.result[ "messages" ] += self.session_.messages_
            else:
               response.result[ "messages" ] = self.session_.messages_
      self.session_.clearMessages()

   def _setSessionOutputs( self, response ):
      """ Updates the `response` CliCommandResponse with any errors,
      warnings, or messages logged in the session. If the
      corresponding array already exists, we append the new outputs to
      the end of that array, otherwise we create a new member"""
      assert isinstance( response, CapiCliCommon.CliCommandResponse )
      response.errors.extend( self.session_.errors_ )
      response.warnings.extend( self.session_.warnings_ )
      response.messages.extend( self.session_.messages_ )

   def _handleException( self, excType, exception, excTraceback, command ):
      t1( "Handling exception", excType or type( exception ), ":", exception )

      exc_info = ( excType, exception, excTraceback )
      try:
         self.session_._handleCliException( exc_info, # pylint: disable-msg=W0212
                                            command )
         return False
      except: # pylint: disable=bare-except
         self.session_.addError( str( exception ) )
         return True

   def _printInternalError( self, excType, exception, excTraceback ):
      CapiCliCommon.logCapiInternalError( excType, exception, excTraceback )

   def _onException( self, exception, wantText, output,
                     execStartTime=None, execDuration=None ):
      # translate error code
      if isinstance( exception, NOT_FOUND_EXCEPTIONS ):
         status = CapiStatus.NOT_FOUND
      elif isinstance( exception, CliCommon.ApiError ):
         status = exception.ERR_CODE
      elif isinstance( exception, CliCommon.AuthzDeniedError ):
         status = CapiStatus.UNAUTHORIZED
      elif isinstance( exception, ConfigMount.ConfigChangeProhibitedError ):
         status = CapiStatus.CONFIG_LOCKED
      elif isinstance( exception, CliParser.AlreadyHandledError ):
         if self.session_.errors_:
            status = CapiStatus.ERROR
         else:
            status = CapiStatus.SUCCESS
      else:
         status = CapiStatus.INTERNAL_ERROR

      response = CapiCliCommon.CliCommandResponse( status,
                                                   execStartTime=execStartTime,
                                                   execDuration=execDuration )
      if wantText:
         response.model = TextResponse( output=output )
      return response

   def _onModelReturn( self, model,
                       globalVersion, revision,
                       execStartTime=None, execDuration=None ):
      """Takes the dict result of a successful command and transforms it into
      a CliCommandResponse with the appropriate status"""
      status = CapiStatus.ERROR if self.session_.errors_ else CapiStatus.SUCCESS
      response = CapiCliCommon.CliCommandResponse( status, model,
                                               execStartTime=execStartTime,
                                               execDuration=execDuration,
                                     globalVersion=globalVersion, revision=revision )
      if not model.__public__:
         response.warnings.append( "Model '%s' is not a public model and is subject "
                                   "to change!" % type( model ).__name__ )

      return response

   def _onConfigCmdReturn( self, output, execStartTime=None, execDuration=None ):
      """ Returns a CliCommandResponse, populated with any
      errors/warnings/messages generated. If output was produced, it
      is inserted into the messages array """
      status = CapiStatus.ERROR if self.session_.errors_ else CapiStatus.SUCCESS
      response = CapiCliCommon.CliCommandResponse( status,
                    execStartTime=execStartTime, execDuration=execDuration )
      if output:
         response.messages = [ output ]
      return response

   def _onEmptyResult( self, wantText, output,
                       execStartTime=None, execDuration=None ):
      result = TextResponse( output=output )
      if not wantText:
         # We didn't get a Model back, but this was not a show
         # commands, so we weren't expecting a model back. Just go
         # and return any errors/warnings/messages that occurred.
         return self._onConfigCmdReturn( output,
                                         execStartTime=execStartTime,
                                         execDuration=execDuration )
      elif not self.session_.errors_:
         # They explicitly asked for text.
         status = CapiStatus.SUCCESS
      else:
         status = CapiStatus.ERROR
      return CapiCliCommon.CliCommandResponse( status, model=result,
                     execStartTime=execStartTime, execDuration=execDuration )

   def shutdown( self ):
      """Must be call to safely dispose of this object."""
      pass # pylint: disable=unnecessary-pass
