# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import os

import CliCommon

def parseArgs( args=None, sysname=True, sysdbsockname=True, standalone=True,
               pdb=True, command=True, disableAaa=True, disableAutomore=True,
               echo=True, echoTimestamp=True, privilege=True,
               plugin=True, noPlugins=True, disableGuards=True,
               standaloneGuards=True, configFile=True, startupConfig=True,
               secureMonitor=True,
               standaloneShellTimeout=True, disableAutoComplete=True,
               completions=True, loadDynamicPlugins=False ):

   def privilegeLevelType( parsedValued ):
      value = int( parsedValued )
      if ( value < CliCommon.MIN_PRIV_LVL ) or ( value > CliCommon.MAX_PRIV_LVL ):
         # pylint: disable-next=consider-using-f-string
         msg = ( "privilege level out of range: valid values "
                 "are %s-%s" % ( CliCommon.MIN_PRIV_LVL, CliCommon.MAX_PRIV_LVL ) )
         raise argparse.ArgumentTypeError( msg )
      return value

   parser = argparse.ArgumentParser()
   group = None
   if command or configFile:
      group = parser.add_mutually_exclusive_group()
   if sysname:
      parser.add_argument( "-s", "--sysname", action="store",
                           default=os.environ.get( "SYSNAME", "ar" ),
                           # pylint: disable-next=consider-using-f-string
                           help="system name (default: %s)" %
                                 os.environ.get( "SYSNAME", "ar" ) )
   if sysdbsockname:
      parser.add_argument( "-k", "--sysdbsockname", action="store",
                           default=os.environ.get( "SYSDBSOCKNAME", None ),
                           help="Specify the unix domain socket of the upstream"
                           " Sysdb process" )
   if standalone:
      parser.add_argument( "-l", "--standalone", action="store_true", default=False,
                           help="run in standalone mode (without Sysdb).  Implies "
                           "--disable-guards."
                           "(also see --standalone-guards" )
   if pdb:
      parser.add_argument( "--pdb", action="store_true", help="run under pdb" )
   if command:
      group.add_argument( "-c", "--command", action="store",
                          help="run COMMAND non-interactively; separate multiple "
                          "commands with carriage returns" )
   if disableAaa:
      parser.add_argument( "-A", "--disable-aaa", action="store_true", default=False,
                           help="Do not communicate with the Aaa agent.  This "
                           "disables authentication, authorization and accounting "
                           "of commands executed in this Cli instance." )
   if disableAutomore:
      parser.add_argument( "-M", "--disable-automore", action="store_true",
                           default=False, help="Do not apply paging filter for show "
                           "commands executed in this Cli instance." )
   if echo:
      parser.add_argument( "-e", "--echo", action="store_true",
                           help="echo commands when running non-interactively" )
   if echoTimestamp:
      parser.add_argument( "-E", "--echo-timestamp", action="store_true",
                           dest="echoTimestamps",
                           help="echo commands when running non-interactively, with "
                                "timestamps on each line." )
   if privilege:
      parser.add_argument( "-p", "--privilege", action="store",
                           type=privilegeLevelType,
                           default=CliCommon.DEFAULT_PRIV_LVL,
                           # pylint: disable-next=consider-using-f-string
                           help="Start the session with this privilege level. "
                           "Legal values are 0-15 "
                           "(default: %s)" % CliCommon.DEFAULT_PRIV_LVL )
   if plugin:
      parser.add_argument( "-i", "--plugin", action="append", dest="plugins",
                           help="Load this plugin explicitly, and suppress "
                           "normal plugin search.  May be specified "
                           "repeatedly." )
   if noPlugins:
      parser.add_argument( "-I", "--no-plugins", action="store_true",
                           help="suppress any plugins from loading" )
   if disableGuards:
      parser.add_argument( "-G", "--disable-guards", action="store_true",
                           default=False, help="Do not guard any commands." )
   if standaloneGuards:
      parser.add_argument( "--standalone-guards", action="store_true", default=False,
                           help="Guard commands, even in standalone mode." )
   if configFile:
      group.add_argument( 'input_file', metavar='config-filename', nargs='?',
                          type=argparse.FileType( 'r' ),
                          help="Configuration filename to parse" )

   if startupConfig:
      parser.add_argument( "--startup-config", action="store_true", default=False,
                           help="Enable flags required for parsing startup-config" )

   if secureMonitor:
      parser.add_argument( "--secure-monitor-operator", action="store_true",
                           default=False, help=argparse.SUPPRESS )

   if disableAutoComplete:
      parser.add_argument( "--disable-autocomplete", action="store_true",
                           default=False,
                           help="Any token that does not match exactly will not "
                           "expanded implicitly" )

   if standaloneShellTimeout:
      parser.add_argument( "-T", "--standalone-shell-timeout", action="store",
                           type=int, default=0,
                           help="Specify timeout for running standalone shell" )

   if completions:
      parser.add_argument( "--completions", action="store",
                           help="Print helps for input" )

   if loadDynamicPlugins:
      parser.add_argument( "--load-dynamic-plugins", action="store_true",
                           default=False, help="force load all CliDynamicPlugin" )

   parser.add_argument( "--abort-on-error", action="store_true", default=False,
                        help="non-interactive mode: "
                             "stop running further commands after an error" )

   options = parser.parse_args( args )

   return options
