#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AaaCliLib
import CliDynamicSymbol
from CliPlugin.AaaCli import ( _getLocalUserConfig, _getAaaStatus, _getSshConfig )
from AaaPluginLib import primarySshKeyId
import Tac

AaaShowModel = CliDynamicSymbol.CliDynamicPlugin( "AaaShowModel" )

def showAaa( mode, args ):
   ret = AaaShowModel.AaaUsers()
   cfg = _getLocalUserConfig( mode )
   enablePasswd = cfg.encryptedEnablePasswd
   ret.enablePassword = enablePasswd if enablePasswd != '' else None
   for user in cfg.acct:
      password = cfg.acct[ user ].encryptedPasswd
      ret.users[ user ] = AaaShowModel.AaaUsers.AaaUser( password=password )

   return ret

#-------------------------------------------------------------------------------
# "show aaa authentication lockout" in enable mode
#-------------------------------------------------------------------------------
def showLockout( mode, args ):
   status = _getAaaStatus( mode )
   cfg = AaaCliLib.configAaa( mode )
   ret = AaaShowModel.AaaLockedUsers()
   if cfg.lockoutTime == cfg.lockoutDisabled:
      return ret
   currTime = Tac.now()
   utcTime = Tac.utcNow()
   for user, userLockoutStatus in status.userLockoutStatus.items():
      numFailedLogins = userLockoutStatus.numFailedLogins
      lastFailedLogin = userLockoutStatus.lastFailedLogin
      if ( numFailedLogins >= cfg.maxLoginAttempts and
           currTime - lastFailedLogin < min( cfg.lockoutWindow, cfg.lockoutTime ) ):
         clearTime = lastFailedLogin + cfg.lockoutTime
         clearTimeUtc = clearTime + utcTime - currTime
         lockTimeUtc = lastFailedLogin + utcTime - currTime
         ret.lockedUsers[ user ] = AaaShowModel.AaaLockedUser(
            lockoutStartTime=lockTimeUtc,
            lockoutEndTime=clearTimeUtc )
   return ret

######### "show role [name ...]" #########
#
# Implemented:
# -----------
#
# show role
# show role [name ... ]
#
# The default role is <default-role-name>
#
# role: <role-name-1>
#         <seq-no> permit <regex-1>
#         <seq-no> deny <regex-2>
#         <seq-no> ...
#
# role: <role-name-2>
#         <seq-no> permit <regex-5>
#         <seq-no> deny <regex-6>
#         <seq-no> ...
# ...
def showRoleByName( mode, args ):
   cfg = _getLocalUserConfig( mode )
   names = args.get( 'ROLE', [] )
   if names:
      names = sorted( x for x in set( names ) if x in cfg.role )
   else:
      names = cfg.role

   ret = AaaShowModel.ShowRoles()
   ret.defaultRole = cfg.defaultRole
   for name in names:
      ret2 = AaaShowModel.Rules()
      role = cfg.role[ name ]
      for seq, rule in role.rule.items():
         ret1 = AaaShowModel.ShowRolesList()
         if rule.modeKey:
            ret1.mode = rule.modeKey
         ret1.sequenceNumber = seq
         ret1.cmdPermission = rule.action
         ret1.cmdRegex = rule.regex
         ret2.rules.append( ret1 )
      ret.roles[ name ] = ret2
   return ret

def fillSshAuthorizedKeys( username, user ):
   '''
      Utility function which checks if the SSH keys are configured for the user
      in SshConfig and populate the primary and secondary keys
   '''
   sshConfig = _getSshConfig()
   if username in sshConfig.user:
      keys = sshConfig.user[ username ].sshAuthKeys
      if primarySshKeyId in keys:
         user.sshAuthorizedKey = keys[ primarySshKeyId ].keyContents
      secKeys = [ k.keyContents for n, k in keys.items() if n != primarySshKeyId ]
      if secKeys:
         user.secondarySshKeys = sorted( secKeys )

#------------------------------------------------------------------------------
# show user-account (deprecated)
# show users accounts
#------------------------------------------------------------------------------
def showUserAccount( mode, args ):
   result = AaaShowModel.Users()
   for username, acct in _getLocalUserConfig( mode ).acct.items():
      user = AaaShowModel.Users.User( username=username,
                                      privLevel=acct.privilegeLevel )
      if acct.role != acct.roleDefault:
         user.role = acct.role
      fillSshAuthorizedKeys( username, user )
      result.users[ username ] = user
   return result
